package com.biz.crm.member.business.member.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.member.business.member.local.entity.ProductCategory;
import com.biz.crm.member.business.member.local.repository.ProductCategoryRepository;
import com.biz.crm.member.business.member.local.service.ProductCategoryService;
import com.biz.crm.member.business.member.sdk.dto.ProductCategoryDto;
import com.biz.crm.member.business.member.sdk.dto.ProductPageDto;
import com.biz.crm.member.business.member.sdk.vo.ProductCategoryVo;
import com.biz.crm.member.business.member.sdk.vo.ProductPageVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import java.text.SimpleDateFormat;
import java.util.*;

/**
 * 商品分类实现
 *
 * @Description
 * @Author monroe
 * @Date 2023/6/14 16:31
 **/
@Service
public class ProductCategoryServiceImpl implements ProductCategoryService {

    @Autowired
    private ProductCategoryRepository productCategoryRepository;
    @Autowired
    private NebulaToolkitService nebulaToolkitService;
    @Autowired
    private RedisMutexService redisMutexService;

    /**
     * 商品分类编码redis key
     * 参数：租户
     */
    public static final String MMS_CATEGORY_PRODUCT_CODE_REDIS_KEY = "mms:category:product:code:index:%s";

    /**
     * 商品分类编码前缀
     */
    public static final String MMS_CATEGORY_PRODUCT_CODE_PREFIX = "SPFL";

    @Override
    public Page<ProductCategoryVo> findByConditions(Pageable pageable, ProductCategoryDto productCategoryDto) {
        if (Objects.isNull(productCategoryDto)) {
            productCategoryDto = new ProductCategoryDto();
        }
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
        Page<ProductPageVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.productCategoryRepository.findByConditions(page,productCategoryDto);
    }

    @Override
    public ProductCategoryVo findDetailById(String id) {
        if(StringUtils.isBlank(id)) {
            return null;
        }
        ProductCategory productCategory = this.productCategoryRepository.getById(id);
        return nebulaToolkitService.copyObjectByWhiteList(productCategory,ProductCategoryVo.class, HashSet.class, LinkedList.class);
    }

    @Override
    public ProductCategoryVo create(ProductCategoryDto productCategoryDto) {
        Validate.notNull(productCategoryDto,"新增实体不能为空");
        Validate.notBlank(productCategoryDto.getCategoryName(),"分类名称不能为空");
        Validate.notNull(productCategoryDto.getSortIndex(),"分类排序不能为空");
        if(productCategoryDto.getSortIndex() <= 0) {
            throw new RuntimeException("排序请输入大于0的整数");
        }

        //生成编码
        productCategoryDto.setCategoryCode(this.generateCode(TenantUtils.getTenantCode()));
        productCategoryDto.setTenantCode(TenantUtils.getTenantCode());
        productCategoryDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        productCategoryDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        ProductCategory productCategory = this.nebulaToolkitService.copyObjectByWhiteList(productCategoryDto, ProductCategory.class, HashSet.class, LinkedList.class);
        this.productCategoryRepository.saveOrUpdate(productCategory);

        return this.nebulaToolkitService.copyObjectByWhiteList(productCategory,ProductCategoryVo.class,HashSet.class,LinkedList.class);
    }

    /**
     * 生成编码
     */
    private String generateCode(String tenantCode) {
        String redisKey = String.format(MMS_CATEGORY_PRODUCT_CODE_REDIS_KEY, tenantCode);
        String index = redisMutexService.getAndIncrement(redisKey, 1, 6);
        SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
        return StringUtils.join(MMS_CATEGORY_PRODUCT_CODE_PREFIX, format.format(new Date()), index);
    }

    @Override
    public ProductCategoryVo update(ProductCategoryDto productCategoryDto) {
        Validate.notNull(productCategoryDto,"修改实体不能为空");
        Validate.notBlank(productCategoryDto.getCategoryName(),"分类名称不能为空");
        Validate.notNull(productCategoryDto.getSortIndex(),"分类排序不能为空");
        Validate.notBlank(productCategoryDto.getId(),"编辑时，主键id不能为空");
        Validate.notBlank(productCategoryDto.getCategoryCode(),"编辑时，分类编码不能为空");

        if(productCategoryDto.getSortIndex() <= 0) {
            throw new RuntimeException("排序请输入大于0的整数");
        }

        ProductCategory productCategory = this.nebulaToolkitService.copyObjectByWhiteList(productCategoryDto, ProductCategory.class, HashSet.class, LinkedList.class);
        this.productCategoryRepository.saveOrUpdate(productCategory);

        return this.nebulaToolkitService.copyObjectByWhiteList(productCategory,ProductCategoryVo.class,HashSet.class,LinkedList.class);
    }

    @Override
    public void enableBatch(List<String> ids) {
        if(CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.productCategoryRepository.lambdaUpdate().set(ProductCategory::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
                .in(ProductCategory::getId,ids).update();

    }

    @Override
    public void disableBatch(List<String> ids) {
        if(CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.productCategoryRepository.lambdaUpdate().set(ProductCategory::getEnableStatus, EnableStatusEnum.DISABLE.getCode())
                .in(ProductCategory::getId,ids).update();
    }

    @Override
    public void deleteBatch(List<String> ids) {
        if(CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.productCategoryRepository.lambdaUpdate().set(ProductCategory::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
                .in(ProductCategory::getId,ids).update();
    }

    @Override
    public void down(String id) {
        if(StringUtils.isBlank(id)) {
            return;
        }
        this.productCategoryRepository.down(id);
    }

    @Override
    public void up(String id) {
        if(StringUtils.isBlank(id)) {
            return;
        }
        this.productCategoryRepository.up(id);
    }
}
