package com.biz.crm.member.business.member.local.service.internal;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.member.business.member.local.entity.Product;
import com.biz.crm.member.business.member.local.entity.ProductExchangeRule;
import com.biz.crm.member.business.member.local.entity.ProductFile;
import com.biz.crm.member.business.member.local.repository.ProductExchangeRuleRepository;
import com.biz.crm.member.business.member.local.repository.ProductFileRepository;
import com.biz.crm.member.business.member.local.repository.ProductRepository;
import com.biz.crm.member.business.member.local.service.ProductService;
import com.biz.crm.member.business.member.sdk.dto.*;
import com.biz.crm.member.business.member.sdk.enums.ProductSourceEnum;
import com.biz.crm.member.business.member.sdk.enums.ShelfStatusEnum;
import com.biz.crm.member.business.member.sdk.vo.*;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import io.swagger.annotations.ApiModelProperty;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * 商品实现类
 *
 * @Description
 * @Author monroe
 * @Date 2023/6/13 16:50
 **/
@Service
public class ProductServiceImpl implements ProductService {

    @Autowired
    private ProductRepository productRepository;
    @Autowired
    private ProductExchangeRuleRepository productExchangeRuleRepository;
    @Autowired
    private ProductFileRepository productFileRepository;

    @Autowired
    private NebulaToolkitService nebulaToolkitService;
    @Autowired
    private RedisMutexService redisMutexService;
    @Autowired
    private LoginUserService loginUserService;

    /**
     * 上架商品编码redis key
     * 参数：租户
     */
    public static final String MMS_SHELF_PRODUCT_CODE_REDIS_KEY = "mms:shelf:product:code:index:%s";

    /**
     * 上架商品编码前缀
     */
    public static final String MMS_SHELF_PRODUCT_CODE_PREFIX = "SJSP";

    @Override
    public Page<ProductPageVo> findByConditions(Pageable pageable, ProductPageDto productPageDto) {
        if (Objects.isNull(productPageDto)) {
            productPageDto = new ProductPageDto();
        }
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
        Page<ProductPageVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return productRepository.findByConditions(page,productPageDto);
    }

    @Override
    public ProductVo findDetailById(String id) {
        if(StringUtils.isBlank(id)) {
            return null;
        }
        Product product = this.productRepository.getById(id);
        Validate.notNull(product,"商品信息不存在");
        ProductVo productVo = this.nebulaToolkitService.copyObjectByWhiteList(product, ProductVo.class, HashSet.class, LinkedList.class);
        //兑换规则
        ProductExchangeRule productExchangeRule = this.productExchangeRuleRepository.lambdaQuery().eq(ProductExchangeRule::getShelfCode, product.getCode()).one();
        if(Objects.nonNull(productExchangeRule)) {
            ProductExchangeRuleVo productExchangeRuleVo = this.nebulaToolkitService.copyObjectByWhiteList(productExchangeRule, ProductExchangeRuleVo.class, HashSet.class, LinkedList.class);
            productVo.setRule(productExchangeRuleVo);
        }
        //商品文件
        List<ProductFile> productFiles = this.productFileRepository.lambdaQuery().eq(ProductFile::getShelfCode, product.getCode())
                .eq(ProductFile::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
                .list();
        if(CollectionUtils.isNotEmpty(productFiles)) {
            List<ProductFileVo> productFileVos = (List<ProductFileVo>) this.nebulaToolkitService.copyCollectionByWhiteList(productFiles, ProductFile.class, ProductFileVo.class, HashSet.class, LinkedList.class);
            productVo.setFiles(productFileVos);
            //拆分图片
            this.splitFiles(productFileVos,productVo);
        }

        return productVo;
    }

    public void splitFiles(List<ProductFileVo> productFileVos, ProductVo productVo) {

        List<ProductFileVo> viewPagerFile = new ArrayList<>();

        List<ProductFileVo> productFile = new ArrayList<>();

        List<ProductFileVo> smallFile = new ArrayList<>();


        for (ProductFileVo productFileVo : productFileVos) {
            if("1".equals(productFileVo.getType())) {
                viewPagerFile.add(productFileVo);
            }else if ("2".equals(productFileVo.getType())) {
                productFile.add(productFileVo);
            }else {
                smallFile.add(productFileVo);
            }
        }

        productVo.setViewPagerFile(viewPagerFile);
        productVo.setProductFile(productFile);
        productVo.setSmallFile(smallFile);
    }

    @Transactional
    @Override
    public ProductVo create(ProductDto productDto) {
        this.createValidate(productDto);
        productDto.setCode(this.generateCode(TenantUtils.getTenantCode()));

        //保存商品基础信息
        Product product = this.nebulaToolkitService.copyObjectByWhiteList(productDto, Product.class, HashSet.class, LinkedList.class);
        this.productRepository.saveOrUpdate(product);
        //保存兑换规则
        ProductExchangeRuleDto rule = productDto.getRule();
        if(Objects.nonNull(rule)) {
            ProductExchangeRule productExchangeRule = this.nebulaToolkitService.copyObjectByWhiteList(rule, ProductExchangeRule.class, HashSet.class, LinkedList.class);
            productExchangeRule.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            productExchangeRule.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            productExchangeRule.setTenantCode(TenantUtils.getTenantCode());
            productExchangeRule.setShelfCode(product.getCode());
            this.productExchangeRuleRepository.saveOrUpdate(productExchangeRule);
        }

        //保存商品文件
        List<ProductFileDto> files = this.getProductFileDtos(productDto);
        if(CollectionUtils.isNotEmpty(files)) {
            files.stream().forEach(e ->{
                e.setShelfCode(product.getCode());
                e.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                e.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                e.setTenantCode(TenantUtils.getTenantCode());
            });
            List<ProductFile> productFiles = (List<ProductFile>) this.nebulaToolkitService.copyCollectionByWhiteList(files, ProductFileDto.class, ProductFile.class, HashSet.class, LinkedList.class);
            this.productFileRepository.saveBatch(productFiles);
        }

        return this.nebulaToolkitService.copyObjectByWhiteList(product, ProductVo.class, HashSet.class, LinkedList.class);
    }



    private List<ProductFileDto> getProductFileDtos(ProductDto productDto) {
        List<ProductFileDto> fileDtos = new ArrayList<>();

        //合并文件
        List<ProductFileDto> viewPagerFile = productDto.getViewPagerFile();
        if(CollectionUtils.isNotEmpty(viewPagerFile)) {
            for (ProductFileDto productFileDto : viewPagerFile) {
                Validate.notBlank(productFileDto.getFileId(),"文件id为空");
                Validate.notBlank(productFileDto.getFileName(),"文件名称为空");
//                Validate.notBlank(productFileDto.getType(),"文件类型为空");
                productFileDto.setType("1");
                fileDtos.add(productFileDto);
            }
        }
        //商品详情
        List<ProductFileDto> productFile = productDto.getProductFile();
        if(CollectionUtils.isNotEmpty(productFile)) {
            for (ProductFileDto productFileDto : productFile) {
                Validate.notBlank(productFileDto.getFileId(),"文件id为空");
                Validate.notBlank(productFileDto.getFileName(),"文件名称为空");
                productFileDto.setType("2");
                fileDtos.add(productFileDto);
            }
        }
        //缩略图
        List<ProductFileDto> smallFile = productDto.getSmallFile();
        if(CollectionUtils.isNotEmpty(smallFile)) {
            for (ProductFileDto productFileDto : smallFile) {
                Validate.notBlank(productFileDto.getFileId(),"文件id为空");
                Validate.notBlank(productFileDto.getFileName(),"文件名称为空");
                productFileDto.setType("3");
                fileDtos.add(productFileDto);
            }
        }
        return fileDtos;
    }

    private void createValidate(ProductDto productDto) {
        Validate.notNull(productDto,"新增实体参数不能为空");
        Validate.notBlank(productDto.getName(),"上架名称不能为空");
        Validate.notBlank(productDto.getProductType(),"商品类型不能为空！");
        Validate.notBlank(productDto.getProductCategoryCode(),"商品分类不能为空！");
        Validate.notBlank(productDto.getSource(),"商品来源不能为空");
        if(ProductSourceEnum.MAIN.getCode().equals(productDto.getSource())) {
            Validate.notBlank(productDto.getUnitCode(),"计量单位不能为空");
        }

        ProductExchangeRuleDto rule = productDto.getRule();
        if(Objects.nonNull(rule)) {
            Validate.notBlank(rule.getLevelCode(),"等级编码不能为空");
            Validate.notBlank(rule.getLevelName(),"等级名称不能为空");
            Validate.notBlank(rule.getExchangePeriod(),"兑换周期编码不能为空");
            Validate.notBlank(rule.getExchangePeriodName(),"兑换周期名称不能为空");
            Validate.notNull(rule.getQuantity(),"可兑数量不能为空");
            Validate.notNull(rule.getIntegralPrice(),"积分价格不能为空");
            Validate.notNull(rule.getInventoryQuantity(),"库存数量不能为空");
            Validate.notNull(rule.getUpDate(),"上架时间不能为空");
            Validate.notNull(rule.getDownDate(),"下架时间不能为空");
        }

        //初始化参数
        productDto.setStatus(ShelfStatusEnum.DOWN.getCode());
        Date now = new Date();
        productDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        productDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        productDto.setTenantCode(TenantUtils.getTenantCode());
//        UserIdentity loginUser = loginUserService.getLoginUser();
//        productDto.setCreateAccount(loginUser.getAccount());
//        productDto.setCreateName(loginUserService.getLoginAccountName());
//        productDto.setModifyAccount(loginUser.getAccount());
//        productDto.setModifyName(loginUserService.getLoginAccountName());
//        productDto.setCreateTime(now);
//        productDto.setModifyTime(now);
    }

    /**
     * 生成编码
     */
    private String generateCode(String tenantCode) {
        String redisKey = String.format(MMS_SHELF_PRODUCT_CODE_REDIS_KEY, tenantCode);
        String index = redisMutexService.getAndIncrement(redisKey, 1, 6);
        SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
        return StringUtils.join(MMS_SHELF_PRODUCT_CODE_PREFIX, format.format(new Date()), index);
    }

    @Transactional
    @Override
    public ProductVo update(ProductDto productDto) {
        this.updateValidate(productDto);
        productDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        productDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        productDto.setTenantCode(TenantUtils.getTenantCode());
        //修嘎基础信息
        Product product = this.nebulaToolkitService.copyObjectByWhiteList(productDto, Product.class, HashSet.class, LinkedList.class);
        this.productRepository.saveOrUpdate(product);
        QueryWrapper<ProductExchangeRule> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("shelf_code",productDto.getCode());
        queryWrapper.eq("tenant_code",productDto.getTenantCode());
        this.productExchangeRuleRepository.remove(queryWrapper);
        //修改兑换规则
        ProductExchangeRuleDto rule = productDto.getRule();
        if(Objects.nonNull(rule)) {
            ProductExchangeRule productExchangeRule = this.nebulaToolkitService.copyObjectByWhiteList(rule, ProductExchangeRule.class, HashSet.class, LinkedList.class);
            productExchangeRule.setShelfCode(product.getCode());
            productExchangeRule.setTenantCode(TenantUtils.getTenantCode());
            productExchangeRule.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            productExchangeRule.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            this.productExchangeRuleRepository.saveOrUpdate(productExchangeRule);
        }
        //修改商品文件
        List<ProductFileDto> files = this.getProductFileDtos(productDto);
        QueryWrapper<ProductFile> wrapper = new QueryWrapper<>();
        wrapper.eq("shelf_code",product.getCode());
        wrapper.eq("tenant_code",product.getTenantCode());
        this.productFileRepository.remove(wrapper);

        if(CollectionUtils.isNotEmpty(files)) {
            List<ProductFile> productFiles = (List<ProductFile>) this.nebulaToolkitService.copyCollectionByWhiteList(files, ProductFileDto.class, ProductFile.class, HashSet.class, LinkedList.class);
            productFiles.stream().forEach(e ->{
                e.setId(null);
                e.setShelfCode(product.getCode());
                e.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                e.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                e.setTenantCode(TenantUtils.getTenantCode());
            });
            this.productFileRepository.saveBatch(productFiles);
        }

        return this.nebulaToolkitService.copyObjectByWhiteList(product, ProductVo.class, HashSet.class, LinkedList.class);
    }

    private void updateValidate(ProductDto productDto) {

        Validate.notNull(productDto,"新增实体参数不能为空");
        Validate.notBlank(productDto.getName(),"上架名称不能为空");
        Validate.notBlank(productDto.getProductType(),"商品类型不能为空！");
        Validate.notBlank(productDto.getProductCategoryCode(),"商品分类不能为空！");
        Validate.notBlank(productDto.getSource(),"商品来源不能为空");
        if(ProductSourceEnum.MAIN.getCode().equals(productDto.getSource())) {
            Validate.notBlank(productDto.getUnitCode(),"计量单位不能为空");
        }        Validate.notBlank(productDto.getId(),"编辑时，主键id不能为空");
        Validate.notBlank(productDto.getCode(),"编辑时，编码不能为空");

        ProductExchangeRuleDto rule = productDto.getRule();
        if(Objects.nonNull(rule)) {
            Validate.notBlank(rule.getLevelCode(),"等级编码不能为空");
            Validate.notBlank(rule.getLevelName(),"等级名称不能为空");
            Validate.notBlank(rule.getExchangePeriod(),"兑换周期编码不能为空");
            Validate.notBlank(rule.getExchangePeriodName(),"兑换周期名称不能为空");
            Validate.notNull(rule.getQuantity(),"可兑数量不能为空");
            Validate.notNull(rule.getIntegralPrice(),"积分价格不能为空");
            Validate.notNull(rule.getInventoryQuantity(),"库存数量不能为空");
            Validate.notNull(rule.getUpDate(),"上架时间不能为空");
            Validate.notNull(rule.getDownDate(),"下架时间不能为空");
        }

        List<ProductFileDto> files = productDto.getFiles();
        if(CollectionUtils.isNotEmpty(files)) {
            for (ProductFileDto file : files) {
                Validate.notBlank(file.getFileId(),"文件id为空");
                Validate.notBlank(file.getFileName(),"文件名称为空");
                Validate.notBlank(file.getType(),"文件类型为空");
            }
        }

    }

    @Override
    public void disableBatch(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.productRepository.lambdaUpdate().set(Product::getEnableStatus, EnableStatusEnum.DISABLE.getCode())
                .in(Product::getId,ids)
                .update();
    }

    @Override
    public void shelfUp(List<String> ids) {

        if (CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.productRepository.lambdaUpdate().set(Product::getStatus, ShelfStatusEnum.UP.getCode())
                .in(Product::getId,ids)
                .update();

    }

    @Override
    public void shelfDown(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.productRepository.lambdaUpdate().set(Product::getStatus, ShelfStatusEnum.DOWN.getCode())
                .in(Product::getId,ids)
                .update();
    }

    @Override
    public void enableBatch(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.productRepository.lambdaUpdate().set(Product::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
                .in(Product::getId,ids)
                .update();
    }

    @Override
    public void deleteBatch(List<String> ids) {

        if (CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.productRepository.lambdaUpdate().set(Product::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
                .in(Product::getId,ids)
                .update();
    }

    @Override
    public void adjust(String id, BigDecimal number) {

        if(StringUtils.isBlank(id) || Objects.isNull(number)) {
            return;
        }

        ProductVo productVo = this.findDetailById(id);
        ProductExchangeRuleVo rule = productVo.getRule();
        Validate.notNull(rule,"兑换规则为空");

        BigDecimal inventoryQuantity = rule.getInventoryQuantity();
        if(number.compareTo(BigDecimal.ZERO) < 0) {
            Validate.isTrue(number.negate().compareTo(inventoryQuantity) < 0,"库存调整超过库存数量");
        }
        rule.setInventoryQuantity(inventoryQuantity.add(number));
        ProductExchangeRule productExchangeRule = this.nebulaToolkitService.copyObjectByWhiteList(rule, ProductExchangeRule.class, HashSet.class, LinkedList.class);
        this.productExchangeRuleRepository.saveOrUpdate(productExchangeRule);


    }
}
