package com.biz.crm.member.business.member.local.service.internal;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.member.business.member.local.entity.AnswerRecord;
import com.biz.crm.member.business.member.local.entity.AnswerUser;
import com.biz.crm.member.business.member.local.entity.Questionnaire;
import com.biz.crm.member.business.member.local.entity.QuestionnaireQuestion;
import com.biz.crm.member.business.member.local.helper.UserSearchHelper;
import com.biz.crm.member.business.member.local.repository.AnswerUserRepository;
import com.biz.crm.member.business.member.local.repository.QuestionnaireRepository;
import com.biz.crm.member.business.member.local.service.AnswerRecordService;
import com.biz.crm.member.business.member.local.service.MemberInfoIntegralRecordService;
import com.biz.crm.member.business.member.local.service.QuestionnaireQuestionService;
import com.biz.crm.member.business.member.local.service.QuestionnaireService;
import com.biz.crm.member.business.member.sdk.dto.AnswerRecordPaginationDto;
import com.biz.crm.member.business.member.sdk.dto.QuestionDto;
import com.biz.crm.member.business.member.sdk.dto.QuestionnaireDto;
import com.biz.crm.member.business.member.sdk.dto.QuestionnairePaginationDto;
import com.biz.crm.member.business.member.sdk.dto.QuestionnaireQuestionDto;
import com.biz.crm.member.business.member.sdk.dto.QuestionnaireSubmitDto;
import com.biz.crm.member.business.member.sdk.enums.IntegralSourceEnum;
import com.biz.crm.member.business.member.sdk.enums.QuestionnaireLifeCycleStatusEnum;
import com.biz.crm.member.business.member.sdk.enums.QuestionnaireStatusEnum;
import com.biz.crm.member.business.member.sdk.vo.AnswerRecordVo;
import com.biz.crm.member.business.member.sdk.vo.MemberInfoIntegralRecordVo;
import com.biz.crm.member.business.member.sdk.vo.QuestionAnswerVo;
import com.biz.crm.member.business.member.sdk.vo.QuestionCountVo;
import com.biz.crm.member.business.member.sdk.vo.QuestionStatisticVo;
import com.biz.crm.member.business.member.sdk.vo.QuestionVo;
import com.biz.crm.member.business.member.sdk.vo.QuestionnaireAnalyzeVo;
import com.biz.crm.member.business.member.sdk.vo.QuestionnaireDetailVo;
import com.biz.crm.member.business.member.sdk.vo.QuestionnaireVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.JsonUtils;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.security.sdk.login.UserIdentity;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * @author 吴平
 * @version 1.0.0
 * @data 2023/6/20 14:48
 * @description 问卷管理
 */
@Service
@Slf4j
public class QuestionnaireServiceImpl implements QuestionnaireService {

  @Autowired
  private LoginUserService loginUserService;

  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Autowired
  private GenerateCodeService generateCodeService;

  @Autowired
  private QuestionnaireRepository questionnaireRepository;

  @Autowired
  private QuestionnaireQuestionService questionnaireQuestionService;

  @Autowired
  private AnswerRecordService answerRecordService;

  @Autowired
  private AnswerUserRepository answerUserRepository;

  @Autowired
  private MemberInfoIntegralRecordService memberInfoIntegralRecordService;

  @Autowired
  private UserSearchHelper userSearchHelper;

  // 需要统计的组件类型
  private final List<String> NEED_STATISTIC_TYPE = Arrays
      .asList("select", "radio", "checkbox");

  /**
   * 新增一个发布状态的问卷
   *
   * @param dto
   * @return
   */
  @Override
  public QuestionnaireVo createQuestionnairePublish(QuestionDto dto) {
    // 获取基本信息
    QuestionnaireDto basicInfo = dto.getQuestionnaireDto();
    this.createValidate(basicInfo);
    Date startTime = basicInfo.getStartTime();
    Date currentTime = new Date(System.currentTimeMillis());
    QuestionnaireStatusEnum publishStatus = QuestionnaireStatusEnum.NO_STARTED;
    if (startTime.before(currentTime)) {
      publishStatus = QuestionnaireStatusEnum.RUNNING;
    }
    return this.createQuestionnaire(dto, publishStatus);
  }

  /**
   * 新增一个问卷
   *
   * @param dto
   * @param statusEnum
   * @return
   */
  @Transactional
  @Override
  public QuestionnaireVo createQuestionnaire(QuestionDto dto, QuestionnaireStatusEnum statusEnum) {
    QuestionnaireDto questionnaireDto = dto.getQuestionnaireDto();
    String questionnaireCode = questionnaireDto.getCode();
    Optional.ofNullable(questionnaireCode).ifPresent(code->{
      Questionnaire questionnaire = this.questionnaireRepository.getByCode(code);
      Validate.notNull(questionnaire, "复制的问卷不存在");
      String pubStatus = questionnaire.getPubStatus();
      Validate.isTrue(!QuestionnaireStatusEnum.NO_STARTED.getValue().equals(pubStatus), "未发布的问卷不能复制");
    });
    questionnaireDto.setId(StringUtils.EMPTY);
    questionnaireDto.setCode(StringUtils.EMPTY);
    // 存储问卷基本信息
    QuestionnaireVo result = this.createBasicQuestionnaire(questionnaireDto, statusEnum);
    // 清空id
    dto.getQuestionDtoList().stream().forEach(question->{
      question.setId(StringUtils.EMPTY);
    });
    // 存储问卷题目
    this.questionnaireQuestionService.createQuestion(dto.getQuestionDtoList(), result.getCode());
    return result;
  }

  /**
   * 分页获取问卷信息
   *
   * @param pageable
   * @param dto
   * @return
   */
  @Override
  public Page<QuestionnaireVo> pageResult(Pageable pageable, QuestionnairePaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new QuestionnairePaginationDto());
    Page<QuestionnaireVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<QuestionnaireVo> result = this.questionnaireRepository.pageInfo(page, dto);
    this.populateCommitNum(result.getRecords());
    return result;
  }

  /**
   * 批量逻辑删除问卷
   *
   * @param ids
   */
  @Transactional
  @Override
  public void logicDelete(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id不能为空");
    this.questionnaireRepository.logicDelete(ids);
  }

  /**
   * 启用
   *
   * @param ids
   */
  @Override
  public void turnOnQuestionnaire(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id不能为空");
    this.questionnaireRepository.modifyEnableStatus(ids, EnableStatusEnum.ENABLE);
  }

  /**
   * 禁用
   *
   * @param ids
   */
  @Override
  public void turnOffQuestionnaire(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id不能为空");
    this.questionnaireRepository.modifyEnableStatus(ids, EnableStatusEnum.DISABLE);
  }

  /**
   * 发布问卷
   *
   * @param questionnaireCode
   */
  @Override
  public void publishQuestionnaire(String questionnaireCode) {
    Validate.notBlank(questionnaireCode, "问卷编码不能为空");
    Questionnaire currentQuestionnaire = this.questionnaireRepository.getByCode(questionnaireCode);
    String pubStatus = currentQuestionnaire.getPubStatus();
    String enableStatus = currentQuestionnaire.getEnableStatus();
    Validate.isTrue(!QuestionnaireStatusEnum.FINISH.getValue().equals(pubStatus), "已结束的问卷不能再发布");
    Validate.isTrue(EnableStatusEnum.ENABLE.getCode().equals(enableStatus), "禁用状态的问卷不能发布");
    Date startTime = currentQuestionnaire.getStartTime();
    Date currentTime = new Date(System.currentTimeMillis());
    QuestionnaireStatusEnum statusEnum = QuestionnaireStatusEnum.RUNNING;
    int compareResult = startTime.compareTo(currentTime);
    if (compareResult >= 0) {
      statusEnum = QuestionnaireStatusEnum.NO_STARTED;
    }
    this.questionnaireRepository.modifyQuestionnaireStatus(questionnaireCode, statusEnum);
  }

  /**
   * 结束问卷
   *
   * @param questionnaireCode
   */
  @Override
  public void finishQuestionnaire(String questionnaireCode) {
    Validate.notBlank(questionnaireCode, "问卷编码不能为空");
    this.questionnaireRepository
        .modifyQuestionnaireStatus(questionnaireCode, QuestionnaireStatusEnum.FINISH);
  }

  /**
   * 根据问卷编码获取问卷详情 详情包括：问卷基本信息和问卷配置题目信息
   *
   * @param questionnaireCode
   * @return
   */
  @Override
  public QuestionnaireDetailVo getQuestionnaireDetailByCode(String questionnaireCode) {
    Validate.notBlank(questionnaireCode, "问卷编码不能为空");
    Questionnaire questionnaire = this.questionnaireRepository.getByCode(questionnaireCode);
    Validate.notNull(questionnaire, "当前问卷不存在");
    QuestionnaireDetailVo result = this.nebulaToolkitService
        .copyObjectByWhiteList(questionnaire, QuestionnaireDetailVo.class, HashSet.class,
            LinkedList.class);
    List<QuestionVo> questionList = this.questionnaireQuestionService
        .listByQuestionnaireCodeOrderBySortIndex(questionnaireCode);
    result.setQuestionVoList(questionList);
    return result;
  }

  /**
   * 修改问卷信息
   *
   * @param dto
   * @return
   */
  @Transactional
  @Override
  public QuestionnaireVo modifyQuestionnaire(QuestionDto dto) {
    // 问卷基本信息
    QuestionnaireDto basicInfo = dto.getQuestionnaireDto();
    QuestionnaireVo result = this.modifyQuestionnaireBasicInfo(basicInfo);
    String questionnaireCode = basicInfo.getCode();
    // 配置题目列表
    List<QuestionnaireQuestionDto> questionDtoList = dto.getQuestionDtoList();
    this.questionnaireQuestionService.logicDeleteByQuestionnaireCode(questionnaireCode);
    // 清空题目配置中的原有id信息
    this.clearQuestionListId(questionDtoList);
    this.questionnaireQuestionService.createQuestion(questionDtoList, questionnaireCode);
    return result;
  }

  /**
   * 复制一个问卷
   *
   * @param questionnaireCode
   * @return
   */
  @Transactional
  @Override
  public QuestionnaireVo copyQuestionnaireByCode(String questionnaireCode) {
    /**
     * 1、获取需要copy的问卷源数据
     * 2、获取需要copy的问卷题目配置源数据
     * 3、将获取到的数据封装到QuestionDto对象中
     * 4、创建一个未发布的问卷
     * */
    // 第1步====================
    Questionnaire sourceQuestionnaire = this.questionnaireRepository.getByCode(questionnaireCode);
    Validate.notNull(sourceQuestionnaire, "拷贝对象不存在");
    sourceQuestionnaire.setId(StringUtils.EMPTY);
    sourceQuestionnaire.setCode(StringUtils.EMPTY);
    QuestionnaireDto questionnaireDto = this.nebulaToolkitService
        .copyObjectByBlankList(sourceQuestionnaire, QuestionnaireDto.class, HashSet.class,
            LinkedList.class);
    // 第2步=================
    List<QuestionVo> sourceQuestionListVo = this.questionnaireQuestionService
        .listByQuestionnaireCodeOrderBySortIndex(questionnaireCode);
    Validate.isTrue(CollectionUtils.isNotEmpty(sourceQuestionListVo), "拷贝问卷题目配置不能为空");
    Collection<QuestionnaireQuestionDto> questionnaireQuestionDtos = this.nebulaToolkitService
        .copyCollectionByWhiteList(sourceQuestionListVo, QuestionVo.class,
            QuestionnaireQuestionDto.class,
            HashSet.class, LinkedList.class);
    this.clearQuestionListId(questionnaireQuestionDtos);
    // 第3步==============
    QuestionDto targetDto = new QuestionDto();
    targetDto.setQuestionnaireDto(questionnaireDto);
    targetDto.setQuestionDtoList((List<QuestionnaireQuestionDto>) questionnaireQuestionDtos);
    // 第4步===============
    QuestionnaireVo result = this
        .createQuestionnaire(targetDto, QuestionnaireStatusEnum.NO_PUBLISH);
    return result;
  }

  /**
   * 问卷分析
   *
   * @param questionnaireCode
   * @return
   */
  @Override
  public List<QuestionnaireAnalyzeVo> getQuestionnaireAnalyze(String questionnaireCode) {
    /**
     * 1、获取问卷对应的题目信息
     * 2、将题目信息转换为方法输出的Vo
     * 3、过滤需要统计的题目并获取题目编码
     * 4、根据题目编码获取用户答题记录，并统计
     * 5、将获取到的统计信息，填充进入返回的对象中
     * */
    Validate.notBlank(questionnaireCode, "问卷编码不能为空");
    // 第1步=================
    List<QuestionVo> questionVoList = this.questionnaireQuestionService
        .listByQuestionnaireCodeOrderBySortIndex(questionnaireCode);
    // 第2步=================
    Collection<QuestionnaireAnalyzeVo> result = this.nebulaToolkitService
        .copyCollectionByWhiteList(questionVoList, QuestionVo.class, QuestionnaireAnalyzeVo.class,
            HashSet.class, LinkedList.class);
    // 第3步=================
    List<QuestionVo> needStatisticQuestionInfoList = questionVoList
        .stream()
        .filter(obj -> NEED_STATISTIC_TYPE.contains(obj.getType()))
        .collect(Collectors.toList());
    List<String> questionCodes = needStatisticQuestionInfoList
        .stream()
        .map(QuestionVo::getCode)
        .distinct()
        .collect(Collectors.toList());

    if (CollectionUtils.isEmpty(questionCodes)) {
      return (List<QuestionnaireAnalyzeVo>) result;
    }
    // 第4步================
    List<AnswerRecord> records = this.answerRecordService.listAnswerRecord(questionCodes);
    Map<String, List<QuestionStatisticVo>> statisticMap = this
        .generateStatisticMap(records, needStatisticQuestionInfoList);
    // 第5步================
    for (QuestionnaireAnalyzeVo analyzeVo : result) {
      String questionCode = analyzeVo.getCode();
      analyzeVo.setAnalyzeList(statisticMap.get(questionCode));
      List<QuestionStatisticVo> analyzeList = analyzeVo.getAnalyzeList();
      if (CollectionUtils.isEmpty(analyzeList)) {
        analyzeVo.setValidCount(0);
      } else {
        Optional<Integer> count = analyzeList
            .stream()
            .map(QuestionStatisticVo::getStatistic)
            .reduce(Integer::sum);
        if(count.isPresent()){
          analyzeVo.setValidCount(count.get());
        }else{
          analyzeVo.setValidCount(0);
        }
      }
    }
    return (List<QuestionnaireAnalyzeVo>) result;
  }

  /**
   * 生成问卷解析统计信息
   *
   * @param records
   * @return
   */
  private Map<String, List<QuestionStatisticVo>> generateStatisticMap(List<AnswerRecord> records,
      List<QuestionVo> questionVoList) {
    Map<String, List<QuestionStatisticVo>> statisticMap = new HashMap<>();
    for (QuestionVo questionInfo : questionVoList) {
      List<QuestionStatisticVo> result = new ArrayList<>();
      String questionCode = questionInfo.getCode();
      if (questionCode == null){
        continue;
      }
      // 获取专属于当前题目的题目记录信息
      List<AnswerRecord> answerRecords = records
          .stream()
          .filter(record -> record.getQuestionCode().equals(questionInfo.getCode()))
          .collect(Collectors.toList());
      // 生成单选统计信息
      this.generateRadioStatistic(result, answerRecords, questionInfo);
      // 生成多选统计信息
      this.generateCheckboxStatistic(result, answerRecords, questionInfo);
      // 生成级联统计信息
      this.generateSelectStatistic(result, answerRecords, questionInfo);
      statisticMap.put(questionCode, result);
    }
    return statisticMap;
  }

  /**
   * 生成级联select类型的统计信息
   *
   * @param result
   * @param answerRecords
   * @param questionInfo
   */
  private void generateSelectStatistic(List<QuestionStatisticVo> result,
      List<AnswerRecord> answerRecords, QuestionVo questionInfo) {
    /**
     * 1、获取题目类型，并判断是否是select且答题记录也不能为空
     * 2、将答题记录按用户输入值进行分组
     * 3、计算并封装统计信息
     */
    // 第1步=================
    String type = questionInfo.getType();
    if (!("select".equals(type)) || CollectionUtils.isEmpty(answerRecords)) {
      return;
    }
    String questionCode = questionInfo.getCode();
    int allCount = answerRecords.size();
    // 第2步================
    Map<String, List<AnswerRecord>> groupByAnswerValueMap = answerRecords
        .stream()
        .collect(Collectors.groupingBy(AnswerRecord::getAnswerValue));
    // 第3步=================
    for (Entry<String, List<AnswerRecord>> groupEntry : groupByAnswerValueMap.entrySet()) {
      String answerValue = groupEntry.getKey();
      List<AnswerRecord> answerRecordList = groupEntry.getValue();
      int statisticCount = answerRecordList.size();
      double rate = 0.0;
      if (allCount != 0) {
        rate = (double) statisticCount / allCount * 100;
      }
      QuestionStatisticVo statisticResult = new QuestionStatisticVo();
      statisticResult.setQuestionCode(questionCode);
      statisticResult.setAnswerValue(answerValue);
      statisticResult.setStatistic(statisticCount);
      statisticResult.setRate(BigDecimal.valueOf(rate));
      try {
        String[] addressArray = answerValue.split(",");
        statisticResult.setProvince(addressArray[0]);
        statisticResult.setCity(addressArray[1]);
        statisticResult.setArea(addressArray[2]);
      } catch (RuntimeException e) {
        log.error(e.getMessage(), e);
        throw new RuntimeException("省市区格式错误,省市区数据格式必须以逗号分割,例如: xxx省,xx市,xx区");
      }
      result.add(statisticResult);
    }
  }

  /**
   * 生成多选统计信息
   *
   * @param result
   * @param answerRecords
   * @param questionInfo
   */
  private void generateCheckboxStatistic(List<QuestionStatisticVo> result,
      List<AnswerRecord> answerRecords, QuestionVo questionInfo) {
    /**
     * 1、获取题目类型，并判断是否是checkbox
     * 2、获取用户的答题记录
     * 3、获取题目的候选值配置
     * 4、计算并封装统计信息
     */
    // 第1步===================
    String type = questionInfo.getType();
    if (!("checkbox".equals(type))) {
      return;
    }
    String questionCode = questionInfo.getCode();
    int allCount = answerRecords.size();
    // 第2步==================
    List<String> answerValueList = answerRecords
        .stream()
        .map(AnswerRecord::getAnswerValue)
        .collect(Collectors.toList());
    // 第3步=================
    String configJsonValue = questionInfo.getConfigJsonValue();
    List<JSONObject> configList = JSONObject.parseObject(configJsonValue, List.class);
    // 第4步=================
    for (JSONObject config : configList) {
      String configLabel = String.valueOf(config.get("label"));
      String configValue = String.valueOf(config.get("value"));
      long statisticCount = answerValueList
          .stream()
          .filter(answerValue -> Arrays.asList(answerValue.split(",")).contains(configLabel))
          .count();
      double rate = 0.0;
      if (allCount != 0) {
        rate = (double) statisticCount / allCount * 100;
      }
      QuestionStatisticVo statisticResult = new QuestionStatisticVo();
      statisticResult.setQuestionCode(questionCode);
      statisticResult.setStatistic((int) statisticCount);
      statisticResult.setAnswerValue(configValue);
      statisticResult.setLabel(configLabel);
      statisticResult.setRate(BigDecimal.valueOf(rate));
      result.add(statisticResult);
    }
  }


  /**
   * 生成单选类型的统计信息
   *
   * @param result
   * @param answerRecords
   * @param questionInfo
   */
  private void generateRadioStatistic(List<QuestionStatisticVo> result,
      List<AnswerRecord> answerRecords, QuestionVo questionInfo) {
    /**
     * 1、获取题目类型，并判断是否是radio
     * 2、将答题记录按用户输入值进行分组
     * 3、获取题目的候选值配置
     * 4、计算并封装统计信息
     */
    // 第1步=====================
    String type = questionInfo.getType();
    if (!("radio".equals(type))) {
      return;
    }
    String questionCode = questionInfo.getCode();
    int allCount = answerRecords.size();
    // 第2步=====================
    Map<String, List<AnswerRecord>> groupByAnswerValueMap = answerRecords
        .stream()
        .collect(Collectors.groupingBy(AnswerRecord::getAnswerValue));
    // 第3步=====================
    String configJsonValue = questionInfo.getConfigJsonValue();
    List<JSONObject> configList = JSON.parseObject(configJsonValue, List.class);
    // 第4步=====================
    for (JSONObject config : configList) {
      String configValue = String.valueOf(config.get("value"));
      String configLabel = String.valueOf(config.get("label"));
      List<AnswerRecord> answerRecordList = groupByAnswerValueMap
          .getOrDefault(configValue, Collections.EMPTY_LIST);
      int statisticSize = answerRecordList.size();
      double rate = 0.0;
      if (allCount != 0) {
        rate = (double) statisticSize / allCount * 100;
      }
      QuestionStatisticVo statisticResult = new QuestionStatisticVo();
      statisticResult.setAnswerValue(configValue);
      statisticResult.setLabel(configLabel);
      statisticResult.setStatistic(statisticSize);
      statisticResult.setQuestionCode(questionCode);
      statisticResult.setRate(BigDecimal.valueOf(rate));
      result.add(statisticResult);
    }
  }

  /**
   * 分页获取题目详情信息
   *
   * @param pageable
   * @param dto
   * @return
   */
  @Override
  public Page<AnswerRecordVo> pageQuestionDetail(Pageable pageable, AnswerRecordPaginationDto dto) {
    return this.answerRecordService.pageAnswerRecord(pageable, dto);
  }

  /**
   * 获取指定用户的问卷答案
   *
   * @param questionnaireCode
   * @param memberCode
   * @return
   */
  @Override
  public List<QuestionAnswerVo> getQuestionAnswer(String questionnaireCode, String memberCode) {
    /**
     * 1、根据问卷编码获取题目信息
     * 2、获取题目编码集合
     * 3、根据题目编码集合和目标用户账号获取用户答题记录
     * 4、将获取到的答题记录填充到返回结果中
     * */
    // 第1步===============
    List<QuestionVo> questionVoList = this.questionnaireQuestionService
        .listByQuestionnaireCodeOrderBySortIndex(questionnaireCode);
    Collection<QuestionAnswerVo> result = this.nebulaToolkitService
        .copyCollectionByWhiteList(questionVoList, QuestionVo.class, QuestionAnswerVo.class,
            HashSet.class, LinkedList.class);
    // 第2步===============
    List<String> questionCodes = questionVoList.stream().map(QuestionVo::getCode).distinct()
        .collect(Collectors.toList());
    // 第3步==============
    List<AnswerRecordVo> answerRecordVoList = this.answerRecordService
        .getByQuestionCodes(questionCodes, memberCode);
    //  第4步============
    Map<String, AnswerRecordVo> answerRecordVoMap = answerRecordVoList.stream()
        .collect(Collectors.toMap(AnswerRecordVo::getQuestionCode, obj -> obj));
    for (QuestionAnswerVo questionAnswerVo : result) {
      String questionCode = questionAnswerVo.getCode();
      questionAnswerVo.setAnswerRecordVo(answerRecordVoMap.get(questionCode));
    }
    return (List<QuestionAnswerVo>) result;
  }

  /**
   * 小程序分页获取调查问卷
   * rangeStatus 该属性值从mms_app_questionnaire中获取
   * 其中: all对应获取全部问卷
   *       allow_join对应获取可参与的问卷
   *       submitted对应获取已提交的问卷
   *
   * @param pageable
   * @return
   */
  @Override
  public Page<QuestionnaireVo> pageQuestionnaireWithStatus(Pageable pageable, String rangeStatus) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    Page<QuestionnaireVo> page = new Page<>(pageable.getPageNumber(),
        pageable.getPageSize());
    Page<QuestionnaireVo> result = page;
    String pubStatus = "";
    if ("allow_join".equals(rangeStatus)){
      pubStatus= QuestionnaireStatusEnum.RUNNING.getValue();
      String memberCode = this.userSearchHelper.getMemberLogin().getMemberCode();
      // 获取当前用户已提交的问卷编码
      List<String> submitQuestionnaireCodes = this.answerUserRepository.lambdaQuery()
          .eq(AnswerUser::getMemberCode, memberCode)
          .eq(AnswerUser::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
          .list()
          .stream()
          .map(AnswerUser::getQuestionnaireCode)
          .collect(Collectors.toList());
      result = this.questionnaireRepository.appPageByPubStatusWithEnableStatus(page, pubStatus, submitQuestionnaireCodes);
    }else if ("all".equals(rangeStatus)){
      result = this.questionnaireRepository.appPageByPubStatusWithEnableStatus(page, pubStatus, Collections.EMPTY_LIST);
    }else {
      String userAccount = this.loginUserService.getLoginUser().getAccount();
      result = this.questionnaireRepository
          .pageQuestionnaireByUserAccount(page, userAccount);
    }
    this.populateLifeCycleStatus(result.getRecords());
    return result;
  }

  /**
   * 小程序分页获取可参与调查问卷
   *
   * @param pageable
   * @return
   */
  @Override
  public Page<QuestionnaireVo> pageQuestionnaireAllowJoin(Pageable pageable) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    QuestionnairePaginationDto paginationDto = new QuestionnairePaginationDto();
    paginationDto.setPubStatus(QuestionnaireStatusEnum.RUNNING.getValue());
    Page<QuestionnaireVo> page = new Page<>(pageable.getPageNumber(),
        pageable.getPageSize());
    Page<QuestionnaireVo> result = this.questionnaireRepository
        .pageInfo(page, paginationDto);
    for (QuestionnaireVo record : result.getRecords()) {
      record.setLifeCycleStatus(QuestionnaireLifeCycleStatusEnum.ALLOW_JOIN.getValue());
    }
    return result;
  }

  /**
   * 小程序分页获取已提交的调查问卷
   *
   * @param pageable
   * @return
   */
  @Override
  public Page<QuestionnaireVo> pageQuestionnaireBySubmitted(Pageable pageable) {
    AbstractCrmUserIdentity loginUser = this.loginUserService.getAbstractLoginUser();
    String userAccount = loginUser.getAccount();
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    Page<QuestionnaireVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<QuestionnaireVo> result = this.questionnaireRepository
        .pageQuestionnaireByUserAccount(page, userAccount);
    for (QuestionnaireVo record : result.getRecords()) {
      record.setLifeCycleStatus(QuestionnaireLifeCycleStatusEnum.SUBMITTED.getValue());
    }
    return result;
  }

  /**
   * 小程序问卷提交
   *
   * @param dtoList
   */
  @Transactional
  @Override
  public QuestionnaireVo submitQuestionnaire(List<QuestionnaireSubmitDto> dtoList) {
    // 问卷编码
    String memberCode = userSearchHelper.getMemberLogin().getMemberCode();
    String memberPhone = userSearchHelper.getMemberLogin().getMemberPhone();
    String questionnaireCode = dtoList.get(0).getQuestionnaireCode();
    Questionnaire questionnaire = this.questionnaireRepository.getByCode(questionnaireCode);
    BigDecimal obtainPoints = questionnaire.getObtainPoints();

    AnswerUser answerUser = new AnswerUser();
    answerUser.setMemberCode(memberCode);
    answerUser.setQuestionnaireCode(questionnaireCode);
    answerUser.setTenantCode(TenantUtils.getTenantCode());
    answerUser.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    answerUser.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    answerUser.setObtainPoints(obtainPoints);
    answerUser.setTel(memberPhone);
    this.answerUserRepository.save(answerUser);

    List<AnswerRecord> answerRecords = new ArrayList<>();
    for (QuestionnaireSubmitDto submitDto : dtoList) {
      String generateAnswerRecordCode = this.generateCodeService.generateCode("QA", 5).get(0);
      String questionCode = submitDto.getQuestionCode();
      String answerValue = submitDto.getAnswerValue();
      AnswerRecord record = new AnswerRecord();
      record.setQuestionCode(questionCode);
      record.setAnswerValue(answerValue);
      record.setCode(generateAnswerRecordCode);
      record.setMemberCode(memberCode);
      record.setTenantCode(TenantUtils.getTenantCode());
      record.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
      record.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
      answerRecords.add(record);
    }
    this.answerRecordService.batchSave(answerRecords);
    MemberInfoIntegralRecordVo memberInfoIntegralRecordVo = new MemberInfoIntegralRecordVo();
    memberInfoIntegralRecordVo.setMemberCode(memberCode);
    memberInfoIntegralRecordVo.setSource(IntegralSourceEnum.QUESTIONNAIRE.getValue());
    memberInfoIntegralRecordVo.setType(IntegralSourceEnum.QUESTIONNAIRE.getType());
    memberInfoIntegralRecordVo.setIntegral(obtainPoints.intValue());
    this.memberInfoIntegralRecordService.commonInternal(memberInfoIntegralRecordVo);

    QuestionnaireVo result = this.nebulaToolkitService
        .copyObjectByWhiteList(questionnaire, QuestionnaireVo.class, HashSet.class,
            LinkedList.class);
    return result;
  }

  /**
   * 小程序获取当前用户已提交的调查问卷
   *
   * @param questionnaireCode
   * @return
   */
  @Override
  public List<QuestionAnswerVo> getQuestionnaireDetailBySubmit(String questionnaireCode) {
    String account = this.loginUserService.getAbstractLoginUser().getAccount();
    LambdaQueryWrapper<AnswerUser> queryUser = new LambdaQueryWrapper<>();
    queryUser.eq(AnswerUser::getQuestionnaireCode, questionnaireCode)
        .eq(AnswerUser::getCreateAccount, account)
        .last("limit 1");
    AnswerUser answerUser = this.answerUserRepository.getOne(queryUser);
    String memberCode = answerUser.getMemberCode();
    List<QuestionAnswerVo> result = this.getQuestionAnswer(questionnaireCode, memberCode);
    return result;
  }

  /**
   * 填充lifeCycleStatus属性值
   *
   * @param records
   */
  private void populateLifeCycleStatus(List<QuestionnaireVo> records) {
    if (CollectionUtils.isEmpty(records)) {
      return;
    }
    for (QuestionnaireVo record : records) {
      String pubStatus = record.getPubStatus();
      // 已过期 finish
      if (QuestionnaireStatusEnum.FINISH.getValue().equals(pubStatus)) {
        record.setLifeCycleStatus(QuestionnaireLifeCycleStatusEnum.EXPIRED.getValue());
      }
      // 未开始
      if (QuestionnaireStatusEnum.NO_STARTED.getValue().equals(pubStatus)) {
        record.setLifeCycleStatus(QuestionnaireLifeCycleStatusEnum.NO_START.getValue());
      }
      // 可参与
      if (QuestionnaireStatusEnum.RUNNING.getValue().equals(pubStatus)) {
        record.setLifeCycleStatus(QuestionnaireLifeCycleStatusEnum.ALLOW_JOIN.getValue());
      }
    }

    // 判断是否已提交
    List<String> questionnaireCodes = records.stream().map(QuestionnaireVo::getCode).distinct()
        .collect(Collectors.toList());
    UserIdentity loginUser = this.loginUserService.getLoginUser();
    String userAccount = loginUser.getAccount();
    LambdaQueryWrapper<AnswerUser> queryUserQuestionnaire = new LambdaQueryWrapper<>();
    queryUserQuestionnaire.eq(AnswerUser::getCreateAccount, userAccount)
        .in(AnswerUser::getQuestionnaireCode, questionnaireCodes);
    List<AnswerUser> currentUserQuestionnaire = this.answerUserRepository
        .list(queryUserQuestionnaire);
    if (CollectionUtils.isEmpty(currentUserQuestionnaire)) {
      return;
    }
    Set<String> joinQuestionnaireCodes = currentUserQuestionnaire.stream()
        .map(AnswerUser::getQuestionnaireCode)
        .collect(Collectors.toSet());
    for (QuestionnaireVo record : records) {
      String questionnaireCode = record.getCode();
      if (joinQuestionnaireCodes.contains(questionnaireCode)) {
        record.setLifeCycleStatus(QuestionnaireLifeCycleStatusEnum.SUBMITTED.getValue());
      }
    }
  }

  /**
   * 将题目列表中的id信息清空
   *
   * @param questionDtoList
   */
  private void clearQuestionListId(Collection<QuestionnaireQuestionDto> questionDtoList) {
    for (QuestionnaireQuestionDto dto : questionDtoList) {
      dto.setId(StringUtils.EMPTY);
    }
  }

  /**
   * 修改问卷基本信息
   *
   * @param basicInfo
   * @return
   */
  private QuestionnaireVo modifyQuestionnaireBasicInfo(QuestionnaireDto basicInfo) {
    this.modifyValidate(basicInfo);
    Questionnaire questionnaire = this.nebulaToolkitService
        .copyObjectByWhiteList(basicInfo, Questionnaire.class, HashSet.class, LinkedList.class);
    this.questionnaireRepository.saveOrUpdate(questionnaire);
    QuestionnaireVo result = this.nebulaToolkitService
        .copyObjectByWhiteList(questionnaire, QuestionnaireVo.class, HashSet.class,
            LinkedList.class);
    return result;
  }

  /**
   * 填充commitCount属性
   *
   * @param list
   */
  private void populateCommitNum(List<QuestionnaireVo> list) {
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    /**
     * 1、获取状态为已结束和正在进行中的问卷的code编码
     * 2、根据获取到的问卷编码统计问卷提交数
     * 3、填充到对应的Vo中
     * */
    // 第1步============
    List<String> questionnaireCodes = list.stream()
        .filter(obj -> {
          String finishValue = QuestionnaireStatusEnum.FINISH.getValue();
          String runningValue = QuestionnaireStatusEnum.RUNNING.getValue();
          String pubStatus = obj.getPubStatus();
          return finishValue.equals(pubStatus) || runningValue.equals(pubStatus);
        })
        .map(QuestionnaireVo::getCode)
        .distinct()
        .collect(Collectors.toList());
    if (CollectionUtils.isEmpty(questionnaireCodes)) {
      return;
    }
    // 第2步=============
    List<QuestionCountVo> countVoList = this.answerUserRepository
        .countCommitGroupByQuestionnaireCode(questionnaireCodes);
    if (CollectionUtils.isEmpty(countVoList)) {
      return;
    }
    // 第3步==============
    Map<String, Integer> countMap = countVoList.stream()
        .distinct()
        .collect(Collectors.toMap(QuestionCountVo::getCode, QuestionCountVo::getCommitCount));
    for (QuestionnaireVo questionnaireVo : list) {
      String code = questionnaireVo.getCode();
      Integer commitCount = countMap.getOrDefault(code, 0);
      questionnaireVo.setCommitCount(commitCount);
    }
  }

  /**
   * 新增一个问卷基本信息
   *
   * @param dto
   * @param statusEnum
   * @return
   */
  private QuestionnaireVo createBasicQuestionnaire(QuestionnaireDto dto,
      QuestionnaireStatusEnum statusEnum) {
    // 新增校验
    this.createValidate(dto);
    // dto转entity
    Questionnaire questionnaire = this.nebulaToolkitService
        .copyObjectByWhiteList(dto, Questionnaire.class, HashSet.class,
            LinkedList.class);
    questionnaire.setCode(this.generateCodeService.generateCode("WJ", 5).get(0));
    questionnaire.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    questionnaire.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    questionnaire.setTenantCode(TenantUtils.getTenantCode());
    questionnaire.setPubStatus(statusEnum.getValue());
    this.questionnaireRepository.save(questionnaire);
    // entity转vo
    QuestionnaireVo result = this.nebulaToolkitService
        .copyObjectByWhiteList(questionnaire, QuestionnaireVo.class, HashSet.class,
            LinkedList.class);
    return result;
  }

  /**
   * 新增问卷校验
   *
   * @param dto
   */
  private void createValidate(QuestionnaireDto dto) {
    Validate.notBlank(dto.getTitle(), "问卷标题不能为空");
    Validate.notNull(dto.getStartTime(), "开始时间不能为空");
    Validate.notNull(dto.getEndTime(), "结束时间不能为空");
    Date currentTime = new Date(System.currentTimeMillis());
    Validate.isTrue(dto.getEndTime().compareTo(currentTime) == 1, "结束时间不能小于当前时间");
  }

  /**
   * 修改校验
   *
   * @param dto
   */
  private void modifyValidate(QuestionnaireDto dto) {
    String questionnaireCode = dto.getCode();
    Questionnaire questionnaire = this.questionnaireRepository.getByCode(questionnaireCode);
    String pubStatus = questionnaire.getPubStatus();
    Validate.isTrue(! QuestionnaireStatusEnum.RUNNING.getValue().equals(pubStatus), "正在进行中的问卷不能编辑");
    Validate.notBlank(dto.getCode(), "问卷编码不能为空");
  }
}
