package com.biz.crm.member.business.member.local.service.internal;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.member.business.member.local.entity.RaffleBaseConfig;
import com.biz.crm.member.business.member.local.entity.RaffleManage;
import com.biz.crm.member.business.member.local.repository.RaffleBaseConfigRepository;
import com.biz.crm.member.business.member.local.repository.RaffleManageRepository;
import com.biz.crm.member.business.member.local.service.RaffleManageService;
import com.biz.crm.member.business.member.local.service.RafflePrizeService;
import com.biz.crm.member.business.member.sdk.dto.*;
import com.biz.crm.member.business.member.sdk.enums.PrizeInventoryEnum;
import com.biz.crm.member.business.member.sdk.enums.PrizeOverSelectEnum;
import com.biz.crm.member.business.member.sdk.enums.PrizeTypeEnum;
import com.biz.crm.member.business.member.sdk.vo.*;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 大转盘业务实现类
 *
 * @Description
 * @Author monroe
 * @Date 2023/6/19 16:02
 **/
@Service
public class RaffleManageServiceImpl implements RaffleManageService {

    @Autowired
    private RaffleManageRepository raffleManageRepository;
    @Autowired
    private RaffleBaseConfigRepository raffleBaseConfigRepository;
    @Autowired
    private RafflePrizeService rafflePrizeService;

    @Autowired
    private NebulaToolkitService nebulaToolkitService;
    @Autowired
    private RedisMutexService redisMutexService;

    /**
     * 大转盘编码redis key
     * 参数：租户
     */
    public static final String MMS_RAFFLE_MANAGE_CODE_REDIS_KEY = "mms:raffle:manage:code:index:%s";

    /**
     * 大转盘管理编码前缀
     */
    public static final String MMS_RAFFLE_MANAGE_CODE_PREFIX = "ZP";

    /**
     * 生成编码
     */
    private String generateCode(String tenantCode) {
        String redisKey = String.format(MMS_RAFFLE_MANAGE_CODE_REDIS_KEY, tenantCode);
        String index = redisMutexService.getAndIncrement(redisKey, 1, 6);
        SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
        return StringUtils.join(MMS_RAFFLE_MANAGE_CODE_PREFIX, format.format(new Date()), index);
    }


    @Override
    public Page<RaffleManagePageVo> findByConditions(Pageable pageable, RaffleManagePageDto raffleManagePageDto) {
        if (Objects.isNull(raffleManagePageDto)) {
            raffleManagePageDto = new RaffleManagePageDto();
        }
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
        Page<RaffleManagePageVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.raffleManageRepository.findByConditions(page, raffleManagePageDto);
    }

    @Override
    public RaffleManageVo findDetailById(String id, String code) {
        if (StringUtils.isBlank(id) && StringUtils.isBlank(code)) {
            return null;
        }

        //根据id或者编码查询活动
        RaffleManage raffleManage = null;
        if (StringUtils.isNotBlank(id)) {
            raffleManage = this.raffleManageRepository.getById(id);
        } else {
            raffleManage = this.raffleManageRepository.lambdaQuery().eq(RaffleManage::getCode, code)
                    .eq(RaffleManage::getTenantCode, TenantUtils.getTenantCode())
                    .eq(RaffleManage::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                    .eq(RaffleManage::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
                    .one();
        }

        Validate.notNull(raffleManage, "活动不存在");
        RaffleManageVo raffleManageVo = this.nebulaToolkitService.copyObjectByWhiteList(raffleManage, RaffleManageVo.class, HashSet.class, LinkedList.class);

        //查询抽奖次数配置
        List<RaffleBaseConfig> baseConfigs = this.raffleBaseConfigRepository.lambdaQuery().eq(RaffleBaseConfig::getRaffleCode, raffleManageVo.getCode()).eq(RaffleBaseConfig::getTenantCode, raffleManage.getTenantCode())
                .eq(RaffleBaseConfig::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
        if (CollectionUtils.isNotEmpty(baseConfigs)) {
            List<RaffleBaseConfigVo> raffleBaseConfigVos = (List<RaffleBaseConfigVo>) this.nebulaToolkitService.copyCollectionByWhiteList(baseConfigs, RaffleBaseConfig.class, RaffleBaseConfigVo.class, HashSet.class, LinkedList.class);
            for (RaffleBaseConfigVo raffleBaseConfigVo : raffleBaseConfigVos) {
                this.transtStringToList(raffleBaseConfigVo, null);
            }
            raffleManageVo.setBaseConfigs(raffleBaseConfigVos);
        }

        //查询奖品配置信息
        List<RafflePrizeVo> prizeVos = rafflePrizeService.findByRaffleCode(TenantUtils.getTenantCode(), raffleManage.getCode());
        if (CollectionUtils.isNotEmpty(prizeVos)) {
            for (RafflePrizeVo prizeVo : prizeVos) {
                this.transtStringToList(null, prizeVo);
            }
            raffleManageVo.setRafflePrizes(prizeVos);
        }

        return raffleManageVo;
    }

    @Transactional
    @Override
    public RaffleManageVo create(RaffleManageDto raffleManageDto) {
        this.createValidate(raffleManageDto);
        raffleManageDto.setCode(this.generateCode(TenantUtils.getTenantCode()));
        raffleManageDto.setTenantCode(TenantUtils.getTenantCode());
        raffleManageDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        raffleManageDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        RaffleManage raffleManage = this.nebulaToolkitService.copyObjectByWhiteList(raffleManageDto, RaffleManage.class, HashSet.class, LinkedList.class);
        this.raffleManageRepository.saveOrUpdate(raffleManage);
        //基础配置
        List<RaffleBaseConfigDto> baseConfigs = raffleManageDto.getBaseConfigs();
        if (CollectionUtils.isNotEmpty(baseConfigs)) {
            for (RaffleBaseConfigDto baseConfig : baseConfigs) {
                baseConfig.setId(null);
                baseConfig.setTenantCode(TenantUtils.getTenantCode());
                baseConfig.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                baseConfig.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                baseConfig.setRaffleCode(raffleManage.getCode());
            }
            List<RaffleBaseConfig> raffleBaseConfigs = (List<RaffleBaseConfig>) this.nebulaToolkitService.copyCollectionByWhiteList(baseConfigs, RaffleBaseConfigDto.class, RaffleBaseConfig.class, HashSet.class, LinkedList.class);
            this.raffleBaseConfigRepository.saveOrUpdateBatch(raffleBaseConfigs);
        }
        //奖品信息
        List<RafflePrizeDto> rafflePrizes = raffleManageDto.getRafflePrizes();
        if (CollectionUtils.isNotEmpty(rafflePrizes)) {
            for (RafflePrizeDto rafflePrize : rafflePrizes) {
                rafflePrize.setRaffleCode(raffleManage.getCode());
            }
            this.rafflePrizeService.saveBatch(rafflePrizes);
        }
        return this.nebulaToolkitService.copyObjectByWhiteList(raffleManage, RaffleManageVo.class, HashSet.class, LinkedList.class);
    }

    private void createValidate(RaffleManageDto raffleManageDto) {

        Validate.notNull(raffleManageDto, "新增实体不能为空");
        Validate.notBlank(raffleManageDto.getActivityName(), "活动名称不能为空");
        Validate.notBlank(raffleManageDto.getBackgroundFileId(), "背景图片不能为空");
        Validate.notBlank(raffleManageDto.getFileId(), "转盘图片不能为空");
        Validate.notBlank(raffleManageDto.getStartButtonFileId(), "开始按钮图片不能为空");
        Validate.notBlank(raffleManageDto.getCuttingLineFileId(), "分割线图片不能为空");
        Validate.notNull(raffleManageDto.getStartDate(), "活动开始时间不能为空");
        Validate.notNull(raffleManageDto.getEndDate(), "活动结束时间不能为空");

        List<RaffleBaseConfigDto> baseConfigs = raffleManageDto.getBaseConfigs();
        if (CollectionUtils.isEmpty(baseConfigs)) {
            throw new RuntimeException("未配置大转盘抽奖次数");
        }
        List<RaffleMemberDto> baseconfigMemberLevels = new ArrayList<>();
        for (RaffleBaseConfigDto baseConfig : baseConfigs) {
            //校验会员级别配置重复
            baseconfigMemberLevels.addAll(baseConfig.getMemberLevels());
            //转换会员等级格式，list -> string
            this.transtListToString(baseConfig.getMemberLevels(), baseConfig, null);
            Validate.notBlank(baseConfig.getMemberLevelCode(), "会员等级不能为空");
            Validate.notNull(baseConfig.getOneRaffleNum(), "一段抽奖次数不能为空");
            Validate.notNull(baseConfig.getOneRaffleUseIntegral(), "一段每次消耗积分不能为空");

        }

        List<String> baseConfigMemberCodeList = baseconfigMemberLevels.stream().map(RaffleMemberDto::getMemberLevelCode).distinct().collect(Collectors.toList());
        Validate.isTrue(baseConfigMemberCodeList.size() == baseconfigMemberLevels.size(), "抽奖次数,会员级别配置重复");

        List<RafflePrizeDto> rafflePrizes = raffleManageDto.getRafflePrizes();
        if (CollectionUtils.isEmpty(rafflePrizes)) {
            throw new RuntimeException("未配置大转盘奖品配置信息");
        }
        List<RaffleMemberDto> rafflePrizeMemberLevels = new ArrayList<>();
        for (RafflePrizeDto rafflePrize : rafflePrizes) {
            //校验会员级别配置重复
            rafflePrizeMemberLevels.addAll(rafflePrize.getMemberLevels());
            this.transtListToString(rafflePrize.getMemberLevels(), null, rafflePrize);
            Validate.notBlank(rafflePrize.getMemberLevelCode(), "会员等级不能为空");
            Validate.notBlank(rafflePrize.getPrizeOverSelect(), "奖品发完不能为空");
            List<RafflePrizeDetailsDto> details = rafflePrize.getDetails();
            validatePrizeDetail(rafflePrize, details);
        }

        List<String> prizeMemberCodeList = rafflePrizeMemberLevels.stream().map(RaffleMemberDto::getMemberLevelCode).distinct().collect(Collectors.toList());
        Validate.isTrue(prizeMemberCodeList.size() == rafflePrizeMemberLevels.size(), "奖品配置,会员级别配置重复");


    }

    private void transtListToString(List<RaffleMemberDto> memberLevels, RaffleBaseConfigDto configDto, RafflePrizeDto prizeDto) {
        if (CollectionUtils.isEmpty(memberLevels)) {
            throw new RuntimeException("会员等级不能为空");
        }
        String memberLevelNames = memberLevels.stream().map(RaffleMemberDto::getMemberLevelName).collect(Collectors.joining(","));
        String memberLevelCodes = memberLevels.stream().map(RaffleMemberDto::getMemberLevelCode).collect(Collectors.joining(","));

        if (!Objects.isNull(configDto)) {
            configDto.setMemberLevelCode(memberLevelCodes);
            configDto.setMemberLevelName(memberLevelNames);
            configDto.setMemberLevelJson(JSON.toJSONString(memberLevels));
        } else {
            prizeDto.setMemberLevelCode(memberLevelCodes);
            prizeDto.setMemberLevelName(memberLevelNames);
            prizeDto.setMemberLevelJson(JSON.toJSONString(memberLevels));
        }

    }

    private void transtStringToList(RaffleBaseConfigVo baseConfig, RafflePrizeVo prizeVo) {

        if (!Objects.isNull(baseConfig)) {
            baseConfig.setMemberLevels(JSONArray.parseArray(baseConfig.getMemberLevelJson(), RaffleMemberDto.class));
        } else {
            prizeVo.setMemberLevels(JSONArray.parseArray(prizeVo.getMemberLevelJson(), RaffleMemberDto.class));
        }


    }

    private void validatePrizeDetail(RafflePrizeDto rafflePrize, List<RafflePrizeDetailsDto> details) {
        if (CollectionUtils.isNotEmpty(details)) {

            //基础校验
            this.validateDefault(details);

            if (PrizeOverSelectEnum.NO_INVENTORY.getCode().equals(rafflePrize.getPrizeOverSelect())) {
                long count = details.stream().filter(e -> e.getPrizeFlag() && PrizeInventoryEnum.WITH_OUT.getCode().equals(e.getInventory())).count();
                Validate.isTrue(count == 1, "奖品发完选择无库存奖品时，必须配置无库存奖品");
            }
            if (PrizeOverSelectEnum.NONE.getCode().equals(rafflePrize.getPrizeOverSelect())) {
                long count = details.stream().filter(e -> PrizeTypeEnum.THANKS.getCode().equals(e.getPrizeType())).count();
                Validate.isTrue(count == 1, "奖品发完选择不中奖时，必须配置谢谢参与");
            }

        }
    }

    public void validateDefault(List<RafflePrizeDetailsDto> details) {
        //校验必填项
        BigDecimal rate = BigDecimal.ZERO;
        for (RafflePrizeDetailsDto detail : details) {
            this.baseVaildate(detail);
            //中奖概率相加必须等于1
            rate = rate.add(detail.getPrizeProb().divide(new BigDecimal(100)));
        }

        if (rate.compareTo(new BigDecimal("1")) != 0) {
            throw new RuntimeException("中奖概率之和必须等于1");
        }

    }

    private void baseVaildate(RafflePrizeDetailsDto detail) {
        //只有有库存才校验
        if (PrizeInventoryEnum.EXITS.getCode().equals(detail.getInventory())) {
            Validate.notNull(detail.getPutCopies(), "投放份数不能为空");
            Validate.notNull(detail.getPutNum(), "投放数量不能为空");
            Validate.notNull(detail.getQuantity(), "数量不能为空");
        }
        Validate.notBlank(detail.getPrizeCode(), "奖品编码不能为空");
        Validate.notNull(detail.getPrizeProb(), "中奖概率不能为空");
        Validate.notBlank(detail.getPrizeType(), "奖品类型不能为空");
    }


    @Transactional
    @Override
    public RaffleManageVo update(RaffleManageDto raffleManageDto) {

        this.updateValidate(raffleManageDto);
        raffleManageDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        raffleManageDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        raffleManageDto.setTenantCode(TenantUtils.getTenantCode());
        RaffleManage raffleManage = this.nebulaToolkitService.copyObjectByWhiteList(raffleManageDto, RaffleManage.class, HashSet.class, LinkedList.class);
        this.raffleManageRepository.saveOrUpdate(raffleManage);

        //逻辑删除基础配置
        this.raffleBaseConfigRepository.lambdaUpdate().set(RaffleBaseConfig::getDelFlag, DelFlagStatusEnum.DELETE.getCode()).eq(RaffleBaseConfig::getRaffleCode, raffleManage.getCode()).update();
        List<RaffleBaseConfigDto> baseConfigs = raffleManageDto.getBaseConfigs();
        if (CollectionUtils.isNotEmpty(baseConfigs)) {
            for (RaffleBaseConfigDto baseConfig : baseConfigs) {
                baseConfig.setId(null);
                baseConfig.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                baseConfig.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                baseConfig.setTenantCode(TenantUtils.getTenantCode());
                baseConfig.setRaffleCode(raffleManage.getCode());
            }
            List<RaffleBaseConfig> raffleBaseConfigs = (List<RaffleBaseConfig>) this.nebulaToolkitService.copyCollectionByWhiteList(baseConfigs, RaffleBaseConfigDto.class, RaffleBaseConfig.class, HashSet.class, LinkedList.class);
            this.raffleBaseConfigRepository.saveBatch(raffleBaseConfigs);
        }

        //修改奖品信息
        List<RafflePrizeDto> rafflePrizes = raffleManageDto.getRafflePrizes();
        this.rafflePrizeService.updateBatch(rafflePrizes, raffleManage.getCode());


        return this.nebulaToolkitService.copyObjectByWhiteList(raffleManage, RaffleManageVo.class, HashSet.class, LinkedList.class);
    }

    /**
     * 编辑参数校验
     *
     * @param raffleManageDto
     */
    private void updateValidate(RaffleManageDto raffleManageDto) {

        Validate.notNull(raffleManageDto, "编辑参数实体不能为空");
        Validate.notBlank(raffleManageDto.getActivityName(), "活动名称不能为空");
        Validate.notBlank(raffleManageDto.getBackgroundFileId(), "背景图片不能为空");
        Validate.notBlank(raffleManageDto.getFileId(), "转盘图片不能为空");
        Validate.notBlank(raffleManageDto.getStartButtonFileId(), "开始按钮图片不能为空");
        Validate.notBlank(raffleManageDto.getCuttingLineFileId(), "分割线图片不能为空");
        Validate.notNull(raffleManageDto.getStartDate(), "活动开始时间不能为空");
        Validate.notNull(raffleManageDto.getEndDate(), "活动结束时间不能为空");
        Validate.notBlank(raffleManageDto.getId(), "编辑时，主键id不能为空");
        Validate.notBlank(raffleManageDto.getCode(), "编辑时，编码不能为空");

        List<RaffleBaseConfigDto> baseConfigs = raffleManageDto.getBaseConfigs();
        if (CollectionUtils.isNotEmpty(baseConfigs)) {
            for (RaffleBaseConfigDto baseConfig : baseConfigs) {
                //转换会员等级格式，list -> string
                this.transtListToString(baseConfig.getMemberLevels(), baseConfig, null);
                Validate.notBlank(baseConfig.getMemberLevelCode(), "会员等级不能为空");
                Validate.notNull(baseConfig.getOneRaffleNum(), "一段抽奖次数不能为空");
                Validate.notNull(baseConfig.getOneRaffleUseIntegral(), "一段每次消耗积分不能为空");
            }
        }

        List<RafflePrizeDto> rafflePrizes = raffleManageDto.getRafflePrizes();
        if (CollectionUtils.isNotEmpty(rafflePrizes)) {
            for (RafflePrizeDto rafflePrize : rafflePrizes) {
                //转换会员等级格式，list -> string
                this.transtListToString(rafflePrize.getMemberLevels(), null, rafflePrize);

                Validate.notBlank(rafflePrize.getMemberLevelCode(), "会员等级不能为空");
                Validate.notBlank(rafflePrize.getPrizeOverSelect(), "奖品发完不能为空");

                //校验奖项
                List<RafflePrizeDetailsDto> details = rafflePrize.getDetails();
                this.validatePrizeDetail(rafflePrize, details);
            }
        }

    }

    @Override
    public void enableBatch(List<String> ids) {

        if (CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.raffleManageRepository.lambdaUpdate().set(RaffleManage::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
                .in(RaffleManage::getId, ids).update();

    }

    @Override
    public void disableBatch(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.raffleManageRepository.lambdaUpdate().set(RaffleManage::getEnableStatus, EnableStatusEnum.DISABLE.getCode())
                .in(RaffleManage::getId, ids).update();
    }

    @Transactional
    @Override
    public void deleteBatch(List<String> ids) {

        if (CollectionUtils.isEmpty(ids)) {
            return;
        }

        List<RaffleManage> raffleManages = this.raffleManageRepository.lambdaQuery().in(RaffleManage::getId, ids).list();
        if (CollectionUtils.isEmpty(raffleManages)) {
            return;
        }
        List<String> codeList = raffleManages.stream().map(RaffleManage::getCode).collect(Collectors.toList());
        //删除大转盘基础信息
        this.raffleManageRepository.lambdaUpdate().set(RaffleManage::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
                .in(RaffleManage::getId, ids).update();

        //删除基础配置
        this.raffleBaseConfigRepository.lambdaUpdate().set(RaffleBaseConfig::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
                .in(RaffleBaseConfig::getRaffleCode, codeList);

        //删除奖品信息
        this.rafflePrizeService.deleteBatch(codeList);


    }

    @Override
    public RaffleManageVo findByActivityCode(String activityCode) {

        if (StringUtils.isBlank(activityCode)) {
            return null;
        }
        RaffleManage one = this.raffleManageRepository.lambdaQuery().eq(RaffleManage::getCode, activityCode).one();
        return this.nebulaToolkitService.copyObjectByWhiteList(one, RaffleManageVo.class, HashSet.class, LinkedList.class);
    }
}
