package com.biz.crm.member.business.member.local.service.internal;

import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.ObjectUtil;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.member.business.member.local.entity.PrizeEntity;
import com.biz.crm.member.business.member.local.entity.RafflePrizeDetails;
import com.biz.crm.member.business.member.local.repository.RafflePrizeDetailsRepository;
import com.biz.crm.member.business.member.local.repository.RafflePrizeRecordRepository;
import com.biz.crm.member.business.member.local.service.PrizeService;
import com.biz.crm.member.business.member.local.service.RafflePrizeDetailsService;
import com.biz.crm.member.business.member.local.utils.AssertUtils;
import com.biz.crm.member.business.member.sdk.enums.PrizeInventoryEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * TODO
 *
 * @Description
 * @Author monroe
 * @Date 2023/6/28 11:40
 **/
@Service
public class RafflePrizeDetailsServiceImpl implements RafflePrizeDetailsService {

    @Autowired
    private RafflePrizeDetailsRepository rafflePrizeDetailsRepository;

    @Autowired(required = false)
    private PrizeService prizeService;

    @Autowired
    private NebulaToolkitService nebulaToolkitService;
    @Autowired
    private RedisMutexService redisMutexService;


    public static final String MMS_RAFFLE_PRIZE_DETAIL_CODE_REDIS_KEY = "mms:raffle:prize:detail:code:index:%s";

    /**
     * 大转盘奖品信息编码前缀
     */
    public static final String MMS_RAFFLE_PRIZE_DETAIL_CODE_PREFIX = "RPD";


    /**
     * 生成编码
     */
    private String generateCode(String tenantCode) {
        String redisKey = String.format(MMS_RAFFLE_PRIZE_DETAIL_CODE_REDIS_KEY, tenantCode);
        String index = redisMutexService.getAndIncrement(redisKey, 1, 6);
        SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
        return StringUtils.join(MMS_RAFFLE_PRIZE_DETAIL_CODE_PREFIX, format.format(new Date()), index);
    }

    @Override
    public List<RafflePrizeDetails> findByRafflePrizeCodes(List<String> prizeCodes) {

        return this.rafflePrizeDetailsRepository.lambdaQuery().in(RafflePrizeDetails::getRafflePrizeCode,prizeCodes)
                .eq(RafflePrizeDetails::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    @Override
    public void saveBatch(List<RafflePrizeDetails> rafflePrizeDetails) {

        //校验奖品库存是否充足
        this.validateInventoryPrize(rafflePrizeDetails);

        List<String> prizeCodes = new ArrayList<>();
        for (RafflePrizeDetails rafflePrizeDetail : rafflePrizeDetails) {
            rafflePrizeDetail.setCode(this.generateCode(TenantUtils.getTenantCode()));
        }

        this.rafflePrizeDetailsRepository.saveOrUpdateBatch(rafflePrizeDetails);
    }


    /**
     * 校验奖品库存是否充足
     * @param rafflePrizeDetails
     */
    public void validateInventoryPrize(List<RafflePrizeDetails> rafflePrizeDetails) {
        //添加奖品分组统计
        Map<String, BigDecimal> putMap = rafflePrizeDetails.stream().filter(e-> StringUtils.isNotBlank(e.getPrizeCode()) && ObjectUtil.isNotNull(e.getPutNum()))
                .collect(Collectors.groupingBy(x-> x.getPrizeCode(),
                Collectors.reducing(BigDecimal.ZERO,
                        t->t.getPutNum(), (v1, v2) -> v1.add(v2))));

        //奖品编码集合
        List<String> prizeCodes = rafflePrizeDetails.stream().map(RafflePrizeDetails::getPrizeCode).collect(Collectors.toList());
        //查询奖品库存数量
        List<PrizeEntity> prizeEntities = this.prizeService.findByCodes(prizeCodes);
        if(CollectionUtils.isEmpty(prizeEntities)) {
            throw new IllegalArgumentException("奖品不存在！");
        }
        //分组
        Map<String, PrizeEntity> prizeMap = prizeEntities.stream().collect(Collectors.toMap(PrizeEntity::getCode, Function.identity()));

        putMap.forEach((k,v) -> {
            PrizeEntity prizeEntity = prizeMap.get(k);
            Validate.notNull(prizeEntity,"奖品不存在");
            if(PrizeInventoryEnum.EXITS.getCode().equals(prizeEntity.getInventory())) {
                AssertUtils.isTrue(v.compareTo(prizeEntity.getInventoryNum()) <= 0, "投放奖品:" + prizeEntity.getPrizeName() + "超过库存数量");
            }
        });

        //扣减奖品的库存
        if(MapUtil.isNotEmpty(putMap)) {
            this.prizeService.updateInventory(putMap);
        }

    }

    @Override
    public void deleteBatch(List<String> prizeCodeList) {
        this.rafflePrizeDetailsRepository.lambdaUpdate().set(RafflePrizeDetails::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
                .in(RafflePrizeDetails::getRafflePrizeCode,prizeCodeList).update();
    }

    @Override
    public RafflePrizeDetails detail(String detailId) {
        return this.rafflePrizeDetailsRepository.getById(detailId);
    }

    @Override
    public void save(RafflePrizeDetails rafflePrizeDetails) {
        this.rafflePrizeDetailsRepository.saveOrUpdate(rafflePrizeDetails);
    }

    @Override
    public void deleteByRaffleCode(String code) {
        if(StringUtils.isBlank(code)) {
            return;
        }

        this.rafflePrizeDetailsRepository.lambdaUpdate().set(RafflePrizeDetails::getDelFlag,DelFlagStatusEnum.DELETE.getCode())
                .eq(RafflePrizeDetails::getRaffleCode,code).update();
    }

    @Override
    public void delete(String id) {
        if(StringUtils.isBlank(id)) {
            return;
        }
        this.rafflePrizeDetailsRepository.removeById(id);
    }

    @Override
    public int reduceInventory(String code, BigDecimal num) {
        return this.rafflePrizeDetailsRepository.reduceInventory(code,num);
    }

}
