package com.biz.crm.member.business.member.local.service.internal;

import cn.hutool.core.collection.CollUtil;
import com.alibaba.fastjson.JSON;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.member.business.member.local.entity.PrizeEntity;
import com.biz.crm.member.business.member.local.entity.RafflePrizeDetails;
import com.biz.crm.member.business.member.local.entity.RafflePrizeEntity;
import com.biz.crm.member.business.member.local.repository.RafflePrizeRepository;
import com.biz.crm.member.business.member.local.service.PrizeService;
import com.biz.crm.member.business.member.local.service.RafflePrizeDetailsService;
import com.biz.crm.member.business.member.local.service.RafflePrizeService;
import com.biz.crm.member.business.member.sdk.constants.PrizeRedisKeys;
import com.biz.crm.member.business.member.sdk.dto.RafflePrizeDetailsDto;
import com.biz.crm.member.business.member.sdk.dto.RafflePrizeDto;
import com.biz.crm.member.business.member.sdk.vo.RafflePrizeDetailsVo;
import com.biz.crm.member.business.member.sdk.vo.RafflePrizeVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.gson.JsonObject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * TODO
 *
 * @Description
 * @Author monroe
 * @Date 2023/6/19 17:50
 **/
@Service
public class RafflePrizeServiceImpl implements RafflePrizeService {

    @Autowired
    private RafflePrizeRepository rafflePrizeRepository;
    @Autowired
    private RafflePrizeDetailsService rafflePrizeDetailsService;
    @Autowired
    private PrizeService prizeService;

    @Autowired
    private NebulaToolkitService nebulaToolkitService;
    @Autowired
    private RedisMutexService redisMutexService;

    /**
     * 大转盘奖品信息编码redis key
     * 参数：租户
     */
    public static final String MMS_RAFFLE_PRIZE_CODE_REDIS_KEY = "mms:raffle:prize:code:index:%s";

    /**
     * 大转盘奖品信息编码前缀
     */
    public static final String MMS_RAFFLE_PRIZE_CODE_PREFIX = "RP";


    /**
     * 生成编码
     */
    private String generateCode(String tenantCode) {
        String redisKey = String.format(MMS_RAFFLE_PRIZE_CODE_REDIS_KEY, tenantCode);
        String index = redisMutexService.getAndIncrement(redisKey, 1, 6);
        SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
        return StringUtils.join(MMS_RAFFLE_PRIZE_CODE_PREFIX, format.format(new Date()), index);
    }

    @Override
    public List<RafflePrizeVo> findByRaffleCode(String tenantCode, String code) {
        //根据大转盘编码查询所有的会员等级配置的奖品
        List<RafflePrizeEntity> entities = this.rafflePrizeRepository.lambdaQuery().eq(RafflePrizeEntity::getRaffleCode, code)
                .eq(RafflePrizeEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(RafflePrizeEntity::getTenantCode, tenantCode).list();

        if (CollectionUtils.isEmpty(entities)) {
            return null;
        }
        List<RafflePrizeVo> rafflePrizeVos = (List<RafflePrizeVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, RafflePrizeEntity.class, RafflePrizeVo.class, HashSet.class, LinkedList.class);

        //根据所有奖品配置编码查询所有奖项
        List<String> rafflePrizeCodes = rafflePrizeVos.stream().map(RafflePrizeVo::getCode).collect(Collectors.toList());
        List<RafflePrizeDetails> details = rafflePrizeDetailsService.findByRafflePrizeCodes(rafflePrizeCodes);

        if (!CollectionUtils.isEmpty(details)) {
            List<RafflePrizeDetailsVo> rafflePrizeDetailsVos = (List<RafflePrizeDetailsVo>) this.nebulaToolkitService.copyCollectionByWhiteList(details, RafflePrizeDetails.class, RafflePrizeDetailsVo.class, HashSet.class, LinkedList.class);

            //查询所有奖品信息封装k,v
            List<String> prizeCodes = rafflePrizeDetailsVos.stream().filter(e -> StringUtils.isNotBlank(e.getPrizeCode())).map(RafflePrizeDetailsVo::getPrizeCode).collect(Collectors.toList());
            List<PrizeEntity> prizeEntities = prizeService.findByCodes(prizeCodes);
            //每一个奖项，填充实时奖品信息
            this.fillPrizeInfo(rafflePrizeDetailsVos, prizeEntities);
            Map<String, List<RafflePrizeDetailsVo>> detailMap = rafflePrizeDetailsVos.stream().collect(Collectors.groupingBy(RafflePrizeDetailsVo::getRafflePrizeCode));
            rafflePrizeVos.stream().forEach(e -> {
                e.setDetails(detailMap.get(e.getCode()));
            });
            return rafflePrizeVos;
        }


        return rafflePrizeVos;
    }

    /**
     * 填充奖品信息
     *
     * @param rafflePrizeDetailsVos
     * @param prizeEntities
     */
    public void fillPrizeInfo(List<RafflePrizeDetailsVo> rafflePrizeDetailsVos, List<PrizeEntity> prizeEntities) {
        if (CollectionUtils.isNotEmpty(prizeEntities)) {

            Map<String, PrizeEntity> prizeEntityMap = prizeEntities.stream().collect(Collectors.toMap(PrizeEntity::getCode, Function.identity()));
            //遍历填充奖品信息
            for (RafflePrizeDetailsVo detailsVo : rafflePrizeDetailsVos) {
                //填充单价，单位，总价，总价值
                PrizeEntity prize = prizeEntityMap.get(detailsVo.getPrizeCode());
                if (Objects.isNull(prize)) {
                    continue;
                }
                detailsVo.setUnitName(prize.getUnitName());
                detailsVo.setUnitPrice(prize.getUnitPrice());
                detailsVo.setPrizeType(prize.getPrizeType());
                detailsVo.setPrizeTypeName(prize.getPrizeTypeName());
                detailsVo.setInventory(prize.getInventory());
                detailsVo.setClaimValidate(prize.getReceiveTimeLimit());
                detailsVo.setUseValidate(prize.getVirtualTimeLimit());
                //使用有效期
                detailsVo.setUseValidate(prize.getVirtualTimeLimit());
                //领取有效期
                detailsVo.setClaimValidate(prize.getReceiveTimeLimit());
                if (Objects.isNull(prize.getUnitPrice()) || Objects.isNull(detailsVo.getQuantity())) {
                    continue;
                }
                //总价
                detailsVo.setTotalPrice(prize.getUnitPrice().multiply(detailsVo.getQuantity()));
                //投放总价值
                detailsVo.setTotalValue(detailsVo.getTotalPrice().multiply(detailsVo.getPutCopies()));
            }
        }
    }

    @Override
    @Transactional
    public void saveBatch(List<RafflePrizeDto> rafflePrizes) {

        //大转盘奖品信息为空
        if (CollectionUtils.isEmpty(rafflePrizes)) {
            return;
        }
        String tenantCode = TenantUtils.getTenantCode();
        List<RafflePrizeDetailsDto> detailsDtos = new ArrayList<>();
        //构建大转盘奖品信息
        for (RafflePrizeDto rafflePrize : rafflePrizes) {
            if (StringUtils.isBlank(rafflePrize.getCode())) {
                rafflePrize.setCode(this.generateCode(TenantUtils.getTenantCode()));
            }
            rafflePrize.setTenantCode(tenantCode);
            rafflePrize.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            rafflePrize.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            List<RafflePrizeDetailsDto> details = rafflePrize.getDetails();
            //构建奖项信息
            this.buildPrizeDetials(tenantCode, detailsDtos, rafflePrize, details);
            //构建抽奖池
            this.buildPrizePoll(rafflePrize);
        }
        //保存奖品配置信息
        List<RafflePrizeEntity> rafflePrizeEntities = (List<RafflePrizeEntity>) this.nebulaToolkitService.copyCollectionByWhiteList(rafflePrizes, RafflePrizeDto.class, RafflePrizeEntity.class, HashSet.class, LinkedList.class);
        this.rafflePrizeRepository.saveOrUpdateBatch(rafflePrizeEntities);

        //保存奖项信息
        if (CollectionUtils.isEmpty(detailsDtos)) {
            return;
        }
        List<RafflePrizeDetails> rafflePrizeDetails = (List<RafflePrizeDetails>) this.nebulaToolkitService.copyCollectionByWhiteList(detailsDtos, RafflePrizeDetailsDto.class, RafflePrizeDetails.class, HashSet.class, LinkedList.class);
        this.rafflePrizeDetailsService.saveBatch(rafflePrizeDetails);
    }

    /**
     * 构建抽奖奖池
     *
     * @param rafflePrize
     */
    public void buildPrizePoll(RafflePrizeDto rafflePrize) {

        List<RafflePrizeDetailsDto> details = rafflePrize.getDetails();
        if (CollectionUtils.isEmpty(details)) {
            return;
        }

        List<Integer> list = new ArrayList<>();

        //算小数最长长度
        Integer maxLength = 0;
        for (RafflePrizeDetailsDto prize : details) {
            BigDecimal probability = prize.getPrizeProb().divide(new BigDecimal(100));
            String doubleVa = probability.doubleValue() + "";
            if (doubleVa.contains(".")) {
                Integer length = doubleVa.length() - (doubleVa.indexOf(".") + 1);
                if (length > maxLength) {
                    maxLength = length;
                }
            }
        }

        Double pow = Math.pow(10, maxLength);
        Long aLong = new Long(pow.longValue());
        for (RafflePrizeDetailsDto prize : details) {
            BigDecimal probability = prize.getPrizeProb();//概率
            Integer prizeLevel = Integer.valueOf(prize.getPrizeLevel());
            long longValue = probability.multiply(new BigDecimal(aLong)).longValue();//奖项奖品数量
            List<Integer> integerList = Stream.generate(() -> prizeLevel).limit(longValue)
                    .collect(Collectors.toList());
            list.addAll(integerList);
        }
        //随机打乱奖池
        Collections.shuffle(list);
        if (CollUtil.isEmpty(list)) {
            return;
        }
        String prizePollStr = JSON.toJSONString(list);
        rafflePrize.setPrizePool(prizePollStr);
    }

    private void buildPrizeDetials(String tenantCode, List<RafflePrizeDetailsDto> detailsDtos, RafflePrizeDto rafflePrize, List<RafflePrizeDetailsDto> details) {
        if (CollectionUtils.isEmpty(details)) {
            return;
        }
        for (RafflePrizeDetailsDto detail : details) {
            detail.setId(null);
            detail.setRaffleCode(rafflePrize.getRaffleCode());
            detail.setRafflePrizeCode(rafflePrize.getCode());
            detail.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            detail.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            detail.setTenantCode(tenantCode);
            //新增时，投放份数= 投放库存份数（减去已中奖份数）
            detail.setPutInventoryCopies(detail.getPutCopies());
            detail.setPutInventoryNum(detail.getPutNum());
            detailsDtos.add(detail);
        }
        rafflePrize.setDetails(details);
    }

    @Override
    public void deleteBatch(List<String> codeList) {
        if (CollectionUtils.isEmpty(codeList)) {
            return;
        }

        List<RafflePrizeEntity> entities = this.rafflePrizeRepository.lambdaQuery().in(RafflePrizeEntity::getRaffleCode, codeList).list();
        if (CollectionUtils.isEmpty(entities)) {
            return;
        }
        List<String> prizeCodeList = entities.stream().map(RafflePrizeEntity::getCode).collect(Collectors.toList());
        this.rafflePrizeRepository.lambdaUpdate().set(RafflePrizeEntity::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
                .in(RafflePrizeEntity::getRaffleCode, codeList).update();

        this.rafflePrizeDetailsService.deleteBatch(prizeCodeList);
    }

    @Override
    @Transactional
    public void updateBatch(List<RafflePrizeDto> rafflePrizes, String code) {

        //逻辑删除奖品配置
//        this.rafflePrizeRepository.lambdaUpdate().set(RafflePrizeEntity::getDelFlag,DelFlagStatusEnum.DELETE.getCode())
//                        .in(RafflePrizeEntity::getRaffleCode,code).update();
        if (CollectionUtils.isEmpty(rafflePrizes)) {
            return;
        }

        //过滤奖项信息（取新增的）
        for (RafflePrizeDto rafflePrize : rafflePrizes) {
            //填充奖项信息
            List<RafflePrizeDetailsDto> details = rafflePrize.getDetails();
            if (CollectionUtils.isEmpty(details)) {
                continue;
            }
            //过滤新增的奖项信息（编辑活动时，不能编辑奖项信息）
            List<RafflePrizeDetailsDto> filter = details.stream().filter(e -> StringUtils.isBlank(e.getId())).collect(Collectors.toList());
            rafflePrize.setDetails(filter);
            //重新构建奖池
            this.buildPrizePoll(rafflePrize);
        }
        //保存
        this.saveBatch(rafflePrizes);


    }

    /**
     * @param detailId 奖项id
     * @param number   调整份数
     */
    @Override
    public void adjust(String detailId, BigDecimal number) {

        if (StringUtils.isBlank(detailId) || Objects.isNull(number)) {
            return;
        }
        RafflePrizeDetails rafflePrizeDetails = this.rafflePrizeDetailsService.detail(detailId);
        Validate.notNull(rafflePrizeDetails, "奖项不存在！");
        //TODO:加奖品锁
        String redisKey = String.format(PrizeRedisKeys.TENANT_PRIZE_LOCK_PREFIX, TenantUtils.getTenantCode() + rafflePrizeDetails.getPrizeCode());
        boolean locked = false;
        try {
            //查询配奖品的库存
            RafflePrizeDetails detail = this.rafflePrizeDetailsService.detail(detailId);
            Validate.notNull(detail, "奖项不存在！");
            PrizeEntity prizeEntity = prizeService.findByCode(detail.getPrizeCode());
            Validate.notNull(prizeEntity, "奖品不存在");

            //奖品库存
            BigDecimal prizeInventoryNum = prizeEntity.getInventoryNum();
            //奖项库存
            Validate.notNull(detail.getQuantity(), "奖项配置数量不能为空");
            BigDecimal raffleInventoryNum = detail.getQuantity().multiply(number);

            if (BigDecimal.ZERO.compareTo(number) > 0) {
                //减少，负数
                Validate.isTrue(raffleInventoryNum.negate().compareTo(number) < 0, "调整数量必须小于大转盘库存数量");
            } else {
                //增加，正数
                Validate.isTrue(prizeInventoryNum.compareTo(number) > 0, "调整数量必须小于大奖品库存数量");
            }

            //修改投放份数，投放数量，投放库存份数，投放库存数量
            detail.setPutCopies(detail.getPutCopies().add(number));
            detail.setPutNum(detail.getPutNum().add(raffleInventoryNum));
            detail.setPutInventoryCopies(detail.getPutInventoryCopies().add(number));
            detail.setPutInventoryNum(detail.getPutInventoryNum().add(raffleInventoryNum));
            this.rafflePrizeDetailsService.save(detail);
            //修改奖品库存
            prizeEntity.setInventoryNum(prizeEntity.getInventoryNum().add(raffleInventoryNum.negate()));
            this.prizeService.save(prizeEntity);

        } finally {
            if (locked) {
                redisMutexService.unlock(redisKey);
            }
        }
    }

    /**
     * 删除奖项 -> 返还库存
     *
     * @param id
     */
    @Override
    public void delete(String id) {

        //TODO:枷锁（关于库存改动都要枷锁）

        if (StringUtils.isBlank(id)) {
            return;
        }
        RafflePrizeDetails detail = this.rafflePrizeDetailsService.detail(id);
        Validate.notNull(detail, "奖项不存在");
        this.rafflePrizeDetailsService.delete(id);

        List<PrizeEntity> prizeEntities = prizeService.findByCodes(Lists.newArrayList(detail.getPrizeCode()));
        if (CollectionUtils.isEmpty(prizeEntities)) {
            throw new RuntimeException("奖品不存在");
        }
        String prizeId = prizeEntities.get(0).getId();
        //修改奖品库存,释放库存
        this.prizeService.adjust(prizeId, detail.getPutInventoryNum());
    }

    @Override
    public RafflePrizeVo findByCode(String rafflePrizeCode, String tenantCode) {
        if (StringUtils.isBlank(rafflePrizeCode)) {
            return null;
        }
        RafflePrizeEntity one = this.rafflePrizeRepository.lambdaQuery().eq(RafflePrizeEntity::getCode, rafflePrizeCode)
                .eq(RafflePrizeEntity::getTenantCode, TenantUtils.getTenantCode())
                .eq(RafflePrizeEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(RafflePrizeEntity::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
                .one();
        return this.nebulaToolkitService.copyObjectByWhiteList(one, RafflePrizeVo.class, HashSet.class, LinkedList.class);
    }

    @Override
    public RafflePrizeVo findByCodeAndLevel(String rafflePrizeCode, String tenantCode, String level) {
        return this.rafflePrizeRepository.findByCodeAndLevel(rafflePrizeCode, tenantCode, level);
    }


}
