package com.biz.crm.member.business.member.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.org.sdk.service.OrgVoService;
import com.biz.crm.mdm.business.org.sdk.vo.OrgVo;
import com.biz.crm.member.business.member.local.entity.MemberInfoArchives;
import com.biz.crm.member.business.member.local.entity.RealOrder;
import com.biz.crm.member.business.member.local.entity.VerificationUser;
import com.biz.crm.member.business.member.local.helper.UserSearchHelper;
import com.biz.crm.member.business.member.local.repository.MemberInfoArchivesRepository;
import com.biz.crm.member.business.member.local.repository.RealOrderRepository;
import com.biz.crm.member.business.member.local.repository.VerificationUserRepository;
import com.biz.crm.member.business.member.local.service.MemberInfoArchivesService;
import com.biz.crm.member.business.member.local.service.MemberInfoService;
import com.biz.crm.member.business.member.local.service.RealOrderService;
import com.biz.crm.member.business.member.sdk.constants.ProcessConstant;
import com.biz.crm.member.business.member.sdk.dto.AppOrderPaginationDto;
import com.biz.crm.member.business.member.sdk.dto.DispatchSalesmanDto;
import com.biz.crm.member.business.member.sdk.dto.RealAndDummyOrderPaginationDto;
import com.biz.crm.member.business.member.sdk.dto.RealOrderDto;
import com.biz.crm.member.business.member.sdk.dto.RealOrderPaginationDto;
import com.biz.crm.member.business.member.sdk.dto.VerificationRealOrderDto;
import com.biz.crm.member.business.member.sdk.enums.AppRealOrderStatusEnum;
import com.biz.crm.member.business.member.sdk.enums.CpsYesOrNoEnum;
import com.biz.crm.member.business.member.sdk.enums.DummyOrderStatusEnum;
import com.biz.crm.member.business.member.sdk.enums.OrderSettleStatusEnum;
import com.biz.crm.member.business.member.sdk.enums.RealOrderStatusEnum;
import com.biz.crm.member.business.member.sdk.enums.RealOrderTypeEnum;
import com.biz.crm.member.business.member.sdk.enums.VerificationOrderStatusEnum;
import com.biz.crm.member.business.member.sdk.enums.VerificationUserTypeEnum;
import com.biz.crm.member.business.member.sdk.vo.MemberInfoArchivesVo;
import com.biz.crm.member.business.member.sdk.vo.MemberInfoVo;
import com.biz.crm.member.business.member.sdk.vo.RealAndDummyOrderVo;
import com.biz.crm.member.business.member.sdk.vo.RealOrderDistributeVo;
import com.biz.crm.member.business.member.sdk.vo.RealOrderVo;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.biz.crm.workflow.sdk.service.ProcessBusinessMappingService;
import com.biz.crm.workflow.sdk.service.ProcessBusinessService;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.JsonUtils;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 实物订单(RealOrder)表服务实现类
 *
 * @author zouhs
 * @date 2023-06-14 14:53:17
 */
@Slf4j
@Service("realOrderService")
public class RealOrderServiceImpl implements RealOrderService {

  @Autowired(required = false)
  private RealOrderRepository realOrderRepository;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired
  private UserSearchHelper userSearchHelper;

  @Autowired(required = false)
  private OrgVoService orgVoService;

  @Autowired
  private MemberInfoArchivesService memberInfoArchivesService;

  @Autowired
  private MemberInfoService memberInfoService;

  @Autowired
  private MemberInfoArchivesRepository memberInfoArchivesRepository;

  @Autowired
  private VerificationUserRepository verificationUserRepository;

  @Autowired(required = false)
  private CustomerVoService customerVoService;

  @Override
  public Page<RealOrderVo> findByConditions(Pageable pageable, RealOrderPaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new RealOrderPaginationDto());
    Page<RealOrderVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.realOrderRepository.findByConditions(page, dto);
  }

  @Override
  public RealOrderVo findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    RealOrder realOrder = this.realOrderRepository.findById(id);
    RealOrderVo realOrderVo = this.nebulaToolkitService
        .copyObjectByWhiteList(realOrder, RealOrderVo.class, HashSet.class, LinkedList.class);
    return realOrderVo;
  }

  @Override
  @Transactional
  public RealOrderVo create(RealOrderVo realOrderVo) {
    this.createValidation(realOrderVo);
    RealOrder realOrder = this.nebulaToolkitService
        .copyObjectByWhiteList(realOrderVo, RealOrder.class, HashSet.class, LinkedList.class);
    realOrder.setTenantCode(TenantUtils.getTenantCode());
    realOrder.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    realOrder.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    MemberInfoVo memberInfoVo = memberInfoService
        .findDetailByMemberCode(realOrder.getUserCode());
    Validate.notNull(memberInfoVo,"会员信息异常");
    MemberInfoArchivesVo memberInfoArchivesVo = memberInfoArchivesService
        .findDetailByMemberCode(realOrder.getUserCode());
    Validate.notNull(memberInfoArchivesVo,"会员档案信息异常");
    // 查询会员组织是否有配送权限 null和否 = 总部，是=市场配送订单
    List<OrgVo> orgVos = orgVoService.findByOrgCodes(Arrays.asList(memberInfoArchivesVo.getOrgCode()));
    if(CollectionUtils.isNotEmpty(orgVos)){
      OrgVo orgVo = orgVos.get(0);
      if(orgVo!=null && Objects.equals("1",orgVo.getAuthority())){
        realOrder.setOrderType(RealOrderTypeEnum.MARKET_ORDER.getValue());
        realOrder.setOrderStatus(RealOrderStatusEnum.STAY_DELIVERY.getValue());
        realOrder.setOrgCode(orgVo.getOrgCode());
        realOrder.setOrgName(orgVo.getOrgName());
      }else{
        realOrder.setOrderType(RealOrderTypeEnum.TOTAL_ORDER.getValue());
        realOrder.setOrderStatus(RealOrderStatusEnum.STAY_SHIP.getValue());
        if(orgVo!=null){
          realOrder.setOrgCode(orgVo.getOrgCode());
          realOrder.setOrgName(orgVo.getOrgName());
        }
      }
    }
    realOrder.setOrderCode(generateCodeService.generateCode("SW",5).get(0));
    realOrder.setUserName(memberInfoVo.getMemberName());
    realOrder.setOrderTime(new Date());
    realOrder.setOrderName(memberInfoVo.getMemberName());
    realOrder.setOrderPhone(memberInfoVo.getMemberPhone());
    this.realOrderRepository.saveOrUpdate(realOrder);
    realOrderVo.setOrderCode(realOrder.getOrderCode());
    realOrderVo.setId(realOrder.getId());
    return realOrderVo;
  }

  @Override
  @Transactional
  public RealOrderVo update(RealOrderVo realOrderVo) {
    this.updateValidation(realOrderVo);
    String currentId = realOrderVo.getId();
    RealOrder current = realOrderRepository.findById(currentId);
    current = Validate.notNull(current, "修改信息不存在");
    current = this.nebulaToolkitService
        .copyObjectByWhiteList(realOrderVo, RealOrder.class, HashSet.class, LinkedList.class);
    this.realOrderRepository.saveOrUpdate(current);
    // 发送修改通知
    // TODO 发送通知
    return realOrderVo;
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.realOrderRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.realOrderRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
  }

  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.realOrderRepository.updateDelFlagByIds(ids);
  }

  @Override
  public void changeTotalOrder(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.realOrderRepository.changeTotalOrder(ids);
  }

  @Override
  public void changeMarketOrder(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.realOrderRepository.changeMarketOrder(ids);
  }

  @Override
  public void changeDistribute(RealOrderDistributeVo realOrderDistributeVo) {
    Validate.notBlank(realOrderDistributeVo.getId(), "请选择订单");
    RealOrder realOrder = this.realOrderRepository.findById(realOrderDistributeVo.getId());
    Validate.notNull(realOrder, "实物订单不存在!");
    // 根据角色判断业务员和办事处负责人
    if (Objects.equals("办事处负责人", realOrderDistributeVo.getRoleName())) {
      realOrder.setOfficeName(realOrderDistributeVo.getSalesmanName());
      realOrder.setOfficePhone(realOrderDistributeVo.getSalesmanPhone());
    } else {
      realOrder.setSalesmanName(realOrderDistributeVo.getSalesmanName());
      realOrder.setSalesmanPhone(realOrderDistributeVo.getSalesmanPhone());
    }
    realOrder.setFlagDistribute(CpsYesOrNoEnum.YES.getValue());
    this.realOrderRepository.updateById(realOrder);
  }

  @Override
  public void shipOrderById(RealOrderVo realOrderVo) {
    Validate.notBlank(realOrderVo.getId(), "请选择订单");
    Validate.notBlank(realOrderVo.getExpressNum(), "快递单号不能为空");
    RealOrder realOrder = this.realOrderRepository.findById(realOrderVo.getId());
    Validate.notNull(realOrder, "实物订单不存在!");
    Validate.isTrue(
        Objects.equals(RealOrderStatusEnum.STAY_SHIP.getValue(), realOrder.getOrderStatus()),
        "待发货状态才允许发货,请检查");
    realOrder.setExpressNum(realOrder.getExpressNum());
    realOrder.setOrderStatus(RealOrderStatusEnum.HAS_SHIP.getValue());
    this.realOrderRepository.updateById(realOrder);
  }

  @Override
  public void cancelShipOrderById(RealOrderVo realOrderVo) {
    Validate.notBlank(realOrderVo.getId(), "请选择订单");
    RealOrder realOrder = this.realOrderRepository.findById(realOrderVo.getId());
    Validate.notNull(realOrder, "实物订单不存在!");
    Validate
        .isTrue(Objects.equals(RealOrderStatusEnum.HAS_SHIP.getValue(), realOrder.getOrderStatus()),
            "待发货状态才允许发货,请检查");
    realOrder.setExpressNum(null);
    realOrder.setOrderStatus(RealOrderStatusEnum.STAY_SHIP.getValue());
    this.realOrderRepository.updateById(realOrder);
  }

  @Override
  public void realOrderImportExpressNum(RealOrderVo realOrderVo) {
    Validate.notBlank(realOrderVo.getOrderCode(), "订单号不能为空");
    Validate.notBlank(realOrderVo.getExpressNum(), "快递单号不能为空");
    RealOrder realOrder = this.realOrderRepository.lambdaQuery()
        .eq(RealOrder::getOrderCode, realOrderVo.getOrderCode()).one();
    if (realOrder != null) {
      if (Objects.equals(RealOrderStatusEnum.STAY_SHIP.getValue(), realOrder.getOrderStatus())) {
        realOrder.setExpressNum(realOrder.getExpressNum());
        realOrder.setOrderStatus(RealOrderStatusEnum.HAS_SHIP.getValue());
        this.realOrderRepository.updateById(realOrder);
      }
    }
  }

  @Override
  @Transactional
  public void settleSubmit(RealOrderVo realOrderVo) {
    Validate.notBlank(realOrderVo.getId(), "请选择订单");
    RealOrder realOrder = this.realOrderRepository.findById(realOrderVo.getId());
    Validate.notNull(realOrder, "实物订单不存在!");
    Validate
        .isTrue(Objects
                .equals(OrderSettleStatusEnum.STAY_SETTLE.getValue(), realOrder.getSettleStatus()),
            "请选择未结算订单!");
    realOrder.setProcessNumber(this.commitProcess(realOrder));
    realOrder.setProcessStatus(ProcessStatusEnum.COMMIT.getDictCode());
    realOrder.setSettleStatus(OrderSettleStatusEnum.LOAD_SETTLE.getValue());
    this.realOrderRepository.updateById(realOrder);
  }

  /**
   * 小程序个人中心分页获取实物订单
   *
   * @param pageable
   * @param dto
   * @return
   */
  @Override
  public Page<RealOrderVo> appPageRealOrder(Pageable pageable, AppOrderPaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new AppOrderPaginationDto());
    this.appPageValidate(dto);
    String orderStatus = dto.getOrderStatus();
    String orderStatusAll = AppRealOrderStatusEnum.ALL.getValue();
    List<String> ordersStatusList = Lists.newArrayList();
    if (orderStatusAll.equals(orderStatus)) {
      ordersStatusList = null;
    }else if(Objects.equals(RealOrderStatusEnum.STAY_SHIP.getValue(),dto.getOrderStatus())){
      ordersStatusList.add(RealOrderStatusEnum.STAY_SHIP.getValue());
      ordersStatusList.add(RealOrderStatusEnum.STAY_DELIVERY.getValue());
    }else if(Objects.equals(RealOrderStatusEnum.HAS_SHIP.getValue(),dto.getOrderStatus())){
      ordersStatusList.add(RealOrderStatusEnum.HAS_SHIP.getValue());
      ordersStatusList.add(RealOrderStatusEnum.IN_DELIVERY.getValue());
    }else {
      ordersStatusList.add(orderStatus);
    }
    dto.setOrderStatusList(ordersStatusList);
    Page<RealOrderVo> page = new Page<>(pageable.getPageNumber(),
        pageable.getPageSize());
    String memberCode = userSearchHelper.getMemberLogin().getMemberCode();
    Page<RealOrderVo> result = this.realOrderRepository.appPageRealOrder(page, dto, memberCode);
    return result;
  }

  @Override
  public Page<RealAndDummyOrderVo> findByConditionsRealAndOrder(Pageable pageable,
      RealAndDummyOrderPaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new RealAndDummyOrderPaginationDto());
    Page<RealAndDummyOrderVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<RealAndDummyOrderVo> byConditionsRealAndOrder = this.realOrderRepository
        .findByConditionsRealAndOrder(page, dto);
    if (byConditionsRealAndOrder != null && CollectionUtils
        .isNotEmpty(byConditionsRealAndOrder.getRecords())) {
      for (RealAndDummyOrderVo record : byConditionsRealAndOrder.getRecords()) {
        if (Objects.equals("0", record.getOrderType())) {
          // 实物订单
          record.setOrderType("实物订单");
          record.setOrderStatus(RealOrderStatusEnum.getNameByCode(record.getOrderStatus()));
        } else {
          // 虚拟订单
          record.setOrderType("虚拟订单");
          record.setOrderStatus(DummyOrderStatusEnum.getNameByCode(record.getOrderStatus()));
        }
      }
    }
    return byConditionsRealAndOrder;
  }

  @Override
  public RealOrderVo findDetailByOrderCode(String orderCode) {
    if (StringUtils.isBlank(orderCode)) {
      return null;
    }
    RealOrder realOrder = this.realOrderRepository.lambdaQuery()
        .eq(RealOrder::getOrderCode, orderCode).one();
    if (realOrder == null) {
      return null;
    }
    RealOrderVo realOrderVo = this.nebulaToolkitService
        .copyObjectByWhiteList(realOrder, RealOrderVo.class, HashSet.class, LinkedList.class);
    return realOrderVo;
  }

  /**
   * 小程序分页获取业务员为当前登录用户的订单信息，根据订单状态获取
   * @return
   */
  @Override
  public Page<RealOrderVo> appPageDeliveryOrder(Pageable pageable, String orderStatus) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    Validate.notBlank(orderStatus, "订单状态不能为空");
    String allValue = VerificationOrderStatusEnum.ALL.getValue();
    if (allValue.equals(orderStatus)){
      orderStatus = StringUtils.EMPTY;
    }
    Page<RealOrderVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    String memberPhone = this.userSearchHelper.getMemberLogin().getMemberPhone();
    VerificationUser verificationUser = verificationUserRepository.lambdaQuery()
        .eq(VerificationUser::getPhone, memberPhone)
        .eq(VerificationUser::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(VerificationUser::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .eq(VerificationUser::getUserType, VerificationUserTypeEnum.STUFF.getCode())
        .one();
    Validate.notNull(verificationUser, "当前登录用户不是实物核销人员或已被禁用，无权获取订单信息");
    String userCode = verificationUser.getUserCode();
    Page<RealOrderVo> result = this.realOrderRepository.appPageRealOrderBySalesmanCode(page, orderStatus, userCode);
    return result;
  }

  /**
   * 小程序分页获取当前登录用户分发订单
   * @param pageable
   * @param orderStatus
   * @return
   */
  @Override
  public Page<RealOrderVo> appPageDispatchOrder(Pageable pageable, String orderStatus) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    Validate.notBlank(orderStatus, "订单状态不能为空");
    String allValue = VerificationOrderStatusEnum.ALL.getValue();
    if (allValue.equals(orderStatus)){
      orderStatus = StringUtils.EMPTY;
    }
    String memberCode = this.userSearchHelper.getMemberLogin().getMemberCode();
    MemberInfoArchives infoArchives = this.memberInfoArchivesRepository.lambdaQuery()
        .eq(MemberInfoArchives::getMemberCode, memberCode)
        .eq(MemberInfoArchives::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .one();
    Validate.notNull(infoArchives, "当前用户没有档案信息，无法获取到分发订单数据");
    String orgCode = infoArchives.getOrgCode();
    Validate.notBlank(orgCode, "当前用户的组织信息不能为空");
    Page<RealOrderVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<RealOrderVo> result = this.realOrderRepository.appPageDispatchOrder(page, orderStatus, orgCode);
    return result;
  }

  /**
   * 小程序为指定实物订单分发业务员
   * @param dto
   */
  @Override
  public void appDispatchSalesmanForOrder(DispatchSalesmanDto dto) {
    this.validationDispatchSalesman(dto);
    String orderCode = dto.getOrderCode();
    RealOrder realOrder = this.realOrderRepository.lambdaQuery()
        .eq(RealOrder::getOrderCode, orderCode)
        .eq(RealOrder::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .eq(RealOrder::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .one();
    Validate.notNull(realOrder, "分发的订单不存在");
    String orderStatus = realOrder.getOrderStatus();
    String stayDeliveryValue = RealOrderStatusEnum.STAY_DELIVERY.getValue();
    Validate.isTrue(stayDeliveryValue.equals(orderStatus), "订单状态不是待配送，不能分发");
    this.realOrderRepository.lambdaUpdate()
        .set(RealOrder::getSalesmanCode, dto.getSalesmanCode())
        .set(RealOrder::getSalesmanName, dto.getSalesmanName())
        .set(RealOrder::getSalesmanPhone, dto.getSalesmanPhone())
        .eq(RealOrder::getOrderCode, dto.getOrderCode())
        .update();
  }

  /**
   * 小程序核销实物订单
   * @param dto
   */
  @Override
  public void appWriteOrderById(VerificationRealOrderDto dto) {
    String realOrderId = dto.getVerificationOrderId();
    RealOrder realOrder = this.realOrderRepository.lambdaQuery()
        .eq(RealOrder::getId, realOrderId)
        .eq(RealOrder::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(RealOrder::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .one();
    Validate.notNull(realOrder, "核销订单不存在");
    String orderStatus = realOrder.getOrderStatus();
    String inDeliveryValue = RealOrderStatusEnum.IN_DELIVERY.getValue();
    Validate.isTrue(inDeliveryValue.equals(orderStatus), "核销失败,订单状态不是配送中");
    String memberPhone = this.userSearchHelper.getMemberLogin().getMemberPhone();
    VerificationUser verificationUser = this.verificationUserRepository.lambdaQuery()
        .eq(VerificationUser::getPhone, memberPhone)
        .eq(VerificationUser::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(VerificationUser::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .eq(VerificationUser::getUserType, VerificationUserTypeEnum.STUFF.getCode())
        .one();
    Validate.notNull(verificationUser, "当前登录用户不是实物核销人员或已被禁用,无权核销该订单");
    String userCode = verificationUser.getUserCode();
    String userName = verificationUser.getUserName();
    String relationAgentCode = dto.getRelationAgentCode();
    String relationAgentName = dto.getRelationAgentName();
    this.realOrderRepository.lambdaUpdate()
        .set(RealOrder::getWriteOffCode, userCode)
        .set(RealOrder::getWriteOffName, userName)
        .set(RealOrder::getWriteOffTime, new Date(System.currentTimeMillis()))
        .set(RealOrder::getOrderStatus, RealOrderStatusEnum.HAS_WRITE_OFF.getValue())
        .set(StringUtils.isNotBlank(relationAgentCode), RealOrder::getRelationAgentCode, relationAgentCode)
        .set(StringUtils.isNotBlank(relationAgentName), RealOrder::getRelationAgentName, relationAgentName)
        .eq(RealOrder::getId, realOrderId);
  }

  /**
   * 小程序配送订单
   * @param orderId
   */
  @Override
  public void appDeliveryOrderById(String orderId) {
    Validate.notBlank(orderId, "订单Id不能为空");
    RealOrder realOrder = this.realOrderRepository.getById(orderId);
    Validate.notNull(realOrder, "此订单不存在");
    String orderStatus = realOrder.getOrderStatus();
    String stayDeliveryValue = RealOrderStatusEnum.STAY_DELIVERY.getValue();
    Validate.isTrue(stayDeliveryValue.equals(orderStatus), "订单的状态不是待配送,去配送失败");
    this.realOrderRepository.lambdaUpdate()
        .set(RealOrder::getOrderStatus, RealOrderStatusEnum.IN_DELIVERY)
        .eq(RealOrder::getId, orderId)
        .eq(RealOrder::getOrderType, RealOrderTypeEnum.MARKET_ORDER.getValue())
        .update();
  }

  /**
   * 小程序根据组织获取经销商
   * @return
   */
  @Override
  public List<CustomerVo> appListAgentCustomer() {
    String memberCode = this.userSearchHelper.getMemberLogin().getMemberCode();
    MemberInfoArchives infoArchives = this.memberInfoArchivesRepository.lambdaQuery()
        .eq(MemberInfoArchives::getMemberCode, memberCode)
        .eq(MemberInfoArchives::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(MemberInfoArchives::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .one();
    Validate.notNull(infoArchives, "当前登录用户档案信息不存在");
    String orgCode = infoArchives.getOrgCode();
    Validate.notBlank(orgCode, "当前登录用户的组织信息为空");
    List<CustomerVo> result = customerVoService.findByOrgCodes(Arrays.asList(orgCode));
    return result;
  }

  /**
   * 校验分发业务员信息
   * @param dto
   */
  private void validationDispatchSalesman(DispatchSalesmanDto dto) {
    Validate.notNull(dto, "业务员信息不能为空");
    Validate.notBlank(dto.getOrderCode(), "订单编码不能为空");
    Validate.notBlank(dto.getSalesmanCode(), "业务员编码不能为空");
    Validate.notBlank(dto.getSalesmanName(), "业务员姓名不能为空");
    Validate.notBlank(dto.getSalesmanPhone(), "业务员手机号不能为空");
  }

  /**
   * 小程序分页查询参数校验
   *
   * @param dto
   */
  private void appPageValidate(AppOrderPaginationDto dto) {
    Validate.notNull(dto, "查询参数不能为空");
    Validate.notBlank(dto.getOrderSource(), "数据来源不能为空");
    Validate.notBlank(dto.getOrderStatus(), "订单状态不能为空");
  }

  private void createValidation(RealOrderVo realOrderVo) {
    this.validation(realOrderVo);
  }

  private void updateValidation(RealOrderVo realOrderVo) {
    this.validation(realOrderVo);
  }

  private void validation(RealOrderVo realOrderVo) {
    Validate.notBlank(realOrderVo.getOrderSource(), "订单来源不能为空");
    Validate.notBlank(realOrderVo.getProductImg(),"商品图片不能为空");
    Validate.notBlank(realOrderVo.getUserCode(),"关联用户会员不能为空");
    Validate.notBlank(realOrderVo.getProductName(),"商品名称不能为空");
//    Validate.notBlank(realOrderVo.getProductCode(),"商品编码不能为空");
    Validate.notNull(realOrderVo.getProductNum(),"商品数量不能为空");
    Validate.notBlank(realOrderVo.getReceiptName(),"收货人不能为空");
    Validate.notBlank(realOrderVo.getReceiptPhone(),"收货人手机号不能为空");
    Validate.notBlank(realOrderVo.getReceiptAddress(),"收货地址不能为空");
  }

  @Autowired
  private ProcessBusinessService processBusinessService;

  @Autowired(required = false)
  private ProcessBusinessMappingService processBusinessMappingService;

  @Value("${crm.business.realorder.process-key:}")
  private String defaultProcessKey;

  private static final Set<String> STATUS_SET =
      Sets.newHashSet(
          ProcessStatusEnum.PREPARE.getDictCode(),
          ProcessStatusEnum.PASS.getDictCode(),
          ProcessStatusEnum.REJECT.getDictCode(),
          ProcessStatusEnum.RECOVER.getDictCode());

  /**
   * 经销商新增提交工作流进行审批，提交成功返回流程实例ID，提交失败则抛出异常
   *
   * @param dto 授信请求DTO
   */
  private String commitProcess(RealOrder dto) {
    ProcessBusinessDto processBusiness = Optional.ofNullable(dto.getProcessBusiness())
        .orElse(new ProcessBusinessDto());
    String processKey = processBusiness.getProcessKey();
    if (StringUtils.isBlank(processKey)) {
      processBusiness.setProcessKey(defaultProcessKey);
    }
    processBusiness.setBusinessNo(dto.getId());
    processBusiness.setBusinessFormJson(JsonUtils.obj2JsonString(dto));
    processBusiness.setBusinessCode(ProcessConstant.MMS_REAL_ORDER_CREATE);
    processBusiness.setProcessTitle(dto.getOrderName() + dto.getOrderCode());
    this.processBusinessMappingService.deleteByBusinessNoAndBusinessCodeAndProcessKey(
        dto.getId(),
        ProcessConstant.MMS_REAL_ORDER_CREATE,
        processBusiness.getProcessKey());
    ProcessBusinessVo processBusinessVo = processBusinessService.processStart(processBusiness);
    return processBusinessVo.getProcessNo();
  }

}
