package com.biz.crm.member.business.member.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.member.business.member.local.entity.SignInRule;
import com.biz.crm.member.business.member.local.repository.SignInRuleRepository;
import com.biz.crm.member.business.member.local.service.SignInRuleDetailService;
import com.biz.crm.member.business.member.local.service.SignInRuleService;
import com.biz.crm.member.business.member.sdk.dto.SignInRulePaginationDto;
import com.biz.crm.member.business.member.sdk.vo.SignInRuleDetailVo;
import com.biz.crm.member.business.member.sdk.vo.SignInRuleVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 签到规则(SignInRule)表服务实现类
 *
 * @author zouhs
 * @date 2023-06-15 16:47:20
 */
@Slf4j
@Service("signInRuleService")
public class SignInRuleServiceImpl implements SignInRuleService {
  
  @Autowired(required = false)
  private SignInRuleRepository signInRuleRepository;

  @Autowired
  private SignInRuleDetailService signInRuleDetailService;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public Page<SignInRuleVo> findByConditions(Pageable pageable, SignInRulePaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new SignInRulePaginationDto());
    Page<SignInRuleVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.signInRuleRepository.findByConditions(page, dto);
  }

  @Override
  public SignInRuleVo findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    SignInRule signInRule = this.signInRuleRepository.findById(id);
    SignInRuleVo signInRuleVo = this.nebulaToolkitService.copyObjectByWhiteList(signInRule, SignInRuleVo.class, HashSet.class, LinkedList.class);
    List<SignInRuleDetailVo> list = this.signInRuleDetailService.findListDetailById(signInRule.getId());
    signInRuleVo.setList(list);
    return signInRuleVo;
  }
  
  @Override
  @Transactional
  public SignInRuleVo create(SignInRuleVo signInRuleVo) {
    this.createValidation(signInRuleVo);
    SignInRule signInRule = this.nebulaToolkitService.copyObjectByWhiteList(signInRuleVo, SignInRule.class, HashSet.class, LinkedList.class);
    signInRule.setTenantCode(TenantUtils.getTenantCode());
    signInRule.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    signInRule.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    this.signInRuleRepository.saveOrUpdate(signInRule);
    // 处理明细
    if(CollectionUtils.isNotEmpty(signInRuleVo.getList())){
      int seq = 1;
      for (SignInRuleDetailVo signInRuleDetailVo : signInRuleVo.getList()) {
        signInRuleDetailVo.setId(null);
        signInRuleDetailVo.setSignInId(signInRule.getId());
        signInRuleDetailVo.setSeq(seq++);
      }
      this.signInRuleDetailService.batchSaveOrUpdate(signInRuleVo.getList());
    }
    return signInRuleVo;
  }

  @Override
  @Transactional
  public SignInRuleVo update(SignInRuleVo signInRuleVo) {
    this.updateValidation(signInRuleVo);
    String currentId = signInRuleVo.getId();
    if(StringUtils.isNotBlank(currentId)){
      SignInRule current = signInRuleRepository.findById(currentId);
      current = Validate.notNull(current, "修改信息不存在");
      current = this.nebulaToolkitService.copyObjectByWhiteList(signInRuleVo, SignInRule.class, HashSet.class, LinkedList.class);
      this.signInRuleRepository.saveOrUpdate(current);
      // 处理明细
      if(CollectionUtils.isNotEmpty(signInRuleVo.getList())){
        int seq = 1;
        for (SignInRuleDetailVo signInRuleDetailVo : signInRuleVo.getList()) {
          signInRuleDetailVo.setId(null);
          signInRuleDetailVo.setSignInId(current.getId());
          signInRuleDetailVo.setSeq(seq++);
        }
        this.signInRuleDetailService.batchSaveOrUpdate(signInRuleVo.getList());
      }
    }else{
      this.create(signInRuleVo);
    }
    return signInRuleVo;
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.signInRuleRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.signInRuleRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
  }
  
  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.signInRuleRepository.updateDelFlagByIds(ids);
  }

  @Override
  public SignInRuleVo findDetail() {
    List<SignInRule> list = this.signInRuleRepository.list();
    if(CollectionUtils.isNotEmpty(list)){
      SignInRuleVo signInRuleVo = this.nebulaToolkitService.copyObjectByWhiteList(list.get(0), SignInRuleVo.class, HashSet.class, LinkedList.class);
      List<SignInRuleDetailVo> signInRuleDetailVos = this.signInRuleDetailService.findListDetailById(signInRuleVo.getId());
      signInRuleVo.setList(signInRuleDetailVos);
      return signInRuleVo;
    }
    return new SignInRuleVo();
  }

  private void createValidation(SignInRuleVo signInRuleVo) {
    this.validation(signInRuleVo);
  }

  private void updateValidation(SignInRuleVo signInRuleVo) {
    this.validation(signInRuleVo);
  }

  private void validation(SignInRuleVo signInRuleVo) {
    Validate.notNull(signInRuleVo.getEveryIntegral(),"每次可获得x积分不能为空");
    Validate.notNull(signInRuleVo.getYearIntegralLimit(),"每次可获得x积分不能为空");
  }
}
