package com.biz.crm.member.business.member.local.service.internal;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.local.entity.UuidFlagOpEntity;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.member.business.member.local.entity.TastingConsultant;
import com.biz.crm.member.business.member.local.entity.TastingRelBenefits;
import com.biz.crm.member.business.member.local.entity.TastingRelCustomer;
import com.biz.crm.member.business.member.local.repository.TastingConsultantRepository;
import com.biz.crm.member.business.member.local.repository.TastingRelBenefitsRepository;
import com.biz.crm.member.business.member.local.repository.TastingRelCustomerRepository;
import com.biz.crm.member.business.member.local.service.MemberInfoBenefitsService;
import com.biz.crm.member.business.member.local.service.TastingConsultantService;
import com.biz.crm.member.business.member.sdk.dto.*;
import com.biz.crm.member.business.member.sdk.enums.MemberInfoBenefitsStatusEnum;
import com.biz.crm.member.business.member.sdk.enums.MemberInfoMemberBenefitsTypeEnum;
import com.biz.crm.member.business.member.sdk.vo.*;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.text.SimpleDateFormat;
import java.util.*;

/**
 * 品鉴顾问业务实现类
 *
 * @Description
 * @Author monroe
 * @Date 2023/6/13 10:57
 **/
@Slf4j
@Service
public class TastingConsultantServiceImpl implements TastingConsultantService {

    @Autowired
    private TastingConsultantRepository tastingConsultantRepository;
    @Autowired
    private TastingRelCustomerRepository tastingRelCustomerRepository;
    @Autowired
    private TastingRelBenefitsRepository tastingRelBenefitsRepository;
    @Autowired
    private NebulaToolkitService nebulaToolkitService;
    @Autowired
    private RedisMutexService redisMutexService;
    @Autowired
    private LoginUserService loginUserService;
    @Autowired
    private MemberInfoBenefitsService memberInfoBenefitsService;

    /**
     * 品鉴顾问编码redis key
     * 参数：租户
     */
    public static final String MMS_TASTING_CONSULTANT_CODE_REDIS_KEY = "mms:tasting:consultant:code:index:%s";

    /**
     * 品鉴顾问编码前缀
     */
    public static final String MMS_TASTING_CONSULTANT_CODE_PREFIX = "PJGW";

    @Override
    public Page<TastingConsultantPageVo> findByConditions(Pageable pageable, TastingConsultantPageDto tastingConsultantPageDto) {
        if (Objects.isNull(tastingConsultantPageDto)) {
            tastingConsultantPageDto = new TastingConsultantPageDto();
        }
        tastingConsultantPageDto.setTenantCode(TenantUtils.getTenantCode());
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
        Page<TastingConsultantPageVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return tastingConsultantRepository.findByConditions(page,tastingConsultantPageDto);
    }

    @Override
    public TastingConsultantVo findDetailById(String id) {
        if(StringUtils.isBlank(id)) {
            return null;
        }
        //基础信息
        TastingConsultant tastingConsultant = this.tastingConsultantRepository.getById(id);
        Validate.notNull(tastingConsultant,"品鉴顾问不存在！");
        TastingConsultantVo tastingConsultantVo = nebulaToolkitService.copyObjectByWhiteList(tastingConsultant, TastingConsultantVo.class, HashSet.class, LinkedList.class);
        //关联客户
        List<TastingRelCustomer> tastingRelCustomers = this.tastingRelCustomerRepository.lambdaQuery().eq(TastingRelCustomer::getTastingCode, tastingConsultantVo.getCode())
                .list();
        if(!CollectionUtils.isEmpty(tastingRelCustomers)) {
            List<TastingRelCustomerVo> tastingRelCustomerVos = (List<TastingRelCustomerVo>) nebulaToolkitService.copyCollectionByWhiteList(tastingRelCustomers, TastingRelCustomer.class, TastingRelCustomerVo.class, HashSet.class, LinkedList.class);
            tastingConsultantVo.setCustomers(tastingRelCustomerVos);
        }
        //关联权益
        List<TastingRelBenefits> tastingRelBenefits = this.tastingRelBenefitsRepository.lambdaQuery().eq(TastingRelBenefits::getTastingCode, tastingConsultantVo.getCode())
                .list();
        if(!CollectionUtils.isEmpty(tastingRelBenefits)) {
            List<TastingRelBenefitsVo> tastingRelBenefitsVos = (List<TastingRelBenefitsVo>) nebulaToolkitService.copyCollectionByWhiteList(tastingRelBenefits, TastingRelBenefits.class, TastingRelBenefitsVo.class, HashSet.class, LinkedList.class);
            tastingConsultantVo.setBenefits(tastingRelBenefitsVos);
        }
        return tastingConsultantVo;
    }

    @Transactional
    @Override
    public TastingConsultantVo create(TastingConsultantDto tastingConsultantDto) {
        this.createValidate(tastingConsultantDto);
        log.info("创建品鉴顾问入参:{}", JSON.toJSONString(tastingConsultantDto));
        TastingConsultant tastingConsultant = this.nebulaToolkitService.copyObjectByWhiteList(tastingConsultantDto, TastingConsultant.class, HashSet.class, LinkedList.class);
        this.tastingConsultantRepository.saveOrUpdate(tastingConsultant);
        //保存关联权益信息
        List<TastingRelBenefitsDto> benefits = tastingConsultantDto.getBenefits();
        if(!CollectionUtils.isEmpty(benefits)) {
            this.saveRelBenefits(tastingConsultant.getCode(), benefits);
            //TODO:发放权益(当新增品鉴顾问时，直接发放配置所有权益)
            this.createMemberBenefits(tastingConsultant.getCode(),benefits);
        }
        //保存关联客户信息
        List<TastingRelCustomerDto> customers = tastingConsultantDto.getCustomers();
        if (!CollectionUtils.isEmpty(customers)) {
            this.saveRelCustomer(tastingConsultant.getCode(), customers);
        }


        return this.nebulaToolkitService.copyObjectByWhiteList(tastingConsultant,TastingConsultantVo.class,HashSet.class,LinkedList.class);
    }

    /**
     * 发放会员权益
     * @param code
     * @param benefits
     */
    private void createMemberBenefits(String code, List<TastingRelBenefitsDto> benefits) {

        for (TastingRelBenefitsDto benefit : benefits) {
            MemberInfoBenefitsVo memberInfoBenefitsVo = new MemberInfoBenefitsVo();
            //品鉴顾问编码
            memberInfoBenefitsVo.setMemberCode(code);
            //权益编码
            memberInfoBenefitsVo.setBenefitsCode(benefit.getBenefitsCode());
            //权益名称
            memberInfoBenefitsVo.setBenefitsName(benefit.getBenefitsName());
            //权益类型
            memberInfoBenefitsVo.setBenefitsType(benefit.getBenefitsType());
            //兑换状态
            memberInfoBenefitsVo.setExchangeStatus(MemberInfoBenefitsStatusEnum.STAY_EXCHANGE.getValue());
            //会员权益类型
            memberInfoBenefitsVo.setMemberBenefitsType(MemberInfoMemberBenefitsTypeEnum.TASTING_BENEFITS.getValue());

            this.memberInfoBenefitsService.create(memberInfoBenefitsVo);

        }

    }

    /**
     * 创建参数校验
     * @param tastingConsultantDto
     */
    private void createValidate(TastingConsultantDto tastingConsultantDto) {
        Validate.notNull(tastingConsultantDto,"新增参数实体不能为空！");
        Validate.notBlank(tastingConsultantDto.getPhone(),"名字不能为空");
        Validate.notBlank(tastingConsultantDto.getCompany(),"单位不能为空");
        Validate.notBlank(tastingConsultantDto.getGender(),"性别不能为空");
        Validate.notNull(tastingConsultantDto.getBirthday(),"生日不能为空");
        Validate.notNull(tastingConsultantDto.getAge(),"年龄不能为空");
        Validate.notBlank(tastingConsultantDto.getPosition(),"职务不能为空");
        Validate.notNull(tastingConsultantDto.getValidateStartDate(),"生效期不能为空");
        Validate.notNull(tastingConsultantDto.getValidateEndDate(),"生效期不能为空");
        //生成编码
        tastingConsultantDto.setCode(this.generateCode(TenantUtils.getTenantCode()));

        //校验关联信息
        List<TastingRelCustomerDto> customers = tastingConsultantDto.getCustomers();
        if(!CollectionUtils.isEmpty(customers)) {
            for (TastingRelCustomerDto customer : customers) {
                Validate.notBlank(customer.getOrgCode(),"组织编码不能为空");
                Validate.notBlank(customer.getOrgName(),"组织名称不能为空");
            }
        }

        List<TastingRelBenefitsDto> benefits = tastingConsultantDto.getBenefits();
        if(!CollectionUtils.isEmpty(benefits)) {
            for (TastingRelBenefitsDto benefit : benefits) {
                Validate.notBlank(benefit.getBenefitsCode(),"权益编码不能为空");
                Validate.notBlank(benefit.getBenefitsName(),"权益名称不能为空");
                Validate.notBlank(benefit.getBenefitsType(),"权益类型编码不能为空");
//                Validate.notBlank(benefit.getBenefitsTypeName(),"权益类型名称不能为空");
                Validate.notNull(benefit.getQuantity(),"数量不能为空");
                Validate.notBlank(benefit.getDeliveryFrequency(),"发放频次不能为空");
//                Validate.notBlank(benefit.getDeliveryFrequencyName(),"发放频次名称不能为空");

            }

        }
        //初始化信息
        Date now = new Date();
        tastingConsultantDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        tastingConsultantDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        tastingConsultantDto.setTenantCode(TenantUtils.getTenantCode());
//        UserIdentity loginUser = loginUserService.getLoginUser();
//        tastingConsultantDto.setCreateAccount(loginUser.getAccount());
//        tastingConsultantDto.setCreateName(loginUserService.getLoginAccountName());
//        tastingConsultantDto.setModifyAccount(loginUser.getAccount());
//        tastingConsultantDto.setModifyName(loginUserService.getLoginAccountName());
//        tastingConsultantDto.setCreateTime(now);
//        tastingConsultantDto.setModifyTime(now);
    }

    /**
     * 生成编码
     */
    private String generateCode(String tenantCode) {
        String redisKey = String.format(MMS_TASTING_CONSULTANT_CODE_REDIS_KEY, tenantCode);
        String index = redisMutexService.getAndIncrement(redisKey, 1, 6);
        SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
        return StringUtils.join(MMS_TASTING_CONSULTANT_CODE_PREFIX, format.format(new Date()), index);
    }

    @Transactional
    @Override
    public TastingConsultantVo update(TastingConsultantDto tastingConsultantDto) {
        this.updateValidate(tastingConsultantDto);
        log.info("修改品鉴顾问入参:{}", JSON.toJSONString(tastingConsultantDto));
        TastingConsultant tastingConsultant = this.nebulaToolkitService.copyObjectByWhiteList(tastingConsultantDto, TastingConsultant.class, HashSet.class, LinkedList.class);
        this.tastingConsultantRepository.saveOrUpdate(tastingConsultant);

        //客户信息(先删除，在新增)
        QueryWrapper<TastingRelCustomer> customerQueryWrapper = new QueryWrapper<>();
        customerQueryWrapper.eq("tasting_code",tastingConsultant.getCode());
        customerQueryWrapper.eq("tenant_code",tastingConsultant.getTenantCode());
        this.tastingRelCustomerRepository.remove(customerQueryWrapper);
        List<TastingRelCustomerDto> customers = tastingConsultantDto.getCustomers();
        this.saveRelCustomer(tastingConsultant.getCode(), customers);

        //权益信息(先删除，在新增)
        QueryWrapper<TastingRelBenefits> benefitsQueryWrapper = new QueryWrapper<>();
        customerQueryWrapper.eq("tasting_code",tastingConsultant.getCode());
        customerQueryWrapper.eq("tenant_code",tastingConsultant.getTenantCode());
        this.tastingRelBenefitsRepository.remove(benefitsQueryWrapper);
        List<TastingRelBenefitsDto> benefits = tastingConsultantDto.getBenefits();
        this.saveRelBenefits(tastingConsultant.getCode(),benefits);
        return this.nebulaToolkitService.copyObjectByWhiteList(tastingConsultant,TastingConsultantVo.class,HashSet.class,LinkedList.class);
    }

    /**
     * 保存关联权益信息
     * @param code
     * @param benefitsDtos
     */
    private void saveRelBenefits(String code, List<TastingRelBenefitsDto> benefitsDtos) {
            List<TastingRelBenefits> tastingRelBenefits = (List<TastingRelBenefits>) this.nebulaToolkitService.copyCollectionByWhiteList(benefitsDtos, TastingRelBenefitsDto.class, TastingRelBenefits.class, HashSet.class, LinkedList.class);
            tastingRelBenefits.stream().forEach(e ->{
                e.setTastingCode(code);
                e.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                e.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                e.setTenantCode(TenantUtils.getTenantCode());
            });
            this.tastingRelBenefitsRepository.saveBatch(tastingRelBenefits);
    }

    /**
     * 保存关联客户信息
     * @param code
     * @param customerDtos
     */
    private void saveRelCustomer(String code, List<TastingRelCustomerDto> customerDtos) {
        List<TastingRelCustomer> tastingRelCustomers = (List<TastingRelCustomer>) this.nebulaToolkitService.copyCollectionByWhiteList(customerDtos, TastingRelCustomerDto.class, TastingRelCustomer.class, HashSet.class, LinkedList.class);
        tastingRelCustomers.stream().forEach(e ->{
            e.setTastingCode(code);
            e.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            e.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            e.setTenantCode(TenantUtils.getTenantCode());
        });
        this.tastingRelCustomerRepository.saveBatch(tastingRelCustomers);
    }

    /**
     * 修改参数校验
     * @param tastingConsultantDto
     */
    private void updateValidate(TastingConsultantDto tastingConsultantDto) {
        Validate.notNull(tastingConsultantDto,"修改参数实体不能为空！");
        Validate.notBlank(tastingConsultantDto.getPhone(),"名字不能为空");
        Validate.notBlank(tastingConsultantDto.getCompany(),"单位不能为空");
        Validate.notBlank(tastingConsultantDto.getGender(),"性别不能为空");
        Validate.notNull(tastingConsultantDto.getBirthday(),"生日不能为空");
        Validate.notNull(tastingConsultantDto.getAge(),"年龄不能为空");
        Validate.notBlank(tastingConsultantDto.getPosition(),"职务不能为空");
        Validate.notNull(tastingConsultantDto.getValidateStartDate(),"生效期不能为空");
        Validate.notNull(tastingConsultantDto.getValidateEndDate(),"生效期不能为空");
        Validate.notBlank(tastingConsultantDto.getCode(),"编码不能为空");
        Validate.notBlank(tastingConsultantDto.getId(),"修改信息时，主键id不能为空");

        //校验关联信息
        List<TastingRelCustomerDto> customers = tastingConsultantDto.getCustomers();
        if(!CollectionUtils.isEmpty(customers)) {
            for (TastingRelCustomerDto customer : customers) {
                Validate.notBlank(customer.getOrgCode(),"组织编码不能为空");
                Validate.notBlank(customer.getOrgName(),"组织名称不能为空");
            }
        }

        List<TastingRelBenefitsDto> benefits = tastingConsultantDto.getBenefits();
        if(!CollectionUtils.isEmpty(benefits)) {
            for (TastingRelBenefitsDto benefit : benefits) {
                Validate.notBlank(benefit.getBenefitsCode(),"权益编码不能为空");
                Validate.notBlank(benefit.getBenefitsName(),"权益名称不能为空");
                Validate.notBlank(benefit.getBenefitsType(),"权益类型编码不能为空");
//                Validate.notBlank(benefit.getBenefitsTypeName(),"权益类型名称不能为空");
                Validate.notNull(benefit.getQuantity(),"数量不能为空");
                Validate.notBlank(benefit.getDeliveryFrequency(),"发放频次不能为空");
//                Validate.notBlank(benefit.getDeliveryFrequencyName(),"发放频次名称不能为空");

            }

        }

        //初始化信息
        Date now = new Date();
        tastingConsultantDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        tastingConsultantDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        tastingConsultantDto.setTenantCode(TenantUtils.getTenantCode());
//        UserIdentity loginUser = loginUserService.getLoginUser();
//        tastingConsultantDto.setModifyAccount(loginUser.getAccount());
//        tastingConsultantDto.setModifyName(loginUserService.getLoginAccountName());
//        tastingConsultantDto.setModifyTime(now);

    }

    @Override
    public void enableBatch(List<String> ids) {
        if(CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.tastingConsultantRepository.lambdaUpdate().set(TastingConsultant::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
                .in(TastingConsultant::getId,ids).update();
    }

    @Override
    public void disableBatch(List<String> ids) {
        if(CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.tastingConsultantRepository.lambdaUpdate().set(TastingConsultant::getEnableStatus, EnableStatusEnum.DISABLE.getCode())
                .in(TastingConsultant::getId,ids).update();
    }

    @Override
    public void deleteBatch(List<String> ids) {

        if (CollectionUtils.isEmpty(ids)) {
            return;
        }

        this.tastingConsultantRepository.lambdaUpdate().set(TastingConsultant::getDelFlag,DelFlagStatusEnum.DELETE.getCode())
                .in(TastingConsultant::getId,ids).update();
    }

    @Override
    public TastingConsultantVo queryFlagTastingByPhone(String memberPhone) {
         Validate.notBlank(memberPhone,"查询品鉴顾问,手机号不能为空!");
        TastingConsultantVo tastingConsultantVo = this.tastingConsultantRepository.queryFlagTastingByPhone(memberPhone);
        return tastingConsultantVo;
    }
}
