package com.biz.crm.member.business.member.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.member.business.member.local.entity.VerificationRole;
import com.biz.crm.member.business.member.local.repository.VerificationRoleRepository;
import com.biz.crm.member.business.member.local.service.VerificationRoleService;
import com.biz.crm.member.business.member.sdk.dto.VerificationRoleDto;
import com.biz.crm.member.business.member.sdk.vo.VerificationRoleVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.text.SimpleDateFormat;
import java.util.*;

/**
 * 角色管理实现类
 *
 * @Description
 * @Author monroe
 * @Date 2023/6/26 16:07
 **/
@Service
public class VerificationRoleServiceImpl implements VerificationRoleService {

    @Autowired
    private VerificationRoleRepository roleRepository;
    @Autowired
    private NebulaToolkitService nebulaToolkitService;
    @Autowired
    private RedisMutexService redisMutexService;

    /**
     * 角色编码redis key
     * 参数：租户
     */
    public static final String MMS_VERIFICATION_ROLE_CODE_REDIS_KEY = "mms:role:code:index:%s";

    /**
     * 角色编码前缀
     */
    public static final String MMS_VERIFICATION_ROLE_CODE_PREFIX = "JS";

    /**
     * 生成编码
     */
    private String generateCode(String tenantCode) {
        String redisKey = String.format(MMS_VERIFICATION_ROLE_CODE_REDIS_KEY, tenantCode);
        String index = redisMutexService.getAndIncrement(redisKey, 1, 6);
        SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
        return StringUtils.join(MMS_VERIFICATION_ROLE_CODE_PREFIX, format.format(new Date()), index);
    }

    @Override
    public Page<VerificationRoleVo> findByConditions(Pageable pageable, VerificationRoleDto roleDto) {
        if (Objects.isNull(roleDto)) {
            roleDto = new VerificationRoleDto();
        }
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
        Page<VerificationRoleVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.roleRepository.findByConditions(page, roleDto);
    }

    @Override
    public VerificationRoleVo findDetailById(String id) {
        if(StringUtils.isBlank(id)) {
            return null;
        }
        VerificationRole roleEntity = this.roleRepository.getById(id);
        Validate.notNull(roleEntity,"角色不存在");
       return this.nebulaToolkitService.copyObjectByWhiteList(roleEntity,VerificationRoleVo.class, HashSet.class,LinkedList.class);
    }

    @Transactional
    @Override
    public VerificationRoleVo create(VerificationRoleDto roleDto) {
        Validate.notNull(roleDto,"新增实体不能为空");
        Validate.notBlank(roleDto.getRoleName(),"角色名称不能为空");
        roleDto.setRoleCode(this.generateCode(TenantUtils.getTenantCode()));
        roleDto.setTenantCode(TenantUtils.getTenantCode());
        roleDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        roleDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        VerificationRole roleEntity = this.nebulaToolkitService.copyObjectByWhiteList(roleDto, VerificationRole.class, HashSet.class, LinkedList.class);
        this.roleRepository.saveOrUpdate(roleEntity);
        return this.nebulaToolkitService.copyObjectByWhiteList(roleEntity, VerificationRoleVo.class, HashSet.class, LinkedList.class);
    }

    @Transactional
    @Override
    public VerificationRoleVo update(VerificationRoleDto roleDto) {
        Validate.notNull(roleDto,"编辑实体不能为空");
        Validate.notBlank(roleDto.getRoleName(),"角色名称不能为空");
        Validate.notBlank(roleDto.getId(),"编辑时,主键id不能为空");
        Validate.notBlank(roleDto.getRoleCode(),"编辑时,角色编码不能为空");
        roleDto.setTenantCode(TenantUtils.getTenantCode());
        roleDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        roleDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        VerificationRole roleEntity = this.nebulaToolkitService.copyObjectByWhiteList(roleDto, VerificationRole.class, HashSet.class, LinkedList.class);
        this.roleRepository.saveOrUpdate(roleEntity);
        return this.nebulaToolkitService.copyObjectByWhiteList(roleEntity, VerificationRoleVo.class, HashSet.class, LinkedList.class);
    }

    @Override
    public void enableBatch(List<String> ids) {
        if(CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.roleRepository.lambdaUpdate().set(VerificationRole::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
                .in(VerificationRole::getId,ids).update();

    }

    @Override
    public void disableBatch(List<String> ids) {
        if(CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.roleRepository.lambdaUpdate().set(VerificationRole::getEnableStatus, EnableStatusEnum.DISABLE.getCode())
                .in(VerificationRole::getId,ids).update();
    }

    @Override
    public void deleteBatch(List<String> ids) {
        if(CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.roleRepository.lambdaUpdate().set(VerificationRole::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
                .in(VerificationRole::getId,ids).update();
    }
}
