package com.biz.crm.member.business.member.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.member.business.member.local.entity.MemberInfoArchives;
import com.biz.crm.member.business.member.local.entity.VerificationUser;
import com.biz.crm.member.business.member.local.helper.UserSearchHelper;
import com.biz.crm.member.business.member.local.repository.MemberInfoArchivesRepository;
import com.biz.crm.member.business.member.local.repository.VerificationUserRepository;
import com.biz.crm.member.business.member.local.service.VerificationUserService;
import com.biz.crm.member.business.member.sdk.dto.VerificationUserDto;
import com.biz.crm.member.business.member.sdk.enums.VerificationUserTypeEnum;
import com.biz.crm.member.business.member.sdk.vo.VerificationUserVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.text.SimpleDateFormat;
import java.util.*;

/**
 * 核销人员实现类
 *
 * @Description
 * @Author monroe
 * @Date 2023/6/26 16:47
 **/
@Service
public class VerificationUserServiceImpl implements VerificationUserService {

    @Autowired
    private VerificationUserRepository verificationUserRepository;

    @Autowired
    private NebulaToolkitService nebulaToolkitService;
    @Autowired
    private RedisMutexService redisMutexService;

    @Autowired
    private UserSearchHelper userSearchHelper;

    @Autowired
    private MemberInfoArchivesRepository memberInfoArchivesRepository;

    /**
     * 核销人员编码redis key
     * 参数：租户
     */
    public static final String MMS_VERIFICATION_USER_CODE_REDIS_KEY = "mms:role:code:index:%s";

    /**
     * 核销人员编码前缀
     */
    public static final String MMS_VERIFICATION_USER_CODE_PREFIX = "HX";

    /**
     * 生成编码
     */
    private String generateCode(String tenantCode) {
        String redisKey = String.format(MMS_VERIFICATION_USER_CODE_REDIS_KEY, tenantCode);
        String index = redisMutexService.getAndIncrement(redisKey, 1, 6);
        SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
        return StringUtils.join(MMS_VERIFICATION_USER_CODE_PREFIX, format.format(new Date()), index);
    }

    @Override
    public Page<VerificationUserVo> findByConditions(Pageable pageable, VerificationUserDto verificationUserDto) {
        if (Objects.isNull(verificationUserDto)) {
            verificationUserDto = new VerificationUserDto();
        }
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
        Page<VerificationUserVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.verificationUserRepository.findByConditions(page, verificationUserDto);
    }

    @Override
    public VerificationUserVo findDetailById(String id) {

        if(StringUtils.isBlank(id)) {
            return null;
        }
        VerificationUser verificationUser = this.verificationUserRepository.getById(id);
        Validate.notNull(verificationUser,"人员不存在");
        return this.nebulaToolkitService.copyObjectByWhiteList(verificationUser,VerificationUserVo.class, HashSet.class,LinkedList.class);
    }

    @Transactional
    @Override
    public VerificationUserVo create(VerificationUserDto verificationUserDto) {
        Validate.notNull(verificationUserDto,"新增实体不能为空");
        Validate.notBlank(verificationUserDto.getRoleCode(),"角色不能为空");
        Validate.notBlank(verificationUserDto.getUserName(),"姓名不能为空");
        Validate.notBlank(verificationUserDto.getUserType(),"人员类型不能为空");
        verificationUserDto.setUserCode(this.generateCode(TenantUtils.getTenantCode()));
        verificationUserDto.setTenantCode(TenantUtils.getTenantCode());
        verificationUserDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        verificationUserDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        VerificationUser verificationUser = this.nebulaToolkitService.copyObjectByWhiteList(verificationUserDto, VerificationUser.class, HashSet.class, LinkedList.class);
        this.verificationUserRepository.saveOrUpdate(verificationUser);
        return this.nebulaToolkitService.copyObjectByWhiteList(verificationUser, VerificationUserVo.class, HashSet.class, LinkedList.class);
    }

    @Transactional
    @Override
    public VerificationUserVo update(VerificationUserDto verificationUserDto) {
        Validate.notNull(verificationUserDto,"编辑实体不能为空");
        Validate.notBlank(verificationUserDto.getRoleCode(),"角色不能为空");
        Validate.notBlank(verificationUserDto.getUserName(),"姓名不能为空");
        Validate.notBlank(verificationUserDto.getUserType(),"人员类型不能为空");
        Validate.notBlank(verificationUserDto.getId(),"编辑时,主键id不能为空");
        Validate.notBlank(verificationUserDto.getUserCode(),"编辑时,人员编码不能为空");
        verificationUserDto.setTenantCode(TenantUtils.getTenantCode());
        verificationUserDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        verificationUserDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        VerificationUser verificationUser = this.nebulaToolkitService.copyObjectByWhiteList(verificationUserDto, VerificationUser.class, HashSet.class, LinkedList.class);
        this.verificationUserRepository.saveOrUpdate(verificationUser);
        return this.nebulaToolkitService.copyObjectByWhiteList(verificationUser, VerificationUserVo.class, HashSet.class, LinkedList.class);
    }

    @Override
    public void enableBatch(List<String> ids) {
        if(CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.verificationUserRepository.lambdaUpdate().set(VerificationUser::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
                .in(VerificationUser::getId,ids).update();

    }

    @Override
    public void disableBatch(List<String> ids) {
        if(CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.verificationUserRepository.lambdaUpdate().set(VerificationUser::getEnableStatus, EnableStatusEnum.DISABLE.getCode())
                .in(VerificationUser::getId,ids).update();

    }

    @Override
    public void deleteBatch(List<String> ids) {
        if(CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.verificationUserRepository.lambdaUpdate().set(VerificationUser::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
                .in(VerificationUser::getId,ids).update();

    }

    /**
     * 小程序根据组织获取分发业务员
     * @Author 吴平
     * @return
     */
    @Override
    public List<VerificationUserVo> listDispatchSalesman() {
        String memberCode = this.userSearchHelper.getMemberLogin().getMemberCode();
        MemberInfoArchives memberInfoArchives = this.memberInfoArchivesRepository.lambdaQuery()
            .eq(MemberInfoArchives::getMemberCode, memberCode)
            .eq(MemberInfoArchives::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
            .eq(MemberInfoArchives::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
            .one();
        Validate.notNull(memberInfoArchives, "当前登录用户的档案信息不存在");
        String orgCode = memberInfoArchives.getOrgCode();
        Validate.notBlank(orgCode, "当前登录用户没有所属组织信息");
        List<VerificationUser> verificationUserList = this.verificationUserRepository.lambdaQuery()
            .eq(VerificationUser::getOrgCode, orgCode)
            .eq(VerificationUser::getUserType, VerificationUserTypeEnum.STUFF.getCode())
            .eq(VerificationUser::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
            .eq(VerificationUser::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
            .list();
        if (CollectionUtils.isEmpty(verificationUserList)) {
            verificationUserList = Collections.EMPTY_LIST;
        }
        Collection<VerificationUserVo> result = this.nebulaToolkitService
            .copyCollectionByWhiteList(verificationUserList,
                VerificationUser.class,
                VerificationUserVo.class,
                HashSet.class,
                LinkedList.class);
        return (List<VerificationUserVo>) result;
    }
}
