package com.biz.crm.sfa.admin.web.mobile.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.LoginUserDetails;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.customer.sdk.dto.CustomerClientDto;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerClientVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerClientVo;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerContactVo;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalClientDto;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalClientVoService;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalClientVo;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalContactVo;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalTagVo;
import com.biz.crm.sfa.admin.web.mobile.dto.ClientDto;
import com.biz.crm.sfa.admin.web.mobile.service.ClientVoService;
import com.biz.crm.sfa.admin.web.mobile.vo.ClientContactVo;
import com.biz.crm.sfa.admin.web.mobile.vo.ClientVo;
import com.biz.crm.sfa.business.client.sdk.enums.ClientTypeEnum;
import com.biz.crm.sfa.business.visit.plan.local.entity.VisitPlanDetail;
import com.biz.crm.sfa.business.visit.plan.local.service.VisitPlanDetailService;
import com.biz.crm.sfa.business.visit.plan.sdk.enums.VisitStatusEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.time.LocalDate;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 客户管理服务实现类
 *
 * @author songjingen
 */
@Service
public class ClientVoServiceImpl implements ClientVoService {

  @Autowired
  private TerminalClientVoService terminalClientVoService;
  @Autowired
  private LoginUserService loginUserService;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private CustomerClientVoService customerClientVoService;
  @Autowired
  private VisitPlanDetailService visitPlanDetailService;

  @Override
  public Page<ClientVo> findByConditions(Pageable pageable, ClientDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (dto == null) {
      dto = new ClientDto();
    }
    String clientType = dto.getClientType();
    if (StringUtils.isBlank(clientType)) {
      return new Page<>();
    }
    if (ClientTypeEnum.TERMINAL.getDictCode().equals(clientType)) {
      return getTerminalClient(pageable, dto);
    }
    if (ClientTypeEnum.DEALER.getDictCode().equals(clientType)) {
      return getCustomerClient(pageable, dto);
    }
    return new Page<>();
  }

  @Override
  public Page<ClientVo> findNotVisitClientByVisitDate(Pageable pageable, ClientDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (dto == null) {
      dto = new ClientDto();
    }
    LoginUserDetails loginDetails = this.loginUserService.getLoginDetails(LoginUserDetails.class);
    //查看当前用户指定日期的拜访计划
    List<VisitPlanDetail> visitPlanDetails = this.visitPlanDetailService.findByVisitDateAndUserName(dto.getVisitDate(), loginDetails.getAccount());
    //查询终端数据
    String clientType = dto.getClientType();
    if (ClientTypeEnum.TERMINAL.getDictCode().equals(clientType)) {
      List<String> excludeClientList = new ArrayList<>();
      if (!CollectionUtils.isEmpty(visitPlanDetails)) {
        excludeClientList = visitPlanDetails.stream().filter(visitPlanDetail -> ClientTypeEnum.TERMINAL.getDictCode().equals(visitPlanDetail.getClientType())).map(VisitPlanDetail::getClientCode).collect(Collectors.toList());
      }
      dto.setExcludeClientList(excludeClientList);
      return getTerminalClient(pageable, dto);
    }
    //查询经销商数据
    if (ClientTypeEnum.DEALER.getDictCode().equals(clientType)) {
      List<String> excludeClientList = new ArrayList<>();
      if (!CollectionUtils.isEmpty(visitPlanDetails)) {
        excludeClientList = visitPlanDetails.stream().filter(visitPlanDetail -> ClientTypeEnum.DEALER.getDictCode().equals(visitPlanDetail.getClientType())).map(VisitPlanDetail::getClientCode).collect(Collectors.toList());
      }
      dto.setExcludeClientList(excludeClientList);
      return getCustomerClient(pageable, dto);
    }
    return null;
  }

  /**
   * 获取终端客户
   *
   * @param pageable
   * @param dto
   * @return
   */
  private Page<ClientVo> getTerminalClient(Pageable pageable, ClientDto dto) {
    LoginUserDetails loginDetails = this.loginUserService.getLoginDetails(LoginUserDetails.class);
    TerminalClientDto terminalClientDto = new TerminalClientDto();
    terminalClientDto.setTenantCode(TenantUtils.getTenantCode());
    if (StringUtils.isBlank(dto.getUserName())) {
      terminalClientDto.setUserName(loginDetails.getAccount());
    } else {
      terminalClientDto.setUserName(dto.getUserName());
    }
    terminalClientDto.setTerminalCode(dto.getClientCode());
    terminalClientDto.setTerminalName(dto.getClientName());
    terminalClientDto.setKeyWord(dto.getKeyWord());
    terminalClientDto.setProcessStatus(dto.getProcessStatus());
    terminalClientDto.setTerminalType(dto.getClientSubType());
    terminalClientDto.setExcludeTerminalCodes(dto.getExcludeClientList());
    Page<TerminalClientVo> page = this.terminalClientVoService.findChildrenPageByTerminalClientDto(pageable, terminalClientDto);
    if (page == null) {
      return new Page<>();
    }
    List<TerminalClientVo> records = page.getRecords();
    //组装数据
    Page<ClientVo> clientVoPage = new Page<ClientVo>();
    BeanUtils.copyProperties(page, clientVoPage);
    ArrayList<ClientVo> clientVos = new ArrayList<>();
    records.stream().forEach(terminalClientVo -> {
      ClientVo clientVo = this.nebulaToolkitService.copyObjectByBlankList(terminalClientVo, ClientVo.class, HashSet.class, ArrayList.class);
      clientVo.setClientCode(terminalClientVo.getTerminalCode());
      clientVo.setClientName(terminalClientVo.getTerminalName());
      clientVo.setClientAddress(StringUtils.joinWith("", terminalClientVo.getProvinceName(), terminalClientVo.getCityName(), terminalClientVo.getDistrictName(), terminalClientVo.getTerminalAddress()));
      clientVo.setClientSubType(terminalClientVo.getTerminalType());
      if (!CollectionUtils.isEmpty(terminalClientVo.getTags())) {
        clientVo.setClientTags(terminalClientVo.getTags().stream().map(TerminalTagVo::getTagDescription).collect(Collectors.toList()));
      }
      List<TerminalContactVo> contacts = terminalClientVo.getContacts();
      if (!CollectionUtils.isEmpty(contacts)) {
        clientVo.setClientContactVos((List<ClientContactVo>) this.nebulaToolkitService.copyCollectionByWhiteList(contacts, TerminalContactVo.class, ClientContactVo.class, HashSet.class, ArrayList.class));
      }
      clientVos.add(clientVo);
    });
    this.buildVisitCount(clientVos, dto.getClientType());
    clientVoPage.setRecords(clientVos);
    return clientVoPage;
  }

  /**
   * 获取经销商客户
   *
   * @param pageable
   * @param dto
   * @return
   */
  private Page<ClientVo> getCustomerClient(Pageable pageable, ClientDto dto) {
    LoginUserDetails loginDetails = this.loginUserService.getLoginDetails(LoginUserDetails.class);
    CustomerClientDto customerClientDto = new CustomerClientDto();
    customerClientDto.setTenantCode(TenantUtils.getTenantCode());
    if (StringUtils.isBlank(dto.getUserName())) {
      customerClientDto.setUserName(loginDetails.getAccount());
    } else {
      customerClientDto.setUserName(dto.getUserName());
    }
    customerClientDto.setCustomerCode(dto.getClientCode());
    customerClientDto.setCustomerName(dto.getClientName());
    customerClientDto.setKeyWord(dto.getKeyWord());
    customerClientDto.setProcessStatus(dto.getProcessStatus());
    customerClientDto.setCustomerType(dto.getClientSubType());
    customerClientDto.setExcludeCustomerCodes(dto.getExcludeClientList());
    Page<CustomerClientVo> page = this.customerClientVoService.findChildrenPageByCustomerClientDto(pageable, customerClientDto);
    if (page == null) {
      return new Page<>();
    }
    List<CustomerClientVo> records = page.getRecords();
    //组装数据
    Page<ClientVo> clientVoPage = new Page<ClientVo>();
    BeanUtils.copyProperties(page, clientVoPage);
    List<ClientVo> clientVos = new ArrayList<>();
    records.stream().forEach(customerClientVo -> {
      ClientVo clientVo = this.nebulaToolkitService.copyObjectByBlankList(customerClientVo, ClientVo.class, HashSet.class, ArrayList.class);
      clientVo.setClientCode(customerClientVo.getCustomerCode());
      clientVo.setClientName(customerClientVo.getCustomerName());
      clientVo.setClientAddress(StringUtils.joinWith("", customerClientVo.getProvinceName(), customerClientVo.getCityName(), customerClientVo.getDistrictName(), customerClientVo.getRegisteredAddress()));
      clientVo.setClientSubType(customerClientVo.getCustomerType());
      List<CustomerContactVo> contacts = customerClientVo.getContactList();
      if (!CollectionUtils.isEmpty(contacts)) {
        clientVo.setClientContactVos((List<ClientContactVo>) this.nebulaToolkitService.copyCollectionByWhiteList(contacts, CustomerContactVo.class, ClientContactVo.class, HashSet.class, ArrayList.class));
      }
      clientVos.add(clientVo);
    });
    this.buildVisitCount(clientVos, dto.getClientType());
    clientVoPage.setRecords(clientVos);
    return clientVoPage;
  }

  /**
   * 组装未拜访数据
   *
   * @param clientVos
   */
  private void buildVisitCount(List<ClientVo> clientVos, String clientType) {
    if (CollectionUtils.isEmpty(clientVos) || StringUtils.isBlank(clientType)) {
      return;
    }
    Set<String> clientCodes = clientVos.stream().map(ClientVo::getClientCode).collect(Collectors.toSet());
    List<VisitPlanDetail> list = this.visitPlanDetailService.findByClientCodesAndClientTypeAndVisitStatus(clientCodes, clientType, VisitStatusEnum.VISIT_TO_COMPLETE.getDictCode());
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    //按照客户编码分组,然后用拜访日期排序,取最后一条数据
    Map<String, List<VisitPlanDetail>> listMap = list.stream().collect(Collectors.groupingBy(VisitPlanDetail::getClientCode));
    LocalDate nowLocalDate = LocalDate.now();
    for (ClientVo clientVo : clientVos) {
      String clientCode = clientVo.getClientCode();
      List<VisitPlanDetail> details = listMap.get(clientCode);
      if (!CollectionUtils.isEmpty(details)) {
        VisitPlanDetail visitPlanDetail = details.stream().sorted(Comparator.comparing(VisitPlanDetail::getVisitDate).reversed()).findFirst().orElse(null);
        LocalDate visitLocalDate = visitPlanDetail.getVisitDate().toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
        clientVo.setVisitCount(Long.valueOf(ChronoUnit.DAYS.between(visitLocalDate, nowLocalDate)).intValue());
      }
    }
  }

}
