package com.biz.crm.sfa.admin.web.strategy.internal;

import com.biz.crm.business.common.sdk.model.LoginUserDetails;
import com.biz.crm.common.sms.sdk.service.ValiditySmsCodeService;
import com.biz.crm.mdm.business.user.sdk.constant.UserConstant;
import com.biz.crm.mdm.business.user.sdk.service.UserValidityCheckService;
import com.biz.crm.mdm.business.user.sdk.vo.UserVo;
import com.biz.crm.sfa.admin.web.strategy.DefaultAuthenticationDecisionStrategy;
import com.bizunited.nebula.common.util.Aes128Utils;
import com.bizunited.nebula.security.sdk.event.AuthenticationDecisionStrategy;
import com.bizunited.nebula.security.sdk.event.DecisionTypes;
import com.bizunited.nebula.security.sdk.vo.LoginDetails;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * 手机号 + 密码方式策略实现
 *
 * @author pengxi
 */
@Slf4j
@Component
public class DefaultPhoneAndSmsCodeStrategy extends DefaultAuthenticationDecisionStrategy implements AuthenticationDecisionStrategy {

  @Autowired
  private UserValidityCheckService userValidityCheckService;

  @Autowired
  private ValiditySmsCodeService validitySmsCodeService;

  @Override
  public DecisionTypes type() {
    return DecisionTypes.PHONEANDSMSCODE;
  }

  @Override
  public boolean onAuthenticate(LoginDetails loginDetails) {
    /*
     * 处理过程如下：
     * 1、首先通过手机号查询当前用户并检查用户有效性
     * 2、检查当前短信验证码的有效性
     * 3、填充security鉴权必要信息：tenantCode,account,password
     * 4、然后完善登录信息中的用户基本信息
     * 5、最后完善登录信息中的岗位组织等信息
     */
    LoginUserDetails loginUserDetails = (LoginUserDetails) loginDetails;
    log.info("HttpServletRequest phone:{}", loginUserDetails.getPhone());
    log.info("HttpServletRequest smsVerificationCode:{}", loginUserDetails.getSmsVerificationCode());
    Validate.notBlank(loginUserDetails.getPhone(), "登录手机号不能为空！");
    Validate.notBlank(loginUserDetails.getSmsVerificationCode(), "短信验证码不能为空！");
    String phone = Aes128Utils.decrypt(loginUserDetails.getPhone(), UserConstant.ENCRYPT_KEY, Aes128Utils.EncodeType.CBC, Aes128Utils.Padding.PKCS_7_PADDING);
    log.error("decrypt phone:{}", phone);
    String smsVerificationCode = Aes128Utils.decrypt(loginUserDetails.getSmsVerificationCode(), UserConstant.ENCRYPT_KEY, Aes128Utils.EncodeType.CBC, Aes128Utils.Padding.PKCS_7_PADDING);
    log.error("decrypt smsVerificationCode:{}", smsVerificationCode);
    // 1、
    UserVo userVo = this.userValidityCheckService.verificationManageByPhone(phone);
    Validate.notNull(userVo, "当前系统不存在此手机号，请联系管理员！");
    // 2、
    Boolean isAvailable = this.validitySmsCodeService.isAvailableVerificationCode(phone, smsVerificationCode);
    Validate.isTrue(isAvailable, "验证码过期或不正确，请重新获取！");
    // 3、4、
    loginUserDetails.setType(DecisionTypes.PHONEANDSMSCODE.getCode());
    super.perfectLoginUserDetails(userVo, loginUserDetails);
    // 5、
    super.perfectLoginPostAndOrg(loginUserDetails);
    return true;
  }
}
