package com.biz.crm.sfa.business.action.scheme.local.service.internal;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.sfa.business.action.scheme.local.entity.SchemeActionEntity;
import com.biz.crm.sfa.business.action.scheme.local.repository.SchemeActionRepository;
import com.biz.crm.sfa.business.action.scheme.local.service.SchemeActionService;
import com.biz.crm.sfa.business.action.scheme.sdk.constant.SchemeActionConstant;
import com.biz.crm.sfa.business.action.scheme.sdk.dto.SchemeActionDto;
import com.biz.crm.sfa.business.action.scheme.sdk.enums.SchemeActionStatus;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessDto;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessMappingDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.biz.crm.workflow.sdk.service.ProcessBusinessMappingService;
import com.biz.crm.workflow.sdk.service.ProcessBusinessService;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessMappingVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;

/**
 * 方案活动表服务实现类
 *
 * @author ning.zhang
 * @date 2022-08-09 17:17:19
 */
@Slf4j
@Service("schemeActionService")
public class SchemeActionServiceImpl implements SchemeActionService {

  @Autowired
  private SchemeActionRepository schemeActionRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private GenerateCodeService generateCodeService;
  @Autowired
  private ProcessBusinessService processBusinessService;
  @Autowired
  private ProcessBusinessMappingService processBusinessMappingService;

  @Override
  @Transactional
  public SchemeActionEntity create(SchemeActionDto dto) {
    this.createValidation(dto);
    SchemeActionEntity entity = this.nebulaToolkitService.copyObjectByBlankList(dto, SchemeActionEntity.class, HashSet.class, ArrayList.class);
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    entity.setActionStatus(SchemeActionStatus.NORMAL.getDictCode());
    this.schemeActionRepository.save(entity);
    if (Boolean.TRUE.equals(dto.getSubmitProcess())) {
      this.commitProcess(dto);
    }
    return entity;
  }

  @Override
  @Transactional
  public SchemeActionEntity update(SchemeActionDto dto) {
    this.updateValidation(dto);
    SchemeActionEntity actionEntity = this.schemeActionRepository.getById(dto.getId());
    Validate.notNull(actionEntity, "方案活动信息不存在");
    Validate.isTrue(actionEntity.getActionCode().equals(dto.getActionCode()), "方案活动编码不能编辑");
    ProcessBusinessMappingDto businessMappingDto = new ProcessBusinessMappingDto();
    businessMappingDto.setBusinessCode(SchemeActionConstant.SCHEME_ACTION_PROCESS_FORM_TYPE);
    businessMappingDto.setBusinessNo(dto.getActionCode());
    ProcessBusinessMappingVo businessMappingVo = this.processBusinessMappingService.findSignalByConditions(businessMappingDto);
    Validate.isTrue(Objects.isNull(businessMappingVo)
            || ProcessStatusEnum.REJECT.getDictCode().equals(businessMappingVo.getProcessStatus())
            || ProcessStatusEnum.RECOVER.getDictCode().equals(businessMappingVo.getProcessStatus())
        , "当前审批状态方案活动不能编辑");
    SchemeActionEntity updateEntity = this.nebulaToolkitService.copyObjectByBlankList(dto, SchemeActionEntity.class, HashSet.class, ArrayList.class);
    updateEntity.setActionStatus(SchemeActionStatus.NORMAL.getDictCode());
    this.schemeActionRepository.updateById(updateEntity);
    if (Boolean.TRUE.equals(dto.getSubmitProcess())) {
      this.commitProcess(dto);
    }
    return updateEntity;
  }

  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    List<SchemeActionEntity> entities = this.schemeActionRepository.listByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据删除个数不匹配");
    //活动删除，审批驳回的，活动时间还没有开始的，没有终端门店参与活动的可以删除
    //TODO 删除校验
    this.schemeActionRepository.updateDelFlagByIds(ids);
  }

  @Override
  @Transactional
  public void closeBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    List<SchemeActionEntity> entities = this.schemeActionRepository.listByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据关闭个数不匹配");
    //活动关闭，审批通过的都能关闭，活动关闭后对以签协议的终端没影响，不能新增终端参与活动
    //TODO 关闭校验
    this.schemeActionRepository.updateActionStatusByIds(ids, SchemeActionStatus.CLOSED);
  }

  /**
   * 方案活动提交工作流进行审批
   *
   * @param dto 方案活动DTO
   */
  private void commitProcess(SchemeActionDto dto) {
    ProcessBusinessDto processBusiness = dto.getProcessBusiness();
    processBusiness.setBusinessNo(dto.getActionCode());
    processBusiness.setBusinessCode(SchemeActionConstant.SCHEME_ACTION_PROCESS_FORM_TYPE);
    this.processBusinessService.processStart(processBusiness);
  }

  /**
   * 在创建Scheme模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(SchemeActionDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setActionCode(generateCodeService.generateCode("SAC", 1).get(0));
    Validate.notBlank(dto.getSchemeCode(), "缺失方案编码");
    Validate.notBlank(dto.getActionName(), "缺失活动名称");
    Validate.notBlank(dto.getExchangeTypeKey(), "缺失兑换方式类型key");
    Validate.notNull(dto.getActionStartDate(), "缺失活动开始时间");
    Validate.notNull(dto.getActionEndDate(), "缺失活动结束时间");
    Validate.notNull(dto.getOaAccount(), "缺失OA核销号");
    Validate.isTrue(Objects.nonNull(dto.getActionCost()) && dto.getActionCost().compareTo(BigDecimal.ZERO) >= 0, "活动费用数据错误");
    Validate.isTrue(Objects.nonNull(dto.getBoxCost()) && dto.getBoxCost().compareTo(BigDecimal.ZERO) >= 0, "单箱成本数据错误");
    Validate.isTrue(Objects.nonNull(dto.getPlanSales()) && dto.getPlanSales().compareTo(BigDecimal.ZERO) >= 0, "计划销量数据错误");
    Validate.isTrue(Objects.nonNull(dto.getPlanStoreQuantity()) && dto.getPlanStoreQuantity().compareTo(BigDecimal.ZERO) >= 0, "计划投入门店数量数据错误");
    Validate.isTrue(dto.getActionStartDate().getTime() < dto.getActionEndDate().getTime(), "活动结束时间需大于活动开始时间");
    Date nowDate = new Date();
    Validate.isTrue(nowDate.getTime() < dto.getActionEndDate().getTime(), "活动结束时间需大于当前时间");
  }

  /**
   * 在修改Scheme模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(SchemeActionDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getId(), "修改信息时，id不能为空！");
    Validate.notBlank(dto.getSchemeCode(), "缺失方案编码");
    Validate.notBlank(dto.getActionCode(), "缺失活动编码");
    Validate.notBlank(dto.getActionName(), "缺失活动名称");
    Validate.notBlank(dto.getExchangeTypeKey(), "缺失兑换方式类型key");
    Validate.notNull(dto.getActionStartDate(), "缺失活动开始时间");
    Validate.notNull(dto.getActionEndDate(), "缺失活动结束时间");
    Validate.notNull(dto.getOaAccount(), "缺失OA核销号");
    Validate.isTrue(Objects.nonNull(dto.getActionCost()) && dto.getActionCost().compareTo(BigDecimal.ZERO) >= 0, "活动费用数据错误");
    Validate.isTrue(Objects.nonNull(dto.getBoxCost()) && dto.getBoxCost().compareTo(BigDecimal.ZERO) >= 0, "单箱成本数据错误");
    Validate.isTrue(Objects.nonNull(dto.getPlanSales()) && dto.getPlanSales().compareTo(BigDecimal.ZERO) >= 0, "计划销量数据错误");
    Validate.isTrue(Objects.nonNull(dto.getPlanStoreQuantity()) && dto.getPlanStoreQuantity().compareTo(BigDecimal.ZERO) >= 0, "计划投入门店数量数据错误");
    Validate.isTrue(dto.getActionStartDate().getTime() < dto.getActionEndDate().getTime(), "活动结束时间需大于活动开始时间");
    Date nowDate = new Date();
    Validate.isTrue(nowDate.getTime() < dto.getActionEndDate().getTime(), "活动结束时间需大于当前时间");
  }
}
