package com.biz.crm.sfa.business.action.scheme.local.service.internal;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.sfa.business.action.scheme.local.entity.SchemeEntity;
import com.biz.crm.sfa.business.action.scheme.local.model.SchemeConditionModel;
import com.biz.crm.sfa.business.action.scheme.local.repository.SchemeRepository;
import com.biz.crm.sfa.business.action.scheme.local.service.SchemeFileService;
import com.biz.crm.sfa.business.action.scheme.local.service.SchemeScopeService;
import com.biz.crm.sfa.business.action.scheme.local.service.SchemeService;
import com.biz.crm.sfa.business.action.scheme.local.service.SchemeStoreTypeService;
import com.biz.crm.sfa.business.action.scheme.sdk.dto.SchemeDto;
import com.biz.crm.sfa.business.action.scheme.sdk.register.SchemeRegister;
import com.biz.crm.sfa.business.action.scheme.sdk.vo.SchemeVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;

/**
 * 方案表服务实现类
 *
 * @author ning.zhang
 * @date 2022-08-05 14:32:18
 */
@Slf4j
@Service("schemeService")
public class SchemeServiceImpl implements SchemeService {

  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private GenerateCodeService generateCodeService;
  @Autowired
  private SchemeRepository schemeRepository;
  @Autowired(required = false)
  private List<SchemeRegister> registers;
  @Autowired
  private SchemeScopeService schemeScopeService;
  @Autowired
  private SchemeFileService schemeFileService;
  @Autowired
  private SchemeStoreTypeService schemeStoreTypeService;

  @Override
  @Transactional
  public SchemeEntity create(SchemeDto dto) {
    this.createValidation(dto);
    SchemeEntity entity = this.nebulaToolkitService.copyObjectByBlankList(dto, SchemeEntity.class, HashSet.class, ArrayList.class);
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    this.schemeRepository.save(entity);
    //绑定关联数据
    this.schemeScopeService.update(entity);
    this.schemeFileService.update(entity);
    this.schemeStoreTypeService.update(entity);
    //新增方案事件通知
    if (!CollectionUtils.isEmpty(registers)) {
      SchemeVo vo = this.nebulaToolkitService.copyObjectByBlankList(entity, SchemeVo.class, HashSet.class, ArrayList.class);
      registers.forEach(register -> {
        register.onCreate(vo);
      });
    }
    return entity;
  }

  @Override
  @Transactional
  public SchemeEntity update(SchemeDto dto) {
    this.updateValidation(dto);
    SchemeConditionModel model = new SchemeConditionModel();
    model.setIds(Sets.newHashSet(dto.getId()));
    model.setTenantCode(dto.getTenantCode());
    List<SchemeEntity> entities = this.schemeRepository.findBySchemeConditionModel(model);
    Validate.isTrue(!CollectionUtils.isEmpty(entities), "方案信息不存在");
    SchemeEntity oldEntity = entities.get(0);
    Validate.isTrue(oldEntity.getSchemeCode().equals(dto.getSchemeCode()), "方案编码不能编辑");
    Validate.isTrue(oldEntity.getTypeKey().equals(dto.getTypeKey()), "方案类型不能编辑");
    Validate.isTrue(oldEntity.getSchemeStartDate().getTime() > System.currentTimeMillis(), "只能编辑未开始方案");
    SchemeEntity updateEntity = this.nebulaToolkitService.copyObjectByBlankList(dto, SchemeEntity.class, HashSet.class, ArrayList.class);
    this.schemeRepository.updateById(updateEntity);
    //绑定关联数据
    this.schemeScopeService.update(updateEntity);
    this.schemeFileService.update(updateEntity);
    this.schemeStoreTypeService.update(updateEntity);
    //更新方案事件通知
    if (!CollectionUtils.isEmpty(registers)) {
      SchemeVo oldVo = this.nebulaToolkitService.copyObjectByBlankList(oldEntity, SchemeVo.class, HashSet.class, ArrayList.class);
      SchemeVo newVo = this.nebulaToolkitService.copyObjectByBlankList(updateEntity, SchemeVo.class, HashSet.class, ArrayList.class);
      registers.forEach(register -> {
        register.onUpdate(oldVo, newVo);
      });
    }
    return updateEntity;
  }

  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    List<SchemeEntity> entities = this.schemeRepository.listByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据删除个数不匹配");
    entities.forEach(entity -> {
      Validate.isTrue(entity.getSchemeStartDate().getTime() > System.currentTimeMillis(), "只能删除未开始方案");
    });
    this.schemeRepository.updateDelFlagByIds(ids);
    //删除方案事件通知
    if (!CollectionUtils.isEmpty(registers)) {
      List<SchemeVo> voList = (List<SchemeVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, SchemeEntity.class
          , SchemeVo.class, HashSet.class, ArrayList.class);
      registers.forEach(register -> {
        register.onDelete(voList);
      });
    }
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    SchemeConditionModel model = new SchemeConditionModel();
    model.setIds(Sets.newHashSet(ids));
    model.setTenantCode(TenantUtils.getTenantCode());
    List<SchemeEntity> entities = this.schemeRepository.findBySchemeConditionModel(model);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据启用个数不匹配");
    this.schemeRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
    //启用方案事件通知
    if (!CollectionUtils.isEmpty(registers)) {
      List<SchemeVo> voList = (List<SchemeVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, SchemeEntity.class
          , SchemeVo.class, HashSet.class, ArrayList.class);
      registers.forEach(register -> {
        register.onEnable(voList);
      });
    }
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    SchemeConditionModel model = new SchemeConditionModel();
    model.setIds(Sets.newHashSet(ids));
    model.setTenantCode(TenantUtils.getTenantCode());
    List<SchemeEntity> entities = this.schemeRepository.findBySchemeConditionModel(model);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据禁用个数不匹配");
    this.schemeRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
    //启用方案事件通知
    if (!CollectionUtils.isEmpty(registers)) {
      List<SchemeVo> voList = (List<SchemeVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, SchemeEntity.class
          , SchemeVo.class, HashSet.class, ArrayList.class);
      registers.forEach(register -> {
        register.onDisable(voList);
      });
    }
  }


  /**
   * 在创建Scheme模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(SchemeDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setSchemeCode(generateCodeService.generateCode("SC", 1).get(0));
    Validate.notBlank(dto.getSchemeName(), "缺失方案名称");
    Validate.notBlank(dto.getTypeKey(), "缺失方案类型key");
    Validate.notNull(dto.getSchemeStartDate(), "缺失方案开始时间");
    Validate.notNull(dto.getSchemeEndDate(), "缺失方案结束时间");
    Validate.isTrue(dto.getSchemeStartDate().getTime() < dto.getSchemeEndDate().getTime(), "方案结束时间需大于方案开始时间");
    Date nowDate = new Date();
    Validate.isTrue(nowDate.getTime() < dto.getSchemeEndDate().getTime(), "方案结束时间需大于当前时间");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getStoreTypeList()), "缺失方案门店类型");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getScopeList()), "缺失方案范围");
  }

  /**
   * 在修改Scheme模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(SchemeDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getId(), "修改信息时，id不能为空！");
    Validate.notBlank(dto.getSchemeCode(), "缺失方案编码");
    Validate.notBlank(dto.getSchemeName(), "缺失方案名称");
    Validate.notBlank(dto.getTypeKey(), "缺失方案类型key");
    Validate.notNull(dto.getSchemeStartDate(), "缺失方案开始时间");
    Validate.notNull(dto.getSchemeEndDate(), "缺失方案结束时间");
    Validate.isTrue(dto.getSchemeStartDate().getTime() < dto.getSchemeEndDate().getTime(), "方案结束时间需大于方案开始时间");
    Date nowDate = new Date();
    Validate.isTrue(nowDate.getTime() < dto.getSchemeEndDate().getTime(), "方案结束时间需大于当前时间");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getStoreTypeList()), "缺失方案门店类型");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getScopeList()), "缺失方案范围");
  }
}
