package com.biz.crm.sfa.business.attendance.local.repository;

import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.sfa.business.attendance.local.entity.AttendanceRecordEntity;
import com.biz.crm.sfa.business.attendance.local.mapper.AttendanceRecordMapper;
import com.biz.crm.sfa.business.attendance.sdk.constant.AttendanceConstant;
import com.biz.crm.sfa.business.attendance.sdk.dto.AttendanceFillClockDto;
import com.biz.crm.sfa.business.attendance.sdk.dto.TodayHistoryRecordDto;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceClockStatus;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;
import java.util.List;

/**
 * 考勤记录表的数据库访问类 {@link AttendanceRecordEntity}
 *
 * @author ning.zhang
 * @date 2022-05-17 09:52:18
 */
@Component
public class AttendanceRecordRepository extends ServiceImpl<AttendanceRecordMapper, AttendanceRecordEntity> {

  /**
   * 通过考勤日期和使用状态删除考勤记录信息
   *
   * @param tenantCode 租户编码
   * @param ruleDate   考勤日期
   * @param usedStatus 使用状态
   */
  public void deleteByRuleDateAndUsedStatus(String tenantCode, String ruleDate, String usedStatus) {
    this.lambdaUpdate()
        .eq(AttendanceRecordEntity::getRuleDate, ruleDate)
        .eq(AttendanceRecordEntity::getTenantCode, tenantCode)
        .eq(AttendanceRecordEntity::getUsedStatus, usedStatus)
        .remove();
  }

  /**
   * 通过考勤日期和使用状态获取考勤记录信息
   *
   * @param tenantCode 租户编码
   * @param ruleDate   考勤日期
   * @param usedStatus 使用状态
   * @return 考勤记录信息
   */
  public List<AttendanceRecordEntity> findByRuleDateAndUsedStatus(String tenantCode, String ruleDate, String usedStatus) {
    return this.lambdaQuery()
        .eq(AttendanceRecordEntity::getRuleDate, ruleDate)
        .eq(AttendanceRecordEntity::getTenantCode, tenantCode)
        .eq(AttendanceRecordEntity::getUsedStatus, usedStatus)
        .list();
  }

  /**
   * 通过考勤日期和用户名和打卡类型获取考勤记录
   *
   * @param tenantCode 租户编码
   * @param userName   用户名
   * @param ruleDate   考勤日期
   * @param clockType  打卡类型
   * @return 考勤记录
   */
  public List<AttendanceRecordEntity> findByUserNameAndRuleDateAndClockType(String tenantCode, String userName
      , String ruleDate, String clockType) {
    return this.lambdaQuery()
        .eq(AttendanceRecordEntity::getRuleDate, ruleDate)
        .eq(AttendanceRecordEntity::getTenantCode, tenantCode)
        .eq(AttendanceRecordEntity::getUserName, userName)
        .eq(AttendanceRecordEntity::getClockType, clockType)
        .list();
  }

  /**
   * 通过用户名和规则时间ID获取考勤记录
   *
   * @param tenantCode 租户编码
   * @param userName   用户名
   * @param ruleTimeId 规则时间ID
   * @return 考勤记录
   */
  public List<AttendanceRecordEntity> findByUserNameAndRuleTimeId(String tenantCode, String userName, String ruleTimeId) {
    return this.lambdaQuery()
        .eq(AttendanceRecordEntity::getRuleTimeId, ruleTimeId)
        .eq(AttendanceRecordEntity::getTenantCode, tenantCode)
        .eq(AttendanceRecordEntity::getUserName, userName)
        .list();
  }

  /**
   * 通过用户名和考勤记录规则ID更新考勤记录使用状态
   *
   * @param tenantCode   租户编码
   * @param userName     用户名
   * @param recordRuleId 考勤记录规则ID
   * @param usedStatus   使用状态
   */
  public void updateUsedStatusByUserNameAndRecordRuleId(String tenantCode, String userName, String recordRuleId, String usedStatus) {
    this.lambdaUpdate()
        .eq(AttendanceRecordEntity::getUserName, userName)
        .eq(AttendanceRecordEntity::getRecordRuleId, recordRuleId)
        .eq(AttendanceRecordEntity::getTenantCode, tenantCode)
        .set(AttendanceRecordEntity::getUsedStatus, usedStatus)
        .update();
  }

  /**
   * 查询今日历史考勤记录信息
   *
   * @param dto 参数dto
   * @return 考勤记录信息
   */
  public List<AttendanceRecordEntity> findByTodayHistoryRecordDto(TodayHistoryRecordDto dto) {
    return this.lambdaQuery()
        .eq(AttendanceRecordEntity::getTenantCode, dto.getTenantCode())
        .eq(AttendanceRecordEntity::getClockType, dto.getClockType())
        .eq(AttendanceRecordEntity::getUserName, dto.getUserName())
        .eq(AttendanceRecordEntity::getRuleDate, LocalDateTime.now().format(AttendanceConstant.YYYY_MM_DD))
        .le(AttendanceRecordEntity::getRuleClockTime, LocalDateTime.now().format(AttendanceConstant.HH_MM_SS))
        .ne(AttendanceRecordEntity::getId, dto.getBeforeRecordId())
        .list();
  }

  /**
   * 通过规则ID集合和考勤规则日期删除数据
   *
   * @param tenantCode 租户编码
   * @param ruleIds    则ID集合
   * @param ruleDate   勤规则日期
   */
  public void deleteByRuleIdsAndRuleDate(String tenantCode, List<String> ruleIds, String ruleDate) {
    this.lambdaUpdate()
        .eq(AttendanceRecordEntity::getRuleDate, ruleDate)
        .eq(AttendanceRecordEntity::getTenantCode, tenantCode)
        .in(AttendanceRecordEntity::getRuleId, ruleIds)
        .remove();
  }

  /**
   * 查询补打卡考勤信息
   *
   * @param dto 参数dto
   * @return 补打卡考勤信息
   */
  public List<AttendanceRecordEntity> findByAttendanceFillClockDto(AttendanceFillClockDto dto) {
    return this.lambdaQuery()
        .eq(AttendanceRecordEntity::getTenantCode, dto.getTenantCode())
        .between(AttendanceRecordEntity::getRuleDate, dto.getBeginTime(), dto.getEndTime())
        .eq(AttendanceRecordEntity::getUserName, dto.getUserName())
        .ne(AttendanceRecordEntity::getClockStatus, AttendanceClockStatus.OK.getDictCode())
        .orderByAsc(AttendanceRecordEntity::getRuleClockTime)
        .list();
  }

  /**
   * 通过考勤记录ID集合查询打卡详情列表
   *
   * @param ids 考勤记录ID集合
   * @return 考勤记录列表
   */
  public List<AttendanceRecordEntity> findByIds(List<String> ids) {
    return this.baseMapper.findByIds(ids);
  }

}