package com.biz.crm.sfa.business.attendance.local.repository;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.sfa.business.attendance.local.entity.AttendanceRuleEntity;
import com.biz.crm.sfa.business.attendance.local.mapper.AttendanceRuleMapper;
import com.biz.crm.sfa.business.attendance.sdk.dto.RuleConditionDto;
import com.biz.crm.sfa.business.attendance.sdk.dto.AttendanceRulePageDto;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * 考勤规则表的数据库访问类 {@link AttendanceRuleEntity}
 *
 * @author ning.zhang
 * @date 2022-05-05 09:43:06
 */
@Component
public class AttendanceRuleRepository extends ServiceImpl<AttendanceRuleMapper, AttendanceRuleEntity> {

  /**
   * 通过考勤规则ID集合获取考勤规则信息
   *
   * @param ids 考勤规则ID集合
   * @return 考勤规则信息
   */
  public List<AttendanceRuleEntity> findByIds(List<String> ids) {
    return this.baseMapper.findByIds(ids);
  }

  /**
   * 通过考勤规则名称获取考勤规则信息
   *
   * @param tenantCode 租户编码
   * @param ruleName 考勤规则名称
   * @return 考勤规则信息
   */
  public AttendanceRuleEntity findByRuleName(String tenantCode, String ruleName) {
    return this.lambdaQuery()
        .eq(AttendanceRuleEntity::getTenantCode, tenantCode)
        .eq(AttendanceRuleEntity::getRuleName, ruleName)
        .one();
  }

  /**
   * 批量删除
   *
   * @param ids ID集合
   */
  public void updateDelFlagByIds(List<String> ids) {
    this.lambdaUpdate()
        .in(AttendanceRuleEntity::getId, ids)
        .set(AttendanceRuleEntity::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
        .update();
  }

  /**
   * 批量更新启用禁用状态
   *
   * @param ids          ID集合
   * @param enableStatus 启用禁用状态
   */
  public void updateEnableStatusByIds(List<String> ids, EnableStatusEnum enableStatus) {
    this.lambdaUpdate()
        .in(AttendanceRuleEntity::getId, ids)
        .set(AttendanceRuleEntity::getEnableStatus, enableStatus.getCode())
        .update();
  }

  /**
   * 考勤规则分页列表
   *
   * @param dto      请求参数dto
   * @param pageable 分页信息
   * @return Page<AttendanceRuleEntity> 考勤规则分页信息
   */
  public Page<AttendanceRuleEntity> findByConditions(Pageable pageable, AttendanceRulePageDto dto) {
    Page<AttendanceRuleEntity> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.baseMapper.findByConditions(page, dto);
  }

  /**
   * 多条件查询考勤规则列表
   *
   * @param dto 条件dto
   * @return 考勤规则列表
   */
  public List<AttendanceRuleEntity> findByRuleConditionDto(RuleConditionDto dto) {
    return this.baseMapper.findByRuleConditionDto(dto);
  }

}
