package com.biz.crm.sfa.business.attendance.local.service.helper;

import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.org.sdk.service.OrgPositionVoService;
import com.biz.crm.mdm.business.org.sdk.vo.OrgPositionVo;
import com.biz.crm.mdm.business.position.sdk.service.PositionVoService;
import com.biz.crm.mdm.business.position.sdk.vo.PositionVo;
import com.biz.crm.mdm.business.user.sdk.service.UserInfoVoService;
import com.biz.crm.mdm.business.user.sdk.vo.UserInfoVo;
import com.biz.crm.sfa.business.attendance.local.entity.AttendanceRuleEntity;
import com.biz.crm.sfa.business.attendance.sdk.dto.RuleConditionDto;
import com.biz.crm.sfa.business.attendance.local.model.RuleExecuteModel;
import com.biz.crm.sfa.business.attendance.local.model.RuleExecuteUserModel;
import com.biz.crm.sfa.business.attendance.local.repository.AttendanceRuleRepository;
import com.biz.crm.sfa.business.attendance.sdk.dto.AttendanceRuleExecuteDto;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceRuleScopeType;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 考勤规则服务辅助类
 *
 * @author ning.zhang
 * @date 2022/5/17
 */
@Slf4j
@Service
public class AttendanceRuleServiceHelper {

  @Autowired
  private AttendanceRuleRepository attendanceRuleRepository;
  @Autowired
  private OrgPositionVoService orgPositionVoService;
  @Autowired
  private UserInfoVoService userInfoVoService;
  @Autowired
  private PositionVoService positionVoService;

  /**
   * 构建规则执行model
   *
   * @param dto 规则执行Dto
   * @return 规则执行model
   */
  public RuleExecuteModel buildRuleExecuteModel(AttendanceRuleExecuteDto dto) {
    RuleExecuteModel executeModel = new RuleExecuteModel();
    dto = ObjectUtils.defaultIfNull(dto, new AttendanceRuleExecuteDto());
    dto.setExecuteDate(StringUtils.isNotBlank(dto.getExecuteDate()) ? dto.getExecuteDate() : LocalDate.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd")));
    RuleConditionDto conditionDto = new RuleConditionDto();
    conditionDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    conditionDto.setTenantCode(TenantUtils.getTenantCode());
    List<AttendanceRuleEntity> ruleEntities = this.attendanceRuleRepository.findByRuleConditionDto(conditionDto);
    Validate.isTrue(!CollectionUtils.isEmpty(ruleEntities), "无可执行规则");
    this.buildModelRuleScopeInfo(executeModel, ruleEntities);
    this.buildModelOrgUserInfo(executeModel);
    this.buildModelRuleUserInfo(executeModel);
    return executeModel;
  }

  /**
   * 构建封装执行model 规则-范围信息
   *
   * @param executeModel 规则执行model
   * @param ruleEntities 规则信息列表
   */
  private void buildModelRuleScopeInfo(RuleExecuteModel executeModel, List<AttendanceRuleEntity> ruleEntities) {
    executeModel.setRuleMap(Maps.newHashMap());
    executeModel.setRulePositionLevelMap(Maps.newHashMap());
    executeModel.setRuleOrgMap(Maps.newHashMap());
    ruleEntities.forEach(ruleEntity -> {
      executeModel.getRuleMap().put(ruleEntity.getId(), ruleEntity);
      ruleEntity.getScopeList().forEach(scopeEntity -> {
        if (AttendanceRuleScopeType.ORG.getDictCode().equals(scopeEntity.getScopeType())) {
          Set<String> orgCodes = executeModel.getRuleOrgMap().getOrDefault(scopeEntity.getRuleId(), Sets.newHashSet());
          orgCodes.add(scopeEntity.getScopeCode());
          executeModel.getRuleOrgMap().put(scopeEntity.getRuleId(), orgCodes);
        }
        if (AttendanceRuleScopeType.POSITION_LEVEL.getDictCode().equals(scopeEntity.getScopeType())) {
          Set<String> positionLevelCodes = executeModel.getRulePositionLevelMap().getOrDefault(scopeEntity.getRuleId(), Sets.newHashSet());
          positionLevelCodes.add(scopeEntity.getScopeCode());
          executeModel.getRulePositionLevelMap().put(scopeEntity.getRuleId(), positionLevelCodes);
        }
      });
    });
  }

  /**
   * 构建封装执行model 组织-用户信息
   *
   * @param executeModel 规则执行model
   */
  private void buildModelOrgUserInfo(RuleExecuteModel executeModel) {
    Validate.isTrue(!executeModel.getRuleOrgMap().isEmpty(), "无可执行规则组织范围");
    List<String> orgCodes = Lists.newArrayList(executeModel.getRuleOrgMap().values().stream()
        .flatMap(Collection::stream).collect(Collectors.toSet()));
    List<List<String>> splitOrgCodes = Lists.partition(orgCodes, 50);
    Map<String, List<RuleExecuteUserModel>> orgUserMap = Maps.newHashMap();
    splitOrgCodes.forEach(codes -> {
      this.buildOrgUserMap(codes, orgUserMap);
    });
    executeModel.setOrgUserMap(orgUserMap);
  }

  /**
   * 构建封装组织用户关系映射
   *
   * @param orgCodes   组织编码
   * @param orgUserMap 待封装组织用户关系映射
   */
  private void buildOrgUserMap(List<String> orgCodes, Map<String, List<RuleExecuteUserModel>> orgUserMap) {
    List<OrgPositionVo> orgPositions = this.orgPositionVoService.findByOrgCodes(orgCodes);
    if (CollectionUtils.isEmpty(orgPositions)) {
      return;
    }
    Set<String> orgPositionCodes = orgPositions.stream().map(OrgPositionVo::getPositionCode).collect(Collectors.toSet());
    List<UserInfoVo> users = this.userInfoVoService.findByPositionCodes(orgPositionCodes);
    if (CollectionUtils.isEmpty(users)) {
      return;
    }
    users = users.stream().filter(userInfoVo -> Boolean.TRUE.equals(userInfoVo.getPrimaryFlag())).collect(Collectors.toList());
    if (CollectionUtils.isEmpty(users)) {
      return;
    }
    List<String> userPositionCodes = users.stream().map(UserInfoVo::getPositionCode).collect(Collectors.toList());
    if (CollectionUtils.isEmpty(userPositionCodes)) {
      return;
    }
    List<PositionVo> positions = this.positionVoService.findByIdsOrCodes(Lists.newLinkedList(), userPositionCodes);
    if (CollectionUtils.isEmpty(positions)) {
      return;
    }
    Map<String, PositionVo> positionMap = positions.stream().collect(Collectors.toMap(PositionVo::getPositionCode, t -> t, (a, b) -> b));
    Map<String, List<RuleExecuteUserModel>> positionUserMap = users.stream().filter(userInfoVo -> Objects.nonNull(positionMap.get(userInfoVo.getPositionCode())))
        .collect(Collectors.groupingBy(UserInfoVo::getPositionCode
            , Collectors.mapping(o -> {
              PositionVo positionVo = positionMap.get(o.getPositionCode());
              RuleExecuteUserModel userModel = new RuleExecuteUserModel();
              userModel.setUserName(o.getUserName());
              userModel.setPositionCode(positionVo.getPositionCode());
              userModel.setPositionLevelCode(positionVo.getPositionLevelCode());
              return userModel;
            }, Collectors.toList())));
    orgPositions.stream().filter(orgPositionVo -> !CollectionUtils.isEmpty(positionUserMap.get(orgPositionVo.getPositionCode())))
        .forEach(orgPositionVo -> {
          List<RuleExecuteUserModel> positionUserModels = positionUserMap.get(orgPositionVo.getPositionCode());
          List<RuleExecuteUserModel> orgUserModels = orgUserMap.get(orgPositionVo.getOrgCode());
          if (CollectionUtils.isEmpty(orgUserModels)) {
            orgUserModels = Lists.newArrayList();
          }
          orgUserModels.addAll(positionUserModels);
          orgUserMap.put(orgPositionVo.getOrgCode(), orgUserModels);
        });
  }

  /**
   * 构建封装执行model 规则-用户信息
   *
   * @param executeModel 规则执行model
   */
  private void buildModelRuleUserInfo(RuleExecuteModel executeModel) {
    Map<String, List<RuleExecuteUserModel>> ruleUserMap = Maps.newHashMap();
    if (!executeModel.getOrgUserMap().isEmpty()) {
      Set<String> userNames = Sets.newHashSet();
      executeModel.getRuleOrgMap().forEach((ruleId, ordCodes) -> {
        for (String orgCode : ordCodes) {
          buildRuleOrgMap(executeModel, ruleUserMap, userNames, ruleId, orgCode);
        }
      });
    }
    executeModel.setRuleUserMap(ruleUserMap);
  }

  /**
   * 构建封装考勤规则用户映射Map
   *
   * @param executeModel 规则执行model
   * @param ruleUserMap  考勤规则用户映射Map
   * @param userNames    已执行规则用户集合
   * @param ruleId       规则ID
   * @param orgCode      组织编码
   */
  private void buildRuleOrgMap(RuleExecuteModel executeModel, Map<String, List<RuleExecuteUserModel>> ruleUserMap
      , Set<String> userNames, String ruleId, String orgCode) {
    List<RuleExecuteUserModel> userModels = executeModel.getOrgUserMap().get(orgCode);
    if (CollectionUtils.isEmpty(userModels)) {
      return;
    }
    //优先执行组织下面的职位级别的用户,如果没有则执行整个组织的用户
    List<RuleExecuteUserModel> executeUsers = userModels.stream().filter(userModel
        -> {
      Set<String> positionCodes = executeModel.getRulePositionLevelMap().get(ruleId);
      return !CollectionUtils.isEmpty(positionCodes) && positionCodes.contains(userModel.getPositionLevelCode());
    }).collect(Collectors.toList());
    if (CollectionUtils.isEmpty(executeUsers)) {
      //排除已经添加了规则的用户
      executeUsers = userModels.stream().filter(model -> !userNames.contains(model.getUserName())).collect(Collectors.toList());
    }
    if (CollectionUtils.isEmpty(executeUsers)) {
      return;
    }
    List<RuleExecuteUserModel> mapUserModelList = ruleUserMap.get(ruleId);
    if (CollectionUtils.isEmpty(mapUserModelList)) {
      mapUserModelList = Lists.newArrayList();
      ruleUserMap.put(ruleId, mapUserModelList);
    }
    mapUserModelList.addAll(executeUsers);
    userNames.addAll(executeUsers.stream().map(RuleExecuteUserModel::getUserName).collect(Collectors.toSet()));
  }
}
