package com.biz.crm.sfa.business.attendance.local.service.internal;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.LoginUserDetails;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.user.sdk.service.UserFeignVoService;
import com.biz.crm.mdm.business.user.sdk.vo.UserVo;
import com.biz.crm.sfa.business.attendance.local.entity.AttendanceFillApplyEntity;
import com.biz.crm.sfa.business.attendance.local.entity.AttendanceRecordEntity;
import com.biz.crm.sfa.business.attendance.local.model.FillApplyConditionModel;
import com.biz.crm.sfa.business.attendance.local.repository.AttendanceFillApplyRepository;
import com.biz.crm.sfa.business.attendance.local.repository.AttendanceRecordRepository;
import com.biz.crm.sfa.business.attendance.local.service.AttendanceFillApplyPictureService;
import com.biz.crm.sfa.business.attendance.local.service.AttendanceFillApplyService;
import com.biz.crm.sfa.business.attendance.sdk.constant.AttendanceConstant;
import com.biz.crm.sfa.business.attendance.sdk.dto.AttendanceFillClockDto;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceClockStatus;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceClockType;
import com.biz.crm.business.common.sdk.utils.DateTimeValidateUtil;
import com.biz.crm.workflow.sdk.constant.enums.ActApproveStatusEnum;
import com.biz.crm.workflow.sdk.dto.StartProcessDto;
import com.biz.crm.workflow.sdk.listener.ProcessListener;
import com.biz.crm.workflow.sdk.vo.response.CommitWorkflowResponse;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.util.Collections;
import java.util.Map;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 补打考勤申请表服务实现类
 *
 * @author ning.zhang
 * @date 2022-05-25 10:40:09
 */
@Slf4j
@Service("attendanceFillApplyService")
public class AttendanceFillApplyServiceImpl implements AttendanceFillApplyService {

  @Autowired
  private AttendanceFillApplyRepository attendanceFillApplyRepository;
  @Autowired
  private LoginUserService loginUserService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private AttendanceRecordRepository attendanceRecordRepository;
  @Autowired
  private AttendanceFillApplyPictureService attendanceFillApplyPictureService;
  @Autowired
  private GenerateCodeService generateCodeService;
  @Autowired
  private NebulaNetEventClient nebulaNetEventClient;
  @Autowired
  private UserFeignVoService userFeignVoService;

  @Override
  @Transactional
  public AttendanceFillApplyEntity create(AttendanceFillClockDto dto) {
    this.createValidation(dto);
    this.validateRecord(dto);
    AttendanceFillApplyEntity entity = this.nebulaToolkitService
        .copyObjectByWhiteList(dto, AttendanceFillApplyEntity.class, HashSet.class, ArrayList.class, "recordIds");
    entity.setRecordIds(String.join(",", dto.getRecordIds()));
    entity.setApplyCode(this.generateCodeService.generateCode("FA", 1).get(0));
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    entity.setProcessStatus(ActApproveStatusEnum.APPROVING.getCode());
    //保存岗位 组织
    List<UserVo> userVoList = this.userFeignVoService.findByUserNames(Collections.singletonList(dto.getUserName()));
    if (!CollectionUtils.isEmpty(userVoList)) {
      Map<String, UserVo> userMap = userVoList.stream().collect(Collectors.toMap(UserVo::getUserName, t -> t, (a, b) -> b));
      UserVo userVo = userMap.get(dto.getUserName());
      entity.setUserRealName(userVo.getFullName());
      entity.setOrgCode(userVo.getOrgCode());
      entity.setOrgName(userVo.getOrgName());
      entity.setPositionCode(userVo.getPositionCode());
      entity.setPositionName(userVo.getPositionName());
    }
    this.attendanceFillApplyRepository.save(entity);
    dto.setId(entity.getId());
    dto.setApplyCode(entity.getApplyCode());
    this.attendanceFillApplyPictureService.update(dto);
    entity.setProcessNumber(this.commitProcess(dto));
    this.attendanceFillApplyRepository.updateById(entity);
    return entity;
  }

  @Override
  @Transactional
  public AttendanceFillApplyEntity update(AttendanceFillClockDto dto) {
    this.updateValidation(dto);
    this.validateRecord(dto);
    AttendanceFillApplyEntity applyEntity = this.attendanceFillApplyRepository.getById(dto.getId());
    Validate.notNull(applyEntity, "补打卡申请记录不存在!");
    Validate.isTrue(ActApproveStatusEnum.REJECTED.getCode().equals(applyEntity.getProcessStatus())
        || ActApproveStatusEnum.INTERRUPT.getCode().equals(applyEntity.getProcessStatus()), "该申请不支持编辑");
    //驳回状态直接创建新的申请,追回的使用原来的申请继续提交
    if (ActApproveStatusEnum.REJECTED.getCode().equals(applyEntity.getProcessStatus())) {
      dto.setId(null);
    }
    AttendanceFillApplyEntity entity = this.nebulaToolkitService
        .copyObjectByWhiteList(dto, AttendanceFillApplyEntity.class, HashSet.class, ArrayList.class, "recordIds");
    entity.setApplyCode(StringUtils.isNotBlank(entity.getId()) ? applyEntity.getApplyCode() : this.generateCodeService.generateCode("FA", 1).get(0));
    entity.setRecordIds(String.join(",", dto.getRecordIds()));
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    entity.setProcessStatus(ActApproveStatusEnum.APPROVING.getCode());
    //保存岗位 组织
    List<UserVo> userVoList = this.userFeignVoService.findByUserNames(Collections.singletonList(dto.getUserName()));
    if (!CollectionUtils.isEmpty(userVoList)) {
      Map<String, UserVo> userMap = userVoList.stream().collect(Collectors.toMap(UserVo::getUserName, t -> t, (a, b) -> b));
      UserVo userVo = userMap.get(dto.getUserName());
      entity.setUserRealName(userVo.getFullName());
      entity.setOrgCode(userVo.getOrgCode());
      entity.setOrgName(userVo.getOrgName());
      entity.setPositionCode(userVo.getPositionCode());
      entity.setPositionName(userVo.getPositionName());
    }
    this.attendanceFillApplyRepository.saveOrUpdate(entity);
    dto.setId(entity.getId());
    dto.setApplyCode(entity.getApplyCode());
    this.attendanceFillApplyPictureService.update(dto);
    entity.setProcessNumber(this.commitProcess(dto));
    this.attendanceFillApplyRepository.updateById(entity);
    return entity;
  }

  /**
   * 补打卡申请工作流进行审批，提交成功返回流程实例ID，提交失败则抛出异常
   *
   * @param dto 补打卡申请信息
   */
  private String commitProcess(AttendanceFillClockDto dto) {
    StartProcessDto startProcessDto = new StartProcessDto();
    startProcessDto.setProcessKey(AttendanceConstant.ATTENDANCE_FILL_CLOCK_APPLY_PROCESS_KEY);
    startProcessDto.setBusinessId(dto.getId());
    startProcessDto.setBusinessNo(dto.getApplyCode());
    startProcessDto.setRemark(dto.getProcessRemark());
    startProcessDto.setFormType(AttendanceConstant.ATTENDANCE_FILL_CLOCK_APPLY_PROCESS_FORM_TYPE);
    startProcessDto.setProcessTitle(AttendanceConstant.ATTENDANCE_FILL_CLOCK_APPLY_PROCESS_NAME);
    startProcessDto.setMenuCode(dto.getCompetenceCode());
    SerializableBiConsumer<ProcessListener, StartProcessDto> sf = ProcessListener::onStartProcess;
    CommitWorkflowResponse response = (CommitWorkflowResponse) this.nebulaNetEventClient.directPublish(startProcessDto, ProcessListener.class, sf);
    Validate.isTrue(StringUtils.isNotEmpty(response.getProcessInstanceId()), "发起流程失败！");
    return response.getProcessInstanceId();
  }

  /**
   * 校验考勤记录信息
   *
   * @param dto 参数dto
   */
  private void validateRecord(AttendanceFillClockDto dto) {
    List<AttendanceRecordEntity> recordEntities = this.attendanceRecordRepository.listByIds(dto.getRecordIds());
    Validate.isTrue(!CollectionUtils.isEmpty(recordEntities)
        && recordEntities.size() == Sets.newHashSet(dto.getRecordIds()).size(), "相关打卡记录不匹配");
    recordEntities.forEach(entity -> {
      Validate.isTrue(entity.getUserName().equals(dto.getUserName()), "不能申请非自身打卡记录");
      Validate.isTrue(!AttendanceClockStatus.OK.getDictCode().equals(entity.getClockStatus()), String.format("%s(%s),不是异常打卡,无需申请"
          , AttendanceClockType.getByDictCode(entity.getClockType()).getValue(), entity.getRuleClockTime()));
      Validate.isTrue(entity.getRuleDate().compareTo(dto.getBeginTime()) >= 0
          && entity.getRuleDate().compareTo(dto.getEndTime()) <= 0, String.format("%s(%s),打卡日期与所选日期不匹配"
          , AttendanceClockType.getByDictCode(entity.getClockType()).getValue(), entity.getRuleClockTime()));
    });
    FillApplyConditionModel model = new FillApplyConditionModel();
    model.setBeginTime(dto.getBeginTime());
    model.setEndTime(dto.getEndTime());
    model.setTenantCode(dto.getTenantCode());
    model.setUserName(dto.getUserName());
    model.setExcludeProcessStatusList(Lists.newArrayList(ActApproveStatusEnum.REJECTED.getCode(), ActApproveStatusEnum.INTERRUPT.getCode()));
    List<AttendanceFillApplyEntity> applyEntities = this.attendanceFillApplyRepository.findByFillApplyConditionModel(model);
    if (CollectionUtils.isEmpty(applyEntities)) {
      return;
    }
    //排除自身之后再比较
    List<String> existsRecordIdList = applyEntities.stream().filter(entity -> !entity.getId().equals(dto.getId()))
        .collect(Collectors.toList()).stream().flatMap(attendanceFillApplyEntity
            -> Lists.newArrayList(attendanceFillApplyEntity.getRecordIds().split(",")).stream()).collect(Collectors.toList());
    existsRecordIdList.retainAll(dto.getRecordIds());
    Validate.isTrue(CollectionUtils.isEmpty(existsRecordIdList), "存在相关的补打卡申请");
  }

  /**
   * 在创建attendanceFillApply模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(AttendanceFillClockDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getBeginTime(), "缺失开始时间");
    Validate.notBlank(dto.getEndTime(), "缺失结束时间");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getRecordIds()), "缺失异常打卡记录");
    Validate.isTrue(DateTimeValidateUtil.validateDate(dto.getBeginTime()), "非法的开始时间格式，可用的格式：[yyyy-MM-dd]");
    Validate.isTrue(DateTimeValidateUtil.validateDate(dto.getEndTime()), "非法的结束时间格式，可用的格式：[yyyy-MM-dd]");
    Validate.isTrue(dto.getBeginTime().compareTo(dto.getEndTime()) <= 0, "开始时间不能大于结束时间");
    LocalDate nowDate = LocalDate.now();
    dto.setApplyDate(LocalDateTime.now().format(AttendanceConstant.YYYY_MM_DD_HH_MM_SS));
    Validate.isTrue(dto.getEndTime().compareTo(nowDate.format(AttendanceConstant.YYYY_MM_DD)) < 0, "只能报备当天之前的异常打卡");
    // 8日之前可修改上月的数据，8日之后，不可修改上月数据
    if (nowDate.getDayOfMonth() > AttendanceConstant.FILL_CLOCK_CAN_UPDATE_LAST_MONTH_DAY_NUM) {
      Validate.isTrue(!LocalDate.parse(dto.getBeginTime()).isBefore(LocalDate.of(nowDate.getYear(), nowDate.getMonth(), 1))
          , "只能报备当月异常打卡");
    } else {
      LocalDate ult = nowDate.minusMonths(1);
      Validate.isTrue(!LocalDate.parse(dto.getBeginTime()).isBefore(LocalDate.of(ult.getYear(), ult.getMonth(), 1))
          , "只能报备上月异常签到");
    }
    LoginUserDetails loginDetails = this.loginUserService.getLoginDetails(LoginUserDetails.class);
    dto.setUserName(loginDetails.getAccount());
  }

  /**
   * 在修改attendanceFillApply模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(AttendanceFillClockDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getId(), "修改信息时，id不能为空！");
    Validate.notBlank(dto.getBeginTime(), "缺失开始时间");
    Validate.notBlank(dto.getEndTime(), "缺失结束时间");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getRecordIds()), "缺失异常打卡记录");
    Validate.isTrue(DateTimeValidateUtil.validateDate(dto.getBeginTime()), "非法的开始时间格式，可用的格式：[yyyy-MM-dd]");
    Validate.isTrue(DateTimeValidateUtil.validateDate(dto.getEndTime()), "非法的结束时间格式，可用的格式：[yyyy-MM-dd]");
    Validate.isTrue(dto.getBeginTime().compareTo(dto.getEndTime()) <= 0, "开始时间不能大于结束时间");
    LocalDate nowDate = LocalDate.now();
    dto.setApplyDate(LocalDateTime.now().format(AttendanceConstant.YYYY_MM_DD_HH_MM_SS));
    Validate.isTrue(dto.getEndTime().compareTo(dto.getApplyDate()) < 0, "只能报备当天之前的异常打卡");
    // 8日之前可修改上月的数据，8日之后，不可修改上月数据
    if (nowDate.getDayOfMonth() > AttendanceConstant.FILL_CLOCK_CAN_UPDATE_LAST_MONTH_DAY_NUM) {
      Validate.isTrue(!LocalDate.parse(dto.getBeginTime()).isBefore(LocalDate.of(nowDate.getYear(), nowDate.getMonth(), 1))
          , "只能报备当月异常打卡");
    } else {
      LocalDate ult = nowDate.minusMonths(1);
      Validate.isTrue(!LocalDate.parse(dto.getBeginTime()).isBefore(LocalDate.of(ult.getYear(), ult.getMonth(), 1))
          , "只能报备上月异常签到");
    }
    LoginUserDetails loginDetails = this.loginUserService.getLoginDetails(LoginUserDetails.class);
    dto.setUserName(loginDetails.getAccount());
  }
}
