package com.biz.crm.sfa.business.attendance.local.service.internal;

import com.alibaba.fastjson.JSON;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.sfa.business.attendance.local.entity.AttendanceRecordRuleEntity;
import com.biz.crm.sfa.business.attendance.local.entity.AttendanceRecordRulePlaceEntity;
import com.biz.crm.sfa.business.attendance.local.entity.AttendanceRuleEntity;
import com.biz.crm.sfa.business.attendance.local.entity.AttendanceRuleSpecialDateEntity;
import com.biz.crm.sfa.business.attendance.local.model.RuleExecuteModel;
import com.biz.crm.sfa.business.attendance.local.repository.AttendanceRecordRepository;
import com.biz.crm.sfa.business.attendance.local.repository.AttendanceRecordRulePlaceRepository;
import com.biz.crm.sfa.business.attendance.local.repository.AttendanceRecordRuleRepository;
import com.biz.crm.sfa.business.attendance.local.service.AttendanceRecordRuleService;
import com.biz.crm.sfa.business.attendance.local.service.AttendanceRecordService;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceSignOrNonType;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceSpecialDateType;
import com.biz.crm.sfa.business.attendance.sdk.service.AttendanceRuleVoService;
import com.biz.crm.sfa.business.attendance.sdk.vo.AttendanceRuleVo;
import com.biz.crm.sfa.business.holiday.sdk.service.HolidayVoService;
import com.biz.crm.sfa.business.holiday.sdk.vo.HolidayVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 考勤记录规则表服务实现类
 *
 * @author ning.zhang
 * @date 2022-05-17 09:52:40
 */
@Slf4j
@Service("attendanceRecordRuleService")
public class AttendanceRecordRuleServiceImpl implements AttendanceRecordRuleService {

  @Autowired
  private AttendanceRecordRuleRepository attendanceRecordRuleRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private HolidayVoService holidayVoService;
  @Autowired
  private AttendanceRecordService attendanceRecordService;
  @Autowired
  private AttendanceRuleVoService attendanceRuleVoService;
  @Autowired
  private AttendanceRecordRepository attendanceRecordRepository;
  @Autowired
  private AttendanceRecordRulePlaceRepository attendanceRecordRulePlaceRepository;

  @Override
  @Transactional
  public void create(RuleExecuteModel executeModel) {
    Validate.notNull(executeModel, "缺失规则执行信息");
    Validate.isTrue(!CollectionUtils.isEmpty(executeModel.getRuleMap()), "缺失考勤规则信息");
    String executeDate = StringUtils.isNotBlank(executeModel.getExecuteDate())
        ? executeModel.getExecuteDate() : LocalDate.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd"));
    HolidayVo holidayVo = holidayVoService.findByDate(executeModel.getExecuteDate());
    //删除未使用的考勤记录
    this.attendanceRecordRepository.deleteByRuleDateAndUsedStatus(TenantUtils.getTenantCode()
        , executeDate, BooleanEnum.FALSE.getCapital());
    executeModel.getRuleMap().keySet().forEach(ruleId -> {
      AttendanceRuleEntity ruleEntity = executeModel.getRuleMap().get(ruleId);
      if (CollectionUtils.isEmpty(executeModel.getRuleUserMap().get(ruleId))) {
        log.info(String.format("规则[%s],没有对应的规则执行用户",ruleId));
        return;
      }
      AttendanceRuleVo attendanceRuleVo = this.attendanceRuleVoService.findById(ruleId);
      Validate.isTrue(Objects.nonNull(attendanceRuleVo) && Objects.nonNull(ruleEntity), "缺失考勤规则信息");
      AttendanceRecordRuleEntity recordRuleEntity = this.nebulaToolkitService.copyObjectByBlankList(ruleEntity, AttendanceRecordRuleEntity.class, HashSet.class, ArrayList.class);
      recordRuleEntity.setId(null);
      recordRuleEntity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
      recordRuleEntity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
      recordRuleEntity.setRuleId(ruleEntity.getId());
      recordRuleEntity.setRuleCode(ruleEntity.getRuleCode());
      recordRuleEntity.setRuleDate(executeDate);
      recordRuleEntity.setRuleYear(executeDate.substring(0, 4));
      recordRuleEntity.setRuleMonth(executeDate.substring(6, 7));
      recordRuleEntity.setRuleYearMonth(executeDate.substring(0, 7));
      recordRuleEntity.setNoWorkAbideInfo(CollectionUtils.isEmpty(attendanceRuleVo.getNoWorkAbideInfoList())
          ? null : JSON.toJSONString(attendanceRuleVo.getNoWorkAbideInfoList()));
      this.buildSignInfo(holidayVo, ruleEntity, recordRuleEntity);
      this.attendanceRecordRuleRepository.save(recordRuleEntity);
      //保存考勤记录规则当前地点信息
      if (!CollectionUtils.isEmpty(attendanceRuleVo.getPlaceList())) {
        List<AttendanceRecordRulePlaceEntity> placeEntities = attendanceRuleVo.getPlaceList().stream().map(placeVo -> {
          AttendanceRecordRulePlaceEntity recordRulePlaceEntity = this.nebulaToolkitService
              .copyObjectByBlankList(placeVo, AttendanceRecordRulePlaceEntity.class, HashSet.class, ArrayList.class);
          recordRulePlaceEntity.setRecordRuleId(recordRuleEntity.getId());
          recordRulePlaceEntity.setId(null);
          return recordRulePlaceEntity;
        }).collect(Collectors.toList());
        this.attendanceRecordRulePlaceRepository.saveBatch(placeEntities);
      }
      this.attendanceRecordService.update(executeModel, recordRuleEntity);
    });
  }

  @Override
  @Transactional
  public void deleteByRuleIdsAndRuleDate(List<String> ruleIds, String ruleDate) {
    Validate.isTrue(!CollectionUtils.isEmpty(ruleIds), "缺失考勤规则ID");
    Validate.notBlank(ruleDate, "缺失考勤规则日期");
    String tenantCode = TenantUtils.getTenantCode();
    this.attendanceRecordRuleRepository.deleteByRuleIdsAndRuleDate(tenantCode, ruleIds, ruleDate);
    this.attendanceRecordRepository.deleteByRuleIdsAndRuleDate(tenantCode, ruleIds, ruleDate);
  }

  /**
   * 构建打卡信息
   *
   * @param holidayVo        节假日信息
   * @param ruleEntity       规则信息
   * @param recordRuleEntity 记录规则信息
   */
  private void buildSignInfo(HolidayVo holidayVo, AttendanceRuleEntity ruleEntity, AttendanceRecordRuleEntity recordRuleEntity) {
    List<AttendanceRuleSpecialDateEntity> specialDateList = ruleEntity.getSpecialDateList();
    // 节假日
    if (Objects.nonNull(holidayVo) && BooleanEnum.TRUE.getCapital().equals(ruleEntity.getSyncHoliday())) {
      recordRuleEntity.setSignMust(BooleanEnum.FALSE.getCapital());
      recordRuleEntity.setSignOrNonType(AttendanceSignOrNonType.HOLIDAY_NO_SIGN.getDictCode());
      return;
    }
    //特殊日期判断
    if (!CollectionUtils.isEmpty(specialDateList)) {
      specialDateList = specialDateList.stream().filter(entity -> recordRuleEntity.getRuleDate().equals(entity.getSpecialDate())).collect(Collectors.toList());
      if (!CollectionUtils.isEmpty(specialDateList)) {
        AttendanceRuleSpecialDateEntity specialDateEntity = specialDateList.get(0);
        if (AttendanceSpecialDateType.MUST_CLOCK.getDictCode().equals(specialDateEntity.getSpecialDateType())) {
          recordRuleEntity.setSignMust(BooleanEnum.TRUE.getCapital());
          recordRuleEntity.setSignOrNonType(AttendanceSignOrNonType.SPECIAL_DAY_SIGN.getDictCode());
          return;
        } else {
          recordRuleEntity.setSignOrNonType(AttendanceSignOrNonType.SPECIAL_DAY_NO_SIGN.getDictCode());
        }
      }
    }
    // 工作日判断
    int weekNo = LocalDate.now().getDayOfWeek().getValue();
    if (ruleEntity.getWorkingDay().contains(String.valueOf(weekNo))) {
      recordRuleEntity.setSignMust(BooleanEnum.TRUE.getCapital());
      if (StringUtils.isEmpty(recordRuleEntity.getSignOrNonType())) {
        recordRuleEntity.setSignOrNonType(AttendanceSignOrNonType.WORKDAY_SIGN.getDictCode());
      }
    } else {
      recordRuleEntity.setSignMust(BooleanEnum.FALSE.getCapital());
      if (StringUtils.isEmpty(recordRuleEntity.getSignOrNonType())) {
        recordRuleEntity.setSignOrNonType(AttendanceSignOrNonType.WORK_DAY_NO_SIGN.getDictCode());
      }
    }
  }
}
