package com.biz.crm.sfa.business.attendance.local.service.internal;

import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.sfa.business.attendance.local.entity.AttendanceRuleEntity;
import com.biz.crm.sfa.business.attendance.local.entity.AttendanceRuleScopeEntity;
import com.biz.crm.sfa.business.attendance.sdk.dto.RuleConditionDto;
import com.biz.crm.sfa.business.attendance.local.model.RuleScopeConditionModel;
import com.biz.crm.sfa.business.attendance.local.repository.AttendanceRuleRepository;
import com.biz.crm.sfa.business.attendance.local.repository.AttendanceRuleScopeRepository;
import com.biz.crm.sfa.business.attendance.local.service.AttendanceRuleScopeService;
import com.biz.crm.sfa.business.attendance.sdk.dto.AttendanceRuleDto;
import com.biz.crm.sfa.business.attendance.sdk.dto.AttendanceRuleScopeDto;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceRuleScopeType;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 考勤规则范围表服务实现类
 *
 * @author ning.zhang
 * @date 2022-05-05 09:44:29
 */
@Slf4j
@Service("attendanceRuleScopeService")
public class AttendanceRuleScopeServiceImpl implements AttendanceRuleScopeService {

  @Autowired
  private AttendanceRuleScopeRepository attendanceRuleScopeRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private AttendanceRuleRepository attendanceRuleRepository;

  @Override
  @Transactional
  public void update(AttendanceRuleDto dto) {
    Validate.notBlank(dto.getId(), "考勤规则ID不能为空");
    this.attendanceRuleScopeRepository.deleteByRuleId(dto.getId());
    if (CollectionUtils.isEmpty(dto.getScopeList())) {
      return;
    }
    this.updateValidation(dto);
    List<AttendanceRuleScopeEntity> entities = dto.getScopeList().stream().map(scopeDto -> {
      AttendanceRuleScopeEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(scopeDto, AttendanceRuleScopeEntity.class, HashSet.class, ArrayList.class);
      entity.setRuleId(dto.getId());
      entity.setContainsChild(BooleanEnum.TRUE.getCapital());
      return entity;
    }).collect(Collectors.toList());
    this.attendanceRuleScopeRepository.saveBatch(entities);
  }

  @Override
  public void validateScope(AttendanceRuleDto dto) {
    if (Objects.isNull(dto) || CollectionUtils.isEmpty(dto.getScopeList())) {
      return;
    }
    List<String> orgCodes = Lists.newArrayList();
    List<String> positionLevelCodes = Lists.newArrayList();
    dto.getScopeList().forEach(scopeVo -> {
      if (AttendanceRuleScopeType.ORG.getDictCode().equals(scopeVo.getScopeType())) {
        orgCodes.add(scopeVo.getScopeCode());
      }
      if (AttendanceRuleScopeType.POSITION_LEVEL.getDictCode().equals(scopeVo.getScopeType())) {
        positionLevelCodes.add(scopeVo.getScopeCode());
      }
    });
    RuleConditionDto conditionDto = new RuleConditionDto();
    conditionDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    conditionDto.setTenantCode(TenantUtils.getTenantCode());
    conditionDto.setExcludeIds(Lists.newArrayList(dto.getId()));
    conditionDto.setOrgCodes(orgCodes);
    List<AttendanceRuleEntity> sameOrgRules = this.attendanceRuleRepository.findByRuleConditionDto(conditionDto);
    //为空说明传入的组织没有冲突的规则，直接跳过下面关于组织下的职位校验
    if (CollectionUtils.isEmpty(sameOrgRules)) {
      return;
    }
    List<String> ruleCodes = sameOrgRules.stream().map(AttendanceRuleEntity::getRuleCode).collect(Collectors.toList());
    RuleScopeConditionModel ruleScopeModel = new RuleScopeConditionModel();
    ruleScopeModel.setRuleCodes(ruleCodes);
    ruleScopeModel.setScopeType(AttendanceRuleScopeType.POSITION_LEVEL.getDictCode());
    ruleScopeModel.setTenantCode(TenantUtils.getTenantCode());
    List<AttendanceRuleScopeEntity> scopeEntities = this.attendanceRuleScopeRepository.findByRuleScopeConditionModel(ruleScopeModel);
    //校验组织相同且职位级别重复的规则
    if (!CollectionUtils.isEmpty(positionLevelCodes)) {
      Map<String, AttendanceRuleScopeEntity> scopeEntityMap = scopeEntities.stream().collect(Collectors.toMap(AttendanceRuleScopeEntity::getScopeCode, t -> t, (a, b) -> b));
      positionLevelCodes.forEach(positionLevelCode -> {
        AttendanceRuleScopeEntity scopeEntity = scopeEntityMap.get(positionLevelCode);
        Validate.isTrue(Objects.isNull(scopeEntity), String.format("职位重复添加,冲突规则编码:%s", Objects.isNull(scopeEntity) ? "" : scopeEntity.getRuleCode()));
      });
      //组织下面没有职位级别,则校验同样没有职位级别,组织相同且工作日有交集的冲突规则
    } else {
      Map<String, AttendanceRuleScopeEntity> scopeEntityMap = scopeEntities.stream().collect(Collectors.toMap(AttendanceRuleScopeEntity::getRuleId, t -> t, (a, b) -> b));
      sameOrgRules.forEach(ruleEntity -> {
        AttendanceRuleScopeEntity scopeEntity = scopeEntityMap.get(ruleEntity.getRuleCode());
        if (Objects.nonNull(scopeEntity)) {
          return;
        }
        List<String> inWorkingDays = Lists.newArrayList(dto.getWorkingDay().split(","));
        List<String> vaWorkingDays = Lists.newArrayList(ruleEntity.getWorkingDay().split(","));
        inWorkingDays.retainAll(vaWorkingDays);
        Validate.isTrue(CollectionUtils.isEmpty(inWorkingDays), String.format("组织重复添加,冲突规则编码:%s",ruleEntity.getRuleCode()));
      });
    }
  }

  /**
   * 在修改attendanceRuleScope模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(AttendanceRuleDto dto) {
    Boolean haveOrg = Boolean.FALSE;
    for (AttendanceRuleScopeDto scopeDto : dto.getScopeList()) {
      Validate.notBlank(scopeDto.getScopeCode(), "缺失范围编码");
      Validate.notBlank(scopeDto.getScopeType(), "缺失范围类型");
      if (AttendanceRuleScopeType.ORG.getDictCode().equals(scopeDto.getScopeType())) {
        haveOrg = Boolean.TRUE;
      }
    }
    Validate.isTrue(haveOrg, "缺失打卡组织");
    this.validateScope(dto);
  }
}
