package com.biz.crm.sfa.business.attendance.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.sfa.business.attendance.local.entity.AttendanceRecordRuleEntity;
import com.biz.crm.sfa.business.attendance.local.entity.AttendanceRuleEntity;
import com.biz.crm.sfa.business.attendance.local.model.RuleExecuteModel;
import com.biz.crm.sfa.business.attendance.local.repository.AttendanceRecordRuleRepository;
import com.biz.crm.sfa.business.attendance.local.repository.AttendanceRuleRepository;
import com.biz.crm.sfa.business.attendance.local.service.AttendanceRecordRuleService;
import com.biz.crm.sfa.business.attendance.local.service.AttendanceRulePlaceService;
import com.biz.crm.sfa.business.attendance.local.service.AttendanceRuleScopeService;
import com.biz.crm.sfa.business.attendance.local.service.AttendanceRuleService;
import com.biz.crm.sfa.business.attendance.local.service.AttendanceRuleSpecialDateService;
import com.biz.crm.sfa.business.attendance.local.service.AttendanceRuleTimeService;
import com.biz.crm.sfa.business.attendance.local.service.helper.AttendanceRuleServiceHelper;
import com.biz.crm.sfa.business.attendance.sdk.constant.AttendanceConstant;
import com.biz.crm.sfa.business.attendance.sdk.dto.AttendanceRuleDto;
import com.biz.crm.sfa.business.attendance.sdk.dto.AttendanceRuleExecuteDto;
import com.biz.crm.sfa.business.attendance.sdk.dto.AttendanceRulePageDto;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceElectronFenceType;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceRuleEffectiveType;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceRuleType;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceSpecialDateType;
import com.biz.crm.sfa.business.attendance.sdk.event.AttendanceRuleEventListener;
import com.biz.crm.sfa.business.attendance.sdk.vo.AttendanceRuleVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 考勤规则表服务实现类
 *
 * @author ning.zhang
 * @date 2022-05-05 09:43:07
 */
@Slf4j
@Service("attendanceRuleService")
public class AttendanceRuleServiceImpl implements AttendanceRuleService {

  @Autowired(required = false)
  private List<AttendanceRuleEventListener> listeners;
  @Autowired
  private AttendanceRuleRepository attendanceRuleRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private GenerateCodeService generateCodeService;
  @Autowired
  private AttendanceRulePlaceService attendanceRulePlaceService;
  @Autowired
  private AttendanceRuleScopeService attendanceRuleScopeService;
  @Autowired
  private AttendanceRuleTimeService attendanceRuleTimeService;
  @Autowired
  private AttendanceRuleSpecialDateService attendanceRuleSpecialDateService;
  @Autowired
  private AttendanceRuleServiceHelper attendanceRuleServiceHelper;
  @Autowired
  private AttendanceRecordRuleService attendanceRecordRuleService;
  @Autowired
  private AttendanceRecordRuleRepository attendanceRecordRuleRepository;

  @Override
  @Transactional
  public AttendanceRuleEntity create(AttendanceRuleDto dto) {
    this.createValidation(dto);
    AttendanceRuleEntity entity = this.nebulaToolkitService.copyObjectByBlankList(dto, AttendanceRuleEntity.class, HashSet.class, ArrayList.class);
    entity.setRuleEffective(AttendanceRuleEffectiveType.TOMORROW.getDictCode());
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    entity.setRuleCode(this.generateCodeService.generateCode("AR", 1).get(0));
    this.attendanceRuleRepository.save(entity);
    //绑定关联数据
    dto.setId(entity.getId());
    this.bindRelationData(dto);
    //新增考勤规则事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      AttendanceRuleVo vo = this.nebulaToolkitService.copyObjectByBlankList(dto, AttendanceRuleVo.class, HashSet.class, ArrayList.class);
      vo.setRuleCode(entity.getRuleCode());
      listeners.forEach(listener -> {
        listener.onCreate(vo);
      });
    }
    return entity;
  }

  @Override
  @Transactional
  public AttendanceRuleEntity update(AttendanceRuleDto dto) {
    this.updateValidation(dto);
    List<AttendanceRuleEntity> entities = this.attendanceRuleRepository.findByIds(Lists.newArrayList(dto.getId()));
    Validate.isTrue(!CollectionUtils.isEmpty(entities), "考勤规则信息不存在");
    AttendanceRuleEntity updateEntity = this.nebulaToolkitService.copyObjectByBlankList(dto, AttendanceRuleEntity.class, HashSet.class, ArrayList.class);
    this.attendanceRuleRepository.updateById(updateEntity);
    //绑定关联数据
    this.bindRelationData(dto);
    //更新考勤规则事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      AttendanceRuleVo oldVo = this.nebulaToolkitService.copyObjectByBlankList(entities.get(0), AttendanceRuleVo.class, HashSet.class, ArrayList.class);
      AttendanceRuleVo newVo = this.nebulaToolkitService.copyObjectByBlankList(dto, AttendanceRuleVo.class, HashSet.class, ArrayList.class);
      newVo.setRuleCode(oldVo.getRuleCode());
      listeners.forEach(listener -> {
        listener.onUpdate(oldVo, newVo);
      });
    }
    return updateEntity;
  }

  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    List<AttendanceRuleEntity> entities = this.attendanceRuleRepository.listByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据删除个数不匹配");
    this.attendanceRuleRepository.updateDelFlagByIds(ids);
    Map<String, AttendanceRuleEntity> ruleEntityMap = entities.stream().collect(Collectors.toMap(AttendanceRuleEntity::getId, t -> t, (a, b) -> b));
    List<AttendanceRecordRuleEntity> recordRuleEntities = this.attendanceRecordRuleRepository.findByRuleIds(TenantUtils.getTenantCode(), ids);
    if (!CollectionUtils.isEmpty(recordRuleEntities)) {
      recordRuleEntities.forEach(entity -> {
        AttendanceRuleEntity ruleEntity = ruleEntityMap.get(entity.getRuleId());
        throw new IllegalArgumentException(String.format("规则[%s]已生成考勤数据,无法删除",ruleEntity.getRuleCode()));
      });
    }
    //删除考勤规则事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      List<AttendanceRuleVo> voList = (List<AttendanceRuleVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, AttendanceRuleEntity.class
          , AttendanceRuleVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(listener -> {
        listener.onDelete(voList);
      });
    }
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    List<AttendanceRuleEntity> entities = this.attendanceRuleRepository.findByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据启用个数不匹配");
    this.attendanceRuleRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
    entities.forEach(ruleEntity -> {
      AttendanceRuleDto ruleDto = this.nebulaToolkitService.copyObjectByBlankList(ruleEntity, AttendanceRuleDto.class, HashSet.class, ArrayList.class);
      this.attendanceRuleScopeService.validateScope(ruleDto);
    });
    //启用考勤规则事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      List<AttendanceRuleVo> voList = (List<AttendanceRuleVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, AttendanceRuleEntity.class
          , AttendanceRuleVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(listener -> {
        listener.onEnable(voList);
      });
    }
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    List<AttendanceRuleEntity> entities = this.attendanceRuleRepository.listByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据禁用个数不匹配");
    this.attendanceRuleRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
    //这些规则被禁用后需要删除当天已经生成的对应规则的打卡明细和规则明细
    this.attendanceRecordRuleService.deleteByRuleIdsAndRuleDate(ids, LocalDate.now().format(AttendanceConstant.YYYY_MM_DD));
    //启用考勤规则事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      List<AttendanceRuleVo> voList = (List<AttendanceRuleVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, AttendanceRuleEntity.class
          , AttendanceRuleVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(listener -> {
        listener.onDisable(voList);
      });
    }
  }

  @Override
  public Page<AttendanceRuleEntity> findByConditions(Pageable pageable, AttendanceRulePageDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new AttendanceRulePageDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    return this.attendanceRuleRepository.findByConditions(pageable, dto);
  }

  @Override
  @Transactional
  public void execute(AttendanceRuleExecuteDto dto) {
    RuleExecuteModel executeModel = this.attendanceRuleServiceHelper.buildRuleExecuteModel(dto);
    executeModel.setExecuteDate(StringUtils.isNotBlank(dto.getExecuteDate())
        ? dto.getExecuteDate() : LocalDate.now().format(AttendanceConstant.YYYY_MM_DD));
    this.attendanceRecordRuleService.create(executeModel);
  }

  @Override
  @DynamicTaskService(cornExpression = "0 0 0 * * ?", taskDesc = "考勤规则执行定时任务")
  public void executeTask() {
    AttendanceRuleExecuteDto dto = new AttendanceRuleExecuteDto();
    dto.setExecuteDate(LocalDate.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd")));
    RuleExecuteModel executeModel = this.attendanceRuleServiceHelper.buildRuleExecuteModel(dto);
    executeModel.setExecuteDate(dto.getExecuteDate());
    this.attendanceRecordRuleService.create(executeModel);
  }

  /**
   * 绑定关联数据
   *
   * @param dto 考勤规则信息dto
   */
  private void bindRelationData(AttendanceRuleDto dto) {
    this.attendanceRulePlaceService.update(dto);
    this.attendanceRuleScopeService.update(dto);
    this.attendanceRuleTimeService.update(dto);
    this.attendanceRuleSpecialDateService.update(dto);
  }

  /**
   * 在创建attendanceRule模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(AttendanceRuleDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getRuleName(), "缺失规则名称");
    Validate.notBlank(dto.getRuleType(), "缺失规则类型");
    Validate.notBlank(dto.getWorkingDay(), "缺失工作日");
    Validate.notBlank(dto.getSyncHoliday(), "缺失是否同步节假日");
    Validate.notBlank(dto.getClockPhotograph(), "缺失打卡是否需要拍照");
    Validate.notBlank(dto.getOffWorkClockType(), "缺失下班打卡类型");
    if (AttendanceRuleType.STATIC.getDictCode().equals(dto.getRuleType())) {
      Validate.isTrue(!CollectionUtils.isEmpty(dto.getTimeList()), "缺失考勤时间");
    }
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getScopeList()), "缺失考勤范围");
    if (!CollectionUtils.isEmpty(dto.getPlaceList())) {
      Validate.notBlank(dto.getElectronFenceType(), "缺失打卡范围类型");
    } else {
      dto.setElectronFenceType(AttendanceElectronFenceType.NONE.getDictCode());
    }
    Validate.isTrue(dto.getRuleName().length() <= 100, "规则名称，在进行编辑时填入值超过了限定长度(100)，请检查!");
    AttendanceRuleEntity ruleEntity = this.attendanceRuleRepository.findByRuleName(dto.getTenantCode(), dto.getRuleName());
    Validate.isTrue(Objects.isNull(ruleEntity), String.format("已存在考勤规则[%s]", dto.getRuleName()));
  }

  /**
   * 在修改attendanceRule模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(AttendanceRuleDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getId(), "修改信息时，id不能为空！");
    Validate.notBlank(dto.getRuleName(), "缺失规则名称");
    Validate.notBlank(dto.getRuleType(), "缺失规则类型");
    Validate.notBlank(dto.getWorkingDay(), "缺失工作日");
    Validate.notBlank(dto.getSyncHoliday(), "缺失是否同步节假日");
    Validate.notBlank(dto.getClockPhotograph(), "缺失打卡是否需要拍照");
    Validate.notBlank(dto.getOffWorkClockType(), "缺失下班打卡类型");

    Validate.isTrue(!CollectionUtils.isEmpty(dto.getScopeList()), "缺失考勤范围");
    if (!CollectionUtils.isEmpty(dto.getPlaceList())) {
      Validate.notBlank(dto.getElectronFenceType(), "缺失打卡范围类型");
    } else {
      dto.setElectronFenceType(AttendanceElectronFenceType.NONE.getDictCode());
    }
    Validate.isTrue(dto.getRuleName().length() <= 100, "规则名称，在进行编辑时填入值超过了限定长度(100)，请检查!");
    AttendanceRuleEntity ruleEntity = this.attendanceRuleRepository.findByRuleName(dto.getTenantCode(), dto.getRuleName());
    Validate.isTrue(Objects.isNull(ruleEntity) || ruleEntity.getRuleName().equals(dto.getRuleName()), String.format("已存在考勤规则[%s]", dto.getRuleName()));
  }
}
