package com.biz.crm.sfa.business.attendance.local.service.internal;

import com.biz.crm.sfa.business.attendance.local.entity.AttendanceRuleTimeEntity;
import com.biz.crm.sfa.business.attendance.local.repository.AttendanceRuleTimeRepository;
import com.biz.crm.sfa.business.attendance.local.service.AttendanceRuleTimeService;
import com.biz.crm.sfa.business.attendance.sdk.constant.AttendanceConstant;
import com.biz.crm.sfa.business.attendance.sdk.dto.AttendanceRuleDto;
import com.biz.crm.sfa.business.attendance.sdk.dto.AttendanceRuleTimeDto;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceRuleType;
import com.biz.crm.business.common.sdk.utils.DateTimeValidateUtil;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 考勤规则打卡时间表服务实现类
 *
 * @author ning.zhang
 * @date 2022-05-05 09:45:02
 */
@Slf4j
@Service("attendanceRuleTimeService")
public class AttendanceRuleTimeServiceImpl implements AttendanceRuleTimeService {

  @Autowired
  private AttendanceRuleTimeRepository attendanceRuleTimeRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Override
  @Transactional
  public void update(AttendanceRuleDto dto) {
    Validate.notBlank(dto.getId(), "考勤规则ID不能为空");
    this.attendanceRuleTimeRepository.deleteByRuleId(dto.getId());
    this.updateValidation(dto);
    List<AttendanceRuleTimeEntity> entities = dto.getTimeList().stream().map(timeDto -> {
      AttendanceRuleTimeEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(timeDto, AttendanceRuleTimeEntity.class, HashSet.class, ArrayList.class);
      entity.setRuleId(dto.getId());
      return entity;
    }).collect(Collectors.toList());
    this.attendanceRuleTimeRepository.saveBatch(entities);
  }

  /**
   * 在修改attendanceRuleTime模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(AttendanceRuleDto dto) {
    if (AttendanceRuleType.STATIC.getDictCode().equals(dto.getRuleType())) {
      Validate.isTrue(!CollectionUtils.isEmpty(dto.getTimeList()), "缺失考勤时间");
      String indexOffWorkTime = "00:00:00";
      String indexOffWorkClockEndTime = indexOffWorkTime;
      String indexOnWorkClockEndTime = indexOffWorkTime;
      for (AttendanceRuleTimeDto timeDto : dto.getTimeList()) {
        int timeNo = dto.getTimeList().indexOf(timeDto) + 1;
        timeDto.setTimeNo(String.valueOf(timeNo));
        Validate.notBlank(timeDto.getOnWorkTime(), String.format("第%s段打卡时间,缺失上班时间", timeNo));
        Validate.notBlank(timeDto.getOffWorkTime(), String.format("第%s段打卡时间,缺失下班时间", timeNo));
        Validate.notBlank(timeDto.getOffWorkClockEndTime(), String.format("第%s段打卡时间,缺失下班打卡结束时间", timeNo));
        Validate.notBlank(timeDto.getOffWorkClockStartTime(), String.format("第%s段打卡时间,缺失下班打卡开始时间", timeNo));
        Validate.notBlank(timeDto.getOnWorkClockEndTime(), String.format("第%s段打卡时间,缺失上班打卡结束时间", timeNo));
        Validate.notBlank(timeDto.getOnWorkClockStartTime(), String.format("第%s段打卡时间,缺失上班打卡开始时间", timeNo));
        Validate.isTrue(DateTimeValidateUtil.validateTime(timeDto.getOnWorkTime()), String.format("第%s段打卡时间,非法的上班时间格式,可用的格式:[HH:mm:ss]", timeNo));
        Validate.isTrue(DateTimeValidateUtil.validateTime(timeDto.getOffWorkTime()), String.format("第%s段打卡时间,非法的下班时间格式,可用的格式:[HH:mm:ss]", timeNo));
        Validate.isTrue(DateTimeValidateUtil.validateTime(timeDto.getOffWorkClockEndTime()), String.format("第%s段打卡时间,非法的下班打卡结束时间格式,可用的格式:[HH:mm:ss]", timeNo));
        Validate.isTrue(DateTimeValidateUtil.validateTime(timeDto.getOffWorkClockStartTime()), String.format("第%s段打卡时间,非法的下班打卡开始时间格式,可用的格式:[HH:mm:ss]", timeNo));
        Validate.isTrue(DateTimeValidateUtil.validateTime(timeDto.getOnWorkClockEndTime()), String.format("第%s段打卡时间,非法的上班打卡结束时间格式,可用的格式:[HH:mm:ss]", timeNo));
        Validate.isTrue(DateTimeValidateUtil.validateTime(timeDto.getOnWorkClockStartTime()), String.format("第%s段打卡时间,非法的上班打卡开始时间格式,可用的格式:[HH:mm:ss]", timeNo));
        Validate.isTrue(timeDto.getOffWorkClockStartTime().compareTo(timeDto.getOffWorkClockEndTime()) < 0
            , String.format("第%s段打卡时间,下班打卡开始时间必须早于下班打卡结束时间", timeNo));
        Validate.isTrue(timeDto.getOnWorkClockStartTime().compareTo(timeDto.getOnWorkClockEndTime()) < 0
            , String.format("第%s段打卡时间,上班打卡开始时间必须早于上班打卡结束时间", timeNo));
        Validate.isTrue(timeDto.getOffWorkClockStartTime().compareTo(timeDto.getOnWorkClockEndTime()) >= 0
            , String.format("第%s段打卡时间,下班打卡开始时间不能早于上班打卡结束时间", timeNo));
        Validate.isTrue(indexOffWorkTime.compareTo(timeDto.getOnWorkTime()) <= 0
            , String.format("第%s段上班时间不能早于第%s段下班时间", timeNo, timeNo - 1));
        Validate.isTrue(indexOnWorkClockEndTime.compareTo(timeDto.getOnWorkClockStartTime()) <= 0
            , String.format("第%s段上班打卡开始时间不能早于第%s段上班打卡结束时间", timeNo, timeNo - 1));
        Validate.isTrue(indexOnWorkClockEndTime.compareTo(timeDto.getOffWorkClockStartTime()) <= 0
            , String.format("第%s段下班打卡开始时间不能早于第%s段上班打卡结束时间", timeNo, timeNo - 1));
        Validate.isTrue(indexOffWorkClockEndTime.compareTo(timeDto.getOnWorkClockStartTime()) <= 0
            , String.format("第%s段上班打卡开始时间不能早于第%s段下班打卡结束时间", timeNo, timeNo - 1));
        Validate.isTrue(indexOffWorkClockEndTime.compareTo(timeDto.getOffWorkClockStartTime()) <= 0
            , String.format("第%s段下班打卡开始时间不能早于第%s段下班打卡结束时间", timeNo, timeNo - 1));
        indexOffWorkTime = timeDto.getOffWorkTime();
        indexOnWorkClockEndTime = timeDto.getOnWorkClockEndTime();
        indexOffWorkClockEndTime = timeDto.getOffWorkClockEndTime();
      }
    } else {
      List<AttendanceRuleTimeDto> timeList = Lists.newArrayList();
      // 自由上下班打卡时，默认值设置
      AttendanceRuleTimeDto timeDto = new AttendanceRuleTimeDto();
      timeDto.setOnWorkTime("00:00:00");
      timeDto.setOffWorkTime("23:59:59");
      timeDto.setOnWorkClockStartTime(timeDto.getOnWorkTime());
      timeDto.setOnWorkClockEndTime(timeDto.getOffWorkTime());
      timeDto.setOffWorkClockStartTime(timeDto.getOnWorkTime());
      timeDto.setOffWorkClockEndTime(timeDto.getOffWorkTime());
      timeDto.setTimeNo("1");
      timeList.add(timeDto);
      dto.setTimeList(timeList);
    }

  }
}
