package com.biz.crm.sfa.business.attendance.local.service.internal;

import com.biz.crm.mdm.business.org.sdk.service.OrgVoService;
import com.biz.crm.mdm.business.org.sdk.vo.OrgVo;
import com.biz.crm.mdm.business.position.level.sdk.service.PositionLevelVoService;
import com.biz.crm.mdm.business.position.level.sdk.vo.PositionLevelVo;
import com.biz.crm.sfa.business.attendance.local.entity.AttendanceRuleEntity;
import com.biz.crm.sfa.business.attendance.local.repository.AttendanceRuleRepository;
import com.biz.crm.sfa.business.attendance.sdk.dto.RuleConditionDto;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceRuleScopeType;
import com.biz.crm.sfa.business.attendance.sdk.event.AttendanceRuleEventListener;
import com.biz.crm.sfa.business.attendance.sdk.service.AttendanceRuleVoService;
import com.biz.crm.sfa.business.attendance.sdk.vo.AttendanceRuleNoWorkAbideVo;
import com.biz.crm.sfa.business.attendance.sdk.vo.AttendanceRuleScopeVo;
import com.biz.crm.sfa.business.attendance.sdk.vo.AttendanceRuleVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 考勤规则Vo表服务接口实现类
 *
 * @author ning.zhang
 * @date 2022/5/6
 */
@Slf4j
@Service
public class AttendanceRuleVoServiceImpl implements AttendanceRuleVoService {

  @Autowired
  private AttendanceRuleRepository attendanceRuleRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private OrgVoService orgVoService;
  @Autowired
  private PositionLevelVoService positionLevelVoService;
  @Autowired(required = false)
  private List<AttendanceRuleEventListener> listeners;


  @Override
  public AttendanceRuleVo findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    List<AttendanceRuleEntity> entities = this.attendanceRuleRepository.findByIds(Lists.newArrayList(id));
    if (CollectionUtils.isEmpty(entities)) {
      return null;
    }
    AttendanceRuleVo attendanceRuleVo = this.nebulaToolkitService.copyObjectByBlankList(entities.get(0), AttendanceRuleVo.class, HashSet.class, ArrayList.class);
    this.perfectScopeInfo(Lists.newArrayList(attendanceRuleVo));
    //考勤规则非工作日遵守规则关联信息事件
    if (!CollectionUtils.isEmpty(listeners)) {
      Set<String> ruleCodes = Sets.newHashSet(attendanceRuleVo.getRuleCode());
      List<AttendanceRuleNoWorkAbideVo> noWorkAbideInfoList = Lists.newArrayList();
      listeners.forEach(listener -> {
        List<AttendanceRuleNoWorkAbideVo> abideList = listener.onRequestByRuleCodes(ruleCodes);
        if (!CollectionUtils.isEmpty(abideList)) {
          noWorkAbideInfoList.addAll(abideList);
        }
      });
      attendanceRuleVo.setNoWorkAbideInfoList(noWorkAbideInfoList);
    }
    return attendanceRuleVo;
  }

  @Override
  public List<AttendanceRuleVo> findByRuleConditionDto(RuleConditionDto dto) {
    if (Objects.isNull(dto)) {
      return Lists.newLinkedList();
    }
    dto.setTenantCode(TenantUtils.getTenantCode());
    List<AttendanceRuleEntity> entities = this.attendanceRuleRepository.findByRuleConditionDto(dto);
    if (CollectionUtils.isEmpty(entities)) {
      return Lists.newLinkedList();
    }
    return (List<AttendanceRuleVo>) this.nebulaToolkitService.copyCollectionByBlankList(entities, AttendanceRuleEntity.class
        , AttendanceRuleVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 完善考勤规则范围信息
   *
   * @param ruleList 考勤规则信息
   */
  private void perfectScopeInfo(List<AttendanceRuleVo> ruleList) {
    if (CollectionUtils.isEmpty(ruleList)) {
      return;
    }
    List<AttendanceRuleScopeVo> scopeList = ruleList.stream().filter(vo -> !CollectionUtils.isEmpty(vo.getScopeList()))
        .flatMap(vo -> vo.getScopeList().stream()).collect(Collectors.toList());
    if (CollectionUtils.isEmpty(scopeList)) {
      return;
    }
    Set<String> orgCodes = Sets.newHashSet();
    Set<String> positionLevelCodes = Sets.newHashSet();
    scopeList.forEach(scopeVo -> {
      if (AttendanceRuleScopeType.ORG.getDictCode().equals(scopeVo.getScopeType())) {
        orgCodes.add(scopeVo.getScopeCode());
      }
      if (AttendanceRuleScopeType.POSITION_LEVEL.getDictCode().equals(scopeVo.getScopeType())) {
        positionLevelCodes.add(scopeVo.getScopeCode());
      }
    });
    Map<String, OrgVo> orgMap = this.buildOrgMap(orgCodes);
    Map<String, PositionLevelVo> positionLevelMap = this.buildPositionLevelMap(positionLevelCodes);
    for (AttendanceRuleScopeVo scopeVo : scopeList) {
      if (AttendanceRuleScopeType.ORG.getDictCode().equals(scopeVo.getScopeType())) {
        OrgVo orgVo = orgMap.getOrDefault(scopeVo.getScopeCode(), new OrgVo());
        scopeVo.setScopeName(orgVo.getOrgName());
      }
      if (AttendanceRuleScopeType.POSITION_LEVEL.getDictCode().equals(scopeVo.getScopeType())) {
        PositionLevelVo positionLevelVo = positionLevelMap.getOrDefault(scopeVo.getScopeCode(), new PositionLevelVo());
        scopeVo.setScopeName(positionLevelVo.getPositionLevelName());
      }
    }
  }

  /**
   * 封装职级映射MAP
   *
   * @param positionLevelCodes 职级编码
   * @return 职级映射MAP
   */
  private Map<String, PositionLevelVo> buildPositionLevelMap(Set<String> positionLevelCodes) {
    Map<String, PositionLevelVo> positionLevelMap = Maps.newHashMap();
    if (!CollectionUtils.isEmpty(positionLevelCodes)) {
      List<PositionLevelVo> positionLevelList = this.positionLevelVoService.findByIdsOrCodes(null, Lists.newArrayList(positionLevelCodes));
      if (!CollectionUtils.isEmpty(positionLevelList)) {
        positionLevelMap = positionLevelList.stream().collect(Collectors.toMap(PositionLevelVo::getPositionLevelCode, t -> t, (a, b) -> b));
      }
    }
    return positionLevelMap;
  }

  /**
   * 封装组织映射MAP
   *
   * @param orgCodes 组织编码
   * @return 组织映射MAP
   */
  private Map<String, OrgVo> buildOrgMap(Set<String> orgCodes) {
    Map<String, OrgVo> orgMap = Maps.newHashMap();
    if (!CollectionUtils.isEmpty(orgCodes)) {
      List<OrgVo> orgList = this.orgVoService.findByOrgCodes(Lists.newArrayList(orgCodes));
      if (!CollectionUtils.isEmpty(orgList)) {
        orgMap = orgList.stream().collect(Collectors.toMap(OrgVo::getOrgCode, t -> t, (a, b) -> b));
      }
    }
    return orgMap;
  }
}
