package com.biz.crm.sfa.business.attendance.local.service.internal;

import com.biz.crm.business.common.identity.FacturerUserDetails;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.user.sdk.service.UserFeignVoService;
import com.biz.crm.mdm.business.user.sdk.vo.UserVo;
import com.biz.crm.sfa.business.attendance.local.entity.AttendanceRecordEntity;
import com.biz.crm.sfa.business.attendance.local.entity.AttendanceRecordRuleEntity;
import com.biz.crm.sfa.business.attendance.local.entity.AttendanceRuleEntity;
import com.biz.crm.sfa.business.attendance.local.entity.AttendanceRuleTimeEntity;
import com.biz.crm.sfa.business.attendance.local.model.RuleExecuteModel;
import com.biz.crm.sfa.business.attendance.local.model.RuleExecuteUserModel;
import com.biz.crm.sfa.business.attendance.local.repository.AttendanceRecordRepository;
import com.biz.crm.sfa.business.attendance.local.repository.AttendanceRecordRuleRepository;
import com.biz.crm.sfa.business.attendance.local.service.AttendanceRecordService;
import com.biz.crm.sfa.business.attendance.sdk.dto.AttendanceFillClockDto;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceClockStatus;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceClockType;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceRuleType;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 考勤记录表服务实现类
 *
 * @author ning.zhang
 * @date 2022-05-17 09:52:18
 */
@Slf4j
@Service("attendanceRecordService")
public class AttendanceRecordServiceImpl implements AttendanceRecordService {

  @Autowired
  private AttendanceRecordRepository attendanceRecordRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private LoginUserService loginUserService;
  @Autowired
  private AttendanceRecordRuleRepository attendanceRecordRuleRepository;
  @Autowired
  private UserFeignVoService userFeignVoService;

  @Override
  @Transactional
  public void update(RuleExecuteModel executeModel, AttendanceRecordRuleEntity recordRuleEntity) {
    Validate.notNull(executeModel, "缺失规则执行信息");
    Validate.notNull(recordRuleEntity, "缺失记录规则信息");
    Validate.notNull(executeModel.getRuleMap().get(recordRuleEntity.getRuleId()), "缺失考勤规则信息");
    //构建考勤记录
    List<AttendanceRecordEntity> recordEntities = this.buildRecordEntity(executeModel, recordRuleEntity);
    if (CollectionUtils.isEmpty(recordEntities)) {
      return;
    }
    //过滤已使用的考勤记录
    List<AttendanceRecordEntity> haveUseRecordEntities = this.attendanceRecordRepository.findByRuleDateAndUsedStatus(TenantUtils.getTenantCode()
        , recordRuleEntity.getRuleDate(), BooleanEnum.TRUE.getCapital());
    if (!CollectionUtils.isEmpty(haveUseRecordEntities)) {
      Set<String> userNames = haveUseRecordEntities.stream().map(AttendanceRecordEntity::getUserName).collect(Collectors.toSet());
      recordEntities = recordEntities.stream().filter(entity -> !userNames.contains(entity.getUserName())).collect(Collectors.toList());
    }
    if (CollectionUtils.isEmpty(recordEntities)) {
      return;
    }
    //分批次保存
    List<List<AttendanceRecordEntity>> splitEntities = Lists.partition(recordEntities, 100);
    splitEntities.forEach(entities -> {
      this.attendanceRecordRepository.saveBatch(entities);
    });
  }

  @Override
  public List<AttendanceRecordEntity> findByAttendanceFillClockDto(AttendanceFillClockDto dto) {
    if (Objects.isNull(dto) || StringUtils.isAnyBlank(dto.getBeginTime(), dto.getEndTime())) {
      return Lists.newLinkedList();
    }
    FacturerUserDetails loginDetails = this.loginUserService.getLoginDetails(FacturerUserDetails.class);
    dto.setUserName(loginDetails.getAccount());
    dto.setTenantCode(TenantUtils.getTenantCode());
    List<AttendanceRecordEntity> recordEntities = this.attendanceRecordRepository.findByAttendanceFillClockDto(dto);
    if (CollectionUtils.isEmpty(recordEntities)) {
      return Lists.newLinkedList();
    }
    Set<String> recordRuleIds = recordEntities.stream().map(AttendanceRecordEntity::getRecordRuleId).collect(Collectors.toSet());
    List<AttendanceRecordRuleEntity> ruleEntities = this.attendanceRecordRuleRepository.listByIds(recordRuleIds);
    if (CollectionUtils.isEmpty(ruleEntities)) {
      return Lists.newLinkedList();
    }
    Map<String, AttendanceRecordRuleEntity> ruleEntityMap = ruleEntities.stream().collect(Collectors.toMap(AttendanceRecordRuleEntity::getId, t -> t, (a, b) -> b));
    recordEntities.forEach(entity -> {
      AttendanceRecordRuleEntity recordRuleEntity = ruleEntityMap.get(entity.getRecordRuleId());
      //若是自由签到就不返回时间信息
      if (AttendanceRuleType.FREE_TIME.getDictCode().equals(recordRuleEntity.getRuleType())) {
        entity.setRuleClockTime("");
      }
    });
    //当一天存在有两条以上的打卡异常记录就添加全天打卡类型
    if (recordEntities.size() > 1) {
      AttendanceRecordEntity recordEntity = new AttendanceRecordEntity();
      recordEntity.setClockType(AttendanceClockType.ON_AND_OFF_WORK.getDictCode());
      recordEntities.add(recordEntity);
    }
    return recordEntities;
  }

  /**
   * 构建考勤记录实体
   *
   * @param executeModel     规则执行model
   * @param recordRuleEntity 考勤记录规则
   * @return 考勤记录列表
   */
  private List<AttendanceRecordEntity> buildRecordEntity(RuleExecuteModel executeModel, AttendanceRecordRuleEntity recordRuleEntity) {
    List<AttendanceRecordEntity> recordEntities = Lists.newArrayList();
    AttendanceRuleEntity ruleEntity = executeModel.getRuleMap().get(recordRuleEntity.getRuleId());
    List<AttendanceRuleTimeEntity> timeList = ruleEntity.getTimeList();
    List<RuleExecuteUserModel> userModels = executeModel.getRuleUserMap().get(recordRuleEntity.getRuleId());

    if (CollectionUtils.isEmpty(userModels)) {
      return Lists.newLinkedList();
    }
    List<String> userNames = userModels.stream().filter(ruleExecuteUserModel -> StringUtils.isNotBlank(ruleExecuteUserModel.getUserName())).map(RuleExecuteUserModel::getUserName).collect(Collectors.toList());
    List<UserVo> userVos = this.userFeignVoService.findByUserNames(userNames);
    Map<String, UserVo> userVoMap = new HashMap<>();
    if (!CollectionUtils.isEmpty(userVos)) {
      userVoMap.putAll(userVos.stream().collect(Collectors.toMap(UserVo::getUserName, t -> t, (a, b) -> b)));
    }


    userModels.forEach(userModel -> {
      timeList.forEach(ruleTimeEntity -> {
        // 上班签到
        UserVo userVo = userVoMap.get(userModel.getUserName());
        AttendanceRecordEntity onWorkEntity = new AttendanceRecordEntity();
        onWorkEntity.setTenantCode(recordRuleEntity.getTenantCode());
        onWorkEntity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        onWorkEntity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        onWorkEntity.setRuleId(recordRuleEntity.getRuleId());
        onWorkEntity.setRecordRuleId(recordRuleEntity.getId());
        onWorkEntity.setRuleTimeId(ruleTimeEntity.getId());
        onWorkEntity.setClockStatus(AttendanceClockStatus.NONE.getDictCode());
        onWorkEntity.setUserName(userModel.getUserName());
        if (Objects.nonNull(userVo)) {
          onWorkEntity.setUserRealName(userVo.getFullName());
          onWorkEntity.setOrgCode(userVo.getOrgName());
          onWorkEntity.setOrgName(userVo.getOrgName());
          onWorkEntity.setPostCode(userVo.getPositionCode());
          onWorkEntity.setPostName(userVo.getPositionName());
        }
        onWorkEntity.setUsedStatus(BooleanEnum.FALSE.getCapital());
        onWorkEntity.setRuleDate(recordRuleEntity.getRuleDate());
        onWorkEntity.setRuleCode(recordRuleEntity.getRuleCode());
        // 下班签退
        AttendanceRecordEntity offWorkEntity = this.nebulaToolkitService.copyObjectByBlankList(onWorkEntity, AttendanceRecordEntity.class, HashSet.class, ArrayList.class);
        onWorkEntity.setClockType(AttendanceClockType.ON_WORK.getDictCode());
        offWorkEntity.setClockType(AttendanceClockType.OFF_WORK.getDictCode());

        // 打卡时间设置-上班
        onWorkEntity.setRuleClockTime(ruleTimeEntity.getOnWorkTime());
        onWorkEntity.setRuleClockStartTime(ruleTimeEntity.getOnWorkClockStartTime());
        onWorkEntity.setRuleClockEndTime(ruleTimeEntity.getOnWorkClockEndTime());

        // 打卡时间设置-下班
        offWorkEntity.setRuleClockTime(ruleTimeEntity.getOffWorkTime());
        offWorkEntity.setRuleClockStartTime(ruleTimeEntity.getOffWorkClockStartTime());
        offWorkEntity.setRuleClockEndTime(ruleTimeEntity.getOffWorkClockEndTime());

        recordEntities.add(onWorkEntity);
        recordEntities.add(offWorkEntity);
      });
    });
    return recordEntities;
  }
}
