package com.biz.crm.sfa.business.attendance.local.service.internal;

import com.alibaba.fastjson.JSON;
import com.biz.crm.business.common.identity.FacturerUserDetails;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.business.common.sdk.utils.DistanceUtil;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalVoService;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalVo;
import com.biz.crm.mdm.business.user.sdk.service.UserInfoVoService;
import com.biz.crm.mdm.business.user.sdk.vo.UserInfoVo;
import com.biz.crm.sfa.business.attendance.local.entity.AttendanceRecordEntity;
import com.biz.crm.sfa.business.attendance.local.entity.AttendanceRecordRuleEntity;
import com.biz.crm.sfa.business.attendance.local.entity.AttendanceRecordRulePlaceEntity;
import com.biz.crm.sfa.business.attendance.local.repository.AttendanceRecordRepository;
import com.biz.crm.sfa.business.attendance.local.repository.AttendanceRecordRulePlaceRepository;
import com.biz.crm.sfa.business.attendance.local.repository.AttendanceRecordRuleRepository;
import com.biz.crm.sfa.business.attendance.local.repository.AttendanceRecordVoRepository;
import com.biz.crm.sfa.business.attendance.local.service.AttendanceRecordPictureService;
import com.biz.crm.sfa.business.attendance.sdk.constant.AttendanceConstant;
import com.biz.crm.sfa.business.attendance.sdk.dto.AttendanceClockDto;
import com.biz.crm.sfa.business.attendance.sdk.dto.AttendancePlaceStatusDto;
import com.biz.crm.sfa.business.attendance.sdk.dto.RuleNoWorkAbideDataDto;
import com.biz.crm.sfa.business.attendance.sdk.dto.TodayHistoryRecordDto;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceClockStatus;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceClockTimeStatus;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceClockType;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceElectronFenceType;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceOffWorkClockType;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendancePlaceStatus;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceRuleType;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceSignOrNonType;
import com.biz.crm.sfa.business.attendance.sdk.event.AttendanceRuleEventListener;
import com.biz.crm.sfa.business.attendance.sdk.service.AttendanceRecordVoService;
import com.biz.crm.sfa.business.attendance.sdk.vo.AttendanceRecordVo;
import com.biz.crm.sfa.business.attendance.sdk.vo.AttendanceRuleNoWorkAbideDataVo;
import com.biz.crm.sfa.business.attendance.sdk.vo.AttendanceRuleNoWorkAbideVo;
import com.biz.crm.sfa.business.client.sdk.enums.ClientTypeEnum;
import com.biz.crm.sfa.business.visit.plan.sdk.dto.VisitPlanDetailQueryDto;
import com.biz.crm.sfa.business.visit.plan.sdk.service.VisitPlanDetailVoService;
import com.biz.crm.sfa.business.visit.plan.sdk.vo.VisitPlanDetailVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 考勤记录Vo服务接口实现类
 *
 * @author ning.zhang
 * @date 2022/5/18
 */
@Slf4j
@Service
public class AttendanceRecordVoServiceImpl implements AttendanceRecordVoService {

  @Autowired
  private AttendanceRecordRepository attendanceRecordRepository;
  @Autowired
  private AttendanceRecordVoRepository attendanceRecordVoRepository;
  @Autowired
  private AttendanceRecordRuleRepository attendanceRecordRuleRepository;
  @Autowired
  private AttendanceRecordRulePlaceRepository attendanceRecordRulePlaceRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private AttendanceRecordPictureService attendanceRecordPictureService;
  @Autowired
  private LoginUserService loginUserService;
  @Autowired
  private VisitPlanDetailVoService visitPlanDetailVoService;
  @Autowired
  private CustomerVoService customerVoService;
  @Autowired
  private TerminalVoService terminalVoService;
  @Autowired
  private UserInfoVoService userInfoVoService;
  @Autowired(required = false)
  private List<AttendanceRuleEventListener> listeners;

  @Override
  public AttendanceRecordVo findByAttendanceClockDto(AttendanceClockDto dto) {
    AttendanceRecordVo attendanceRecordVo = new AttendanceRecordVo();
    attendanceRecordVo.setClockTimeStatus(AttendanceClockTimeStatus.NOT_CLOCK_TIME.getDictCode());
    if (Objects.isNull(dto) || StringUtils.isBlank(dto.getClockType())) {
      return attendanceRecordVo;
    }
    FacturerUserDetails loginDetails = this.loginUserService.getLoginDetails(FacturerUserDetails.class);
    List<AttendanceRecordEntity> recordEntities = this.attendanceRecordRepository.findByUserNameAndRuleDateAndClockType(TenantUtils.getTenantCode()
        , loginDetails.getAccount(), LocalDate.now().format(AttendanceConstant.YYYY_MM_DD), dto.getClockType());
    if (CollectionUtils.isEmpty(recordEntities)) {
      attendanceRecordVo.setClockTimeStatus(AttendanceClockTimeStatus.TODAY_NOT_NEED_CLOCK.getDictCode());
      return attendanceRecordVo;
    }
    String nowTime = LocalDateTime.now().format(AttendanceConstant.HH_MM_SS);
    Map<String, List<AttendanceRecordVo>> attendanceRecordMap = recordEntities.stream()
        .sorted(Comparator.comparing(AttendanceRecordEntity::getRuleClockTime))
        .filter(entity -> nowTime.compareTo(entity.getRuleClockEndTime()) <= 0).map(this::buildAttendanceRecordVo)
        .collect(Collectors.groupingBy(AttendanceRecordVo::getClockTimeStatus));
    List<AttendanceRecordVo> notNeedList = attendanceRecordMap.get(AttendanceClockTimeStatus.TODAY_NOT_NEED_CLOCK.getDictCode());
    List<AttendanceRecordVo> canList = attendanceRecordMap.get(AttendanceClockTimeStatus.CURRENT_CAN_CLOCK.getDictCode());
    List<AttendanceRecordVo> notTimeList = attendanceRecordMap.get(AttendanceClockTimeStatus.NOT_CLOCK_TIME.getDictCode());
    //优先级在打卡时间范围内->不在打卡时间范围内->不需要打卡
    if (!CollectionUtils.isEmpty(canList)) {
      attendanceRecordVo = canList.get(0);
    } else if (!CollectionUtils.isEmpty(notTimeList)) {
      attendanceRecordVo = notTimeList.get(0);
    } else if (!CollectionUtils.isEmpty(notNeedList)) {
      attendanceRecordVo = notNeedList.get(0);
    }
    return attendanceRecordVo;
  }

  @Override
  @Transactional
  public void updateByAttendanceClockDto(AttendanceClockDto dto) {
    Validate.notNull(dto, "缺失考勤打卡信息");
    Validate.notBlank(dto.getClockType(), "缺失打卡类型");
    Validate.notBlank(dto.getRecordId(), "缺失考勤记录ID");
    Validate.notNull(dto.getClockLatitude(), "缺失定位信息");
    Validate.notNull(dto.getClockLongitude(), "缺失定位信息");
    AttendanceRecordEntity recordEntity = this.attendanceRecordRepository.getById(dto.getRecordId());
    Validate.notNull(recordEntity, "考勤记录不存在");
    Validate.isTrue(dto.getClockType().equals(recordEntity.getClockType()), "考勤打卡类型不匹配");
    FacturerUserDetails loginDetails = this.loginUserService.getLoginDetails(FacturerUserDetails.class);
    Validate.isTrue(loginDetails.getAccount().equals(recordEntity.getUserName()), "打卡人与打卡记录不匹配");
    AttendanceRecordRuleEntity recordRuleEntity = this.attendanceRecordRuleRepository.getById(recordEntity.getRecordRuleId());
    Validate.notNull(recordEntity, "考勤记录规则不存在");
    //校验打卡记录基本信息
    this.validateRecord(dto, recordEntity, recordRuleEntity);
    //校验考勤打卡地点
    String clockPlaceStatus = this.validatePlace(dto, recordRuleEntity, loginDetails);
    Validate.isTrue(!(AttendancePlaceStatus.EX.getDictCode().equals(clockPlaceStatus) && AttendanceElectronFenceType.NO_OUT_SIGN.getDictCode().equals(recordRuleEntity.getElectronFenceType()))
        , "不允许考勤范围外打卡");
    //构建封装考勤记录状态
    this.buildClockStatus(recordEntity, recordRuleEntity);
    recordEntity.setClockPlaceStatus(clockPlaceStatus);
    recordEntity.setRemark(dto.getRemark());
    recordEntity.setExceptionRemarks(dto.getRemark());
    recordEntity.setClockPlace(dto.getClockPlace());
    recordEntity.setClockLongitude(dto.getClockLongitude());
    recordEntity.setClockLatitude(dto.getClockLatitude());
    this.attendanceRecordRepository.updateById(recordEntity);
    //将该用户当天所有的考勤数据标记为已使用，防止已打卡的考勤数据被WEB端的手动刷新规则功能冲掉
    this.attendanceRecordRepository.updateUsedStatusByUserNameAndRecordRuleId(recordEntity.getTenantCode()
        , recordEntity.getUserName(), recordEntity.getRecordRuleId(), BooleanEnum.TRUE.getCapital());
    this.attendanceRecordPictureService.update(dto);
  }

  @Override
  public List<AttendanceRecordVo> findByTodayHistoryRecordDto(TodayHistoryRecordDto dto) {
    if (Objects.isNull(dto) || StringUtils.isAnyBlank(dto.getClockType(), dto.getBeforeRecordId())) {
      return Lists.newLinkedList();
    }
    dto.setTenantCode(TenantUtils.getTenantCode());
    FacturerUserDetails loginDetails = this.loginUserService.getLoginDetails(FacturerUserDetails.class);
    dto.setUserName(loginDetails.getAccount());
    List<AttendanceRecordEntity> recordEntities = this.attendanceRecordRepository.findByTodayHistoryRecordDto(dto);
    if (CollectionUtils.isEmpty(recordEntities)) {
      return Lists.newLinkedList();
    }
    return (List<AttendanceRecordVo>) this.nebulaToolkitService.copyCollectionByWhiteList(recordEntities, AttendanceRecordEntity.class
        , AttendanceRecordVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public List<AttendanceRecordVo> findByIds(List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return Lists.newLinkedList();
    }
    List<AttendanceRecordEntity> recordEntities = this.attendanceRecordRepository.findByIds(ids);
    if (CollectionUtils.isEmpty(recordEntities)) {
      return Lists.newLinkedList();
    }
    List<AttendanceRecordVo> recordList = (List<AttendanceRecordVo>) this.nebulaToolkitService.copyCollectionByBlankList(recordEntities, AttendanceRecordEntity.class
        , AttendanceRecordVo.class, HashSet.class, ArrayList.class);
    this.buildUserInfo(recordList);
    return recordList;
  }

  @Override
  public AttendanceRecordVo findByAttendancePlaceStatusDto(AttendancePlaceStatusDto dto) {
    if (StringUtils.isBlank(dto.getRecordId())) {
      return null;
    }
    AttendanceRecordEntity recordEntity = this.attendanceRecordRepository.getById(dto.getRecordId());
    if (Objects.isNull(recordEntity)) {
      return null;
    }
    AttendanceRecordRuleEntity recordRuleEntity = this.attendanceRecordRuleRepository.getById(recordEntity.getRecordRuleId());
    if (Objects.isNull(recordRuleEntity)) {
      return null;
    }
    AttendanceClockDto clockDto = this.nebulaToolkitService.copyObjectByBlankList(dto, AttendanceClockDto.class, HashSet.class, ArrayList.class);
    FacturerUserDetails loginDetails = this.loginUserService.getLoginDetails(FacturerUserDetails.class);
    clockDto.setClockType(recordEntity.getClockType());
    AttendanceRecordVo recordVo = this.nebulaToolkitService.copyObjectByBlankList(recordEntity, AttendanceRecordVo.class, HashSet.class, ArrayList.class);
    recordVo.setClockPlaceStatus(this.validatePlace(clockDto, recordRuleEntity, loginDetails));
    return recordVo;
  }

  @Override
  public List<AttendanceRecordVo> findByHistoryRecordDto(TodayHistoryRecordDto dto) {
    if (Objects.isNull(dto)) {
      dto = new TodayHistoryRecordDto();
    }
    dto.setTenantCode(TenantUtils.getTenantCode());
    List<AttendanceRecordVo> recordEntities = this.attendanceRecordVoRepository.findByHistoryRecordDto(dto);
    return recordEntities;
  }

  /**
   * 构建封装用户信息
   *
   * @param recordList 考勤记录
   */
  private void buildUserInfo(List<AttendanceRecordVo> recordList) {
    if (CollectionUtils.isEmpty(recordList)) {
      return;
    }
    Set<String> userNames = recordList.stream().map(AttendanceRecordVo::getUserName).collect(Collectors.toSet());
    List<UserInfoVo> userInfoList = this.userInfoVoService.findByUserNames(userNames);
    if (CollectionUtils.isEmpty(userInfoList)) {
      return;
    }
    Map<String, UserInfoVo> userInfoMap = userInfoList.stream().collect(Collectors.toMap(UserInfoVo::getUserName, t -> t, (a, b) -> b));
    recordList.forEach(attendanceRecordVo -> {
      UserInfoVo userInfoVo = userInfoMap.getOrDefault(attendanceRecordVo.getUserName(), new UserInfoVo());
      attendanceRecordVo.setUserRealName(userInfoVo.getFullName());
    });
  }

  /**
   * 构建封装考勤记录状态
   *
   * @param recordEntity     考勤记录
   * @param recordRuleEntity 考勤规则记录
   */
  private void buildClockStatus(AttendanceRecordEntity recordEntity, AttendanceRecordRuleEntity recordRuleEntity) {
    //如果上班打卡为更新打卡不需要更新打卡时间和打卡状态
    if (AttendanceClockType.ON_WORK.getDictCode().equals(recordEntity.getClockType())
        && !AttendanceClockStatus.NONE.getDictCode().equals(recordEntity.getClockStatus())) {
      return;
    }
    String ruleDate = recordEntity.getRuleDate();
    //获取规则配置的打卡时间sfaWorkSignRecord
    LocalDateTime ruleClockTime = LocalDateTime.parse(String.format("%s %s", ruleDate, recordEntity.getRuleClockTime())
        , AttendanceConstant.YYYY_MM_DD_HH_MM_SS);
    LocalDateTime nowDateTime = LocalDateTime.now();
    String clockStatus = null;
    if (AttendanceRuleType.FREE_TIME.getDictCode().equals(recordRuleEntity.getRuleType())) {
      //自由时间上下班
      clockStatus = AttendanceClockStatus.OK.getDictCode();
    } else if (AttendanceClockType.ON_WORK.getDictCode().equals(recordEntity.getClockType())) {
      //上班迟到判定
      clockStatus = nowDateTime.compareTo(ruleClockTime) > 0 ? AttendanceClockStatus.BE_LATE.getDictCode() : AttendanceClockStatus.OK.getDictCode();
    } else if (AttendanceClockType.OFF_WORK.getDictCode().equals(recordEntity.getClockType())) {
      //下班早退判定
      clockStatus = nowDateTime.compareTo(ruleClockTime) < 0 ? AttendanceClockStatus.LEAVE_EARLY.getDictCode() : AttendanceClockStatus.OK.getDictCode();
    }
    Validate.isTrue(StringUtils.isNotBlank(clockStatus), "打卡状态异常");
    //考勤时间和状态
    recordEntity.setClockTime(nowDateTime.format(AttendanceConstant.HH_MM_SS));
    recordEntity.setClockStatus(clockStatus);
  }

  /**
   * 校验考勤打卡地点
   *
   * @param dto              参数dto
   * @param recordRuleEntity 考勤记录规则
   * @param loginUserDetails 用户
   * @return 打卡地点状态
   */
  private String validatePlace(AttendanceClockDto dto, AttendanceRecordRuleEntity recordRuleEntity, FacturerUserDetails loginUserDetails) {
    String electronFenceType = recordRuleEntity.getElectronFenceType();
    // 默认打卡地点异常
    String clockPlaceStatus = AttendancePlaceStatus.EX.getDictCode();
    // 无打卡范围信息默认为正常状态
    if (StringUtils.isBlank(electronFenceType) || AttendanceElectronFenceType.NONE.getDictCode().equals(electronFenceType)) {
      clockPlaceStatus = AttendancePlaceStatus.OK.getDictCode();
      return clockPlaceStatus;
    }
    // 允许范围外打卡，地点记录为正常
    if (AttendanceElectronFenceType.OUT_SIGN_OK.getDictCode().equals(electronFenceType)) {
      clockPlaceStatus = AttendancePlaceStatus.OK.getDictCode();
      return clockPlaceStatus;
    }
    // 打卡地点列表
    List<AttendanceRecordRulePlaceEntity> placeList = this.attendanceRecordRulePlaceRepository.findByRecordRuleId(recordRuleEntity.getId());
    Validate.isTrue(!CollectionUtils.isEmpty(placeList), "不允许考勤范围外打卡");
    // 判断是否范围外
    for (AttendanceRecordRulePlaceEntity placeEntity : placeList) {
      //判断是否是同步拜访地址,找拜访计划明细中的当天需要拜访的地址,校验当前打卡点是否在其范围内
      if (Boolean.TRUE.equals(this.validateVisitPlanPlace(dto, loginUserDetails, placeEntity))) {
        clockPlaceStatus = AttendancePlaceStatus.OK.getDictCode();
      }
      //匹配地点打卡类型
      if (dto.getClockType().equals(placeEntity.getClockType()) || AttendanceClockType.ON_AND_OFF_WORK.getDictCode().equals(placeEntity.getClockType())) {
        //计算两个坐标之前的距离
        double distance = DistanceUtil.calculatePointDistance(dto.getClockLatitude().doubleValue(), dto.getClockLongitude().doubleValue(),
            placeEntity.getPlaceLatitude().doubleValue(), placeEntity.getPlaceLongitude().doubleValue());
        if (BigDecimal.valueOf(placeEntity.getPlaceRange()).compareTo(BigDecimal.valueOf(distance)) >= 0) {
          clockPlaceStatus = AttendancePlaceStatus.OK.getDictCode();
        }
      }
      if (AttendancePlaceStatus.OK.getDictCode().equals(clockPlaceStatus)) {
        break;
      }
    }
    return clockPlaceStatus;
  }

  /**
   * 校验拜访地址打卡范围
   *
   * @param dto              参数dto
   * @param loginUserDetails 用户
   * @param placeEntity      地点信息
   * @return 是否有效范围打卡
   */
  private Boolean validateVisitPlanPlace(AttendanceClockDto dto, FacturerUserDetails loginUserDetails
      , AttendanceRecordRulePlaceEntity placeEntity) {
    if (!AttendanceConstant.VISIT_PLAN_PLACE.equals(placeEntity.getPlaceName())) {
      return Boolean.FALSE;
    }
    VisitPlanDetailQueryDto queryDto = new VisitPlanDetailQueryDto();
    queryDto.setVisitUserName(loginUserDetails.getUsername());
    queryDto.setVisitPostCode(loginUserDetails.getPostCode());
    queryDto.setVisitDate(new Date());
    List<VisitPlanDetailVo> planDetailVos = this.visitPlanDetailVoService.findByConditions(queryDto);
    if (CollectionUtils.isEmpty(planDetailVos)) {
      return Boolean.FALSE;
    }
    List<Pair<BigDecimal, BigDecimal>> latLonList = Lists.newArrayList();
    List<Pair<BigDecimal, BigDecimal>> customerLatLonList = this.buildCustomerLatLon(planDetailVos);
    if (!CollectionUtils.isEmpty(customerLatLonList)) {
      latLonList.addAll(customerLatLonList);
    }
    List<Pair<BigDecimal, BigDecimal>> terminalLatLonList = this.buildTerminalLatLon(planDetailVos);
    if (!CollectionUtils.isEmpty(terminalLatLonList)) {
      latLonList.addAll(terminalLatLonList);
    }
    if (CollectionUtils.isEmpty(latLonList)) {
      return Boolean.FALSE;
    }
    Boolean result = Boolean.FALSE;
    for (Pair<BigDecimal, BigDecimal> latLonPair : latLonList) {
      //计算两个坐标之前的距离
      double distance = DistanceUtil.calculatePointDistance(dto.getClockLatitude().doubleValue(), dto.getClockLongitude().doubleValue(),
          latLonPair.getLeft().doubleValue(), latLonPair.getRight().doubleValue());
      if (BigDecimal.valueOf(placeEntity.getPlaceRange()).compareTo(BigDecimal.valueOf(distance)) >= 0) {
        result = Boolean.TRUE;
      }
    }
    return result;
  }

  /**
   * 构建拜访经销商经纬度列表
   *
   * @param planDetailVos 拜访地点
   * @return 经销商纬度列表
   */
  private List<Pair<BigDecimal, BigDecimal>> buildCustomerLatLon(List<VisitPlanDetailVo> planDetailVos) {
    List<Pair<BigDecimal, BigDecimal>> latLonList = Lists.newArrayList();
    List<String> customerCodes = planDetailVos.stream().filter(visitPlanDetailVo -> ClientTypeEnum.DEALER.getDictCode().equals(visitPlanDetailVo.getClientType()))
        .map(VisitPlanDetailVo::getClientCode).collect(Collectors.toList());
    if (!CollectionUtils.isEmpty(customerCodes)) {
      List<CustomerVo> customerVoList = this.customerVoService.findByCustomerCodes(customerCodes);
      List<Pair<BigDecimal, BigDecimal>> customerLatLonList = Lists.newArrayList();
      if (!CollectionUtils.isEmpty(customerVoList)) {
        customerLatLonList = customerVoList.stream().filter(customerVo -> Objects.nonNull(customerVo.getLatitude()) && Objects.nonNull(customerVo.getLongitude()))
            .map(customerVo -> Pair.of(customerVo.getLatitude(), customerVo.getLongitude())).collect(Collectors.toList());
      }
      if (!CollectionUtils.isEmpty(customerLatLonList)) {
        latLonList.addAll(customerLatLonList);
      }
    }
    return latLonList;
  }

  /**
   * 构建拜访终端经纬度列表
   *
   * @param planDetailVos 拜访地点
   * @return 终端经纬度列表
   */
  private List<Pair<BigDecimal, BigDecimal>> buildTerminalLatLon(List<VisitPlanDetailVo> planDetailVos) {
    List<Pair<BigDecimal, BigDecimal>> latLonList = Lists.newArrayList();
    List<String> terminalCodes = planDetailVos.stream().filter(visitPlanDetailVo -> ClientTypeEnum.TERMINAL.getDictCode().equals(visitPlanDetailVo.getClientType()))
        .map(VisitPlanDetailVo::getClientCode).collect(Collectors.toList());
    if (!CollectionUtils.isEmpty(terminalCodes)) {
      List<TerminalVo> terminalVoList = this.terminalVoService.findDetailsByIdsOrTerminalCodes(Lists.newLinkedList(), terminalCodes);
      List<Pair<BigDecimal, BigDecimal>> terminalLatLonList = Lists.newArrayList();
      if (!CollectionUtils.isEmpty(terminalVoList)) {
        terminalLatLonList = terminalVoList.stream().filter(terminalVo -> Objects.nonNull(terminalVo.getLatitude()) && Objects.nonNull(terminalVo.getLongitude()))
            .map(customerVo -> Pair.of(customerVo.getLatitude(), customerVo.getLongitude())).collect(Collectors.toList());
      }
      if (!CollectionUtils.isEmpty(terminalLatLonList)) {
        latLonList.addAll(terminalLatLonList);
      }
    }
    return latLonList;
  }

  /**
   * 校验打卡记录基本信息
   *
   * @param dto              参数dto
   * @param recordEntity     考勤记录
   * @param recordRuleEntity 考勤记录规则
   */
  private void validateRecord(AttendanceClockDto dto, AttendanceRecordEntity recordEntity, AttendanceRecordRuleEntity recordRuleEntity) {
    String ruleClockStartTime = recordEntity.getRuleClockStartTime();
    String ruleClockEndTime = recordEntity.getRuleClockEndTime();
    String nowTime = LocalTime.now().format(AttendanceConstant.HH_MM_SS);
    Validate.isTrue(ruleClockStartTime.compareTo(nowTime) <= 0 && ruleClockEndTime.compareTo(nowTime) >= 0
        , String.format("当前时间不能%s", AttendanceClockType.getByDictCode(dto.getClockType()).getValue()));
    //下班需要打卡，且必须打上班卡才能打下班卡
    if (AttendanceClockType.OFF_WORK.getDictCode().equals(recordEntity.getClockType())
        && AttendanceOffWorkClockType.GOTO_WORK_REQUIRED.getDictCode().equals(recordRuleEntity.getOffWorkClockType())) {
      // 获取上班签到记录,当前打卡组的上班签到记录
      List<AttendanceRecordEntity> recordEntities = this.attendanceRecordRepository
          .findByUserNameAndRuleTimeId(TenantUtils.getTenantCode(), recordEntity.getUserName(), recordEntity.getRuleTimeId());
      recordEntities = recordEntities.stream().filter(entity -> AttendanceClockType.ON_WORK.getDictCode().equals(entity.getClockType())).collect(Collectors.toList());
      Validate.isTrue(!CollectionUtils.isEmpty(recordEntities), "上班打卡记录不存在");
      Validate.isTrue(!AttendanceClockStatus.NONE.getDictCode().equals(recordEntities.get(0).getClockStatus()), "未上班打卡，不能下班打卡！");
    }
    // 拍照校验
    if (BooleanEnum.TRUE.getCapital().equals(recordRuleEntity.getClockPhotograph())) {
      Validate.isTrue(!CollectionUtils.isEmpty(dto.getPictureList()), "考勤照片不能为空");
    }
  }

  /**
   * 构建封装考勤记录VO
   *
   * @param entity 考勤记录实体
   * @return 考勤记录VO
   */
  private AttendanceRecordVo buildAttendanceRecordVo(AttendanceRecordEntity entity) {
    AttendanceRecordVo attendanceRecordVo = this.nebulaToolkitService.copyObjectByBlankList(entity, AttendanceRecordVo.class, HashSet.class, ArrayList.class);
    AttendanceRecordRuleEntity recordRuleEntity = this.attendanceRecordRuleRepository.getById(entity.getRecordRuleId());
    if (Objects.isNull(recordRuleEntity)) {
      attendanceRecordVo.setClockTimeStatus(AttendanceClockTimeStatus.TODAY_NOT_NEED_CLOCK.getDictCode());
      return attendanceRecordVo;
    }

    //设置打卡时间状态字段
    String ruleClockStartTime = entity.getRuleClockStartTime();
    String ruleClockEndTime = entity.getRuleClockEndTime();
    String nowTime = LocalTime.now().format(AttendanceConstant.HH_MM_SS);
    attendanceRecordVo.setClockTimeStatus(AttendanceClockTimeStatus.CURRENT_CAN_CLOCK.getDictCode());
    attendanceRecordVo.setClockPhotograph(recordRuleEntity.getClockPhotograph());
    Pair<Boolean, Boolean> noWorkAbideFlagPair = this.buildNoWorkAbideFlag(recordRuleEntity, entity);
    //非工作日且规则配置了非工作日不遵守考勤规则,则不限制打卡时间范围
    if (BooleanEnum.FALSE.getCapital().equals(recordRuleEntity.getSignMust()) && Boolean.FALSE.equals(noWorkAbideFlagPair.getLeft())) {
      attendanceRecordVo.setClockTimeStatus(AttendanceClockTimeStatus.CURRENT_CAN_CLOCK.getDictCode());
      //遵守时间打卡范围限制
    } else if (!(ruleClockStartTime.compareTo(nowTime) <= 0 && ruleClockEndTime.compareTo(nowTime) >= 0)) {
      attendanceRecordVo.setClockTimeStatus(AttendanceClockTimeStatus.NOT_CLOCK_TIME.getDictCode());
    }

    //设置是否工作日打卡操作字段
    String executeAsWorkingDay = BooleanEnum.FALSE.getCapital();
    //工作日配置了工作日特殊日期不用打卡,不用以工作日进行打卡操作
    if (BooleanEnum.TRUE.getCapital().equals(recordRuleEntity.getSignMust()) && !AttendanceSignOrNonType.SPECIAL_DAY_NO_SIGN.getDictCode().equals(recordRuleEntity.getSignOrNonType())) {
      executeAsWorkingDay = BooleanEnum.TRUE.getCapital();
    }
    //存在工作日调整(非工作日)申请时以工作日进行打卡操作
    if (BooleanEnum.FALSE.getCapital().equals(recordRuleEntity.getSignMust()) && Boolean.TRUE.equals(noWorkAbideFlagPair.getRight())) {
      executeAsWorkingDay = BooleanEnum.TRUE.getCapital();
    }
    attendanceRecordVo.setExecuteAsWorkingDay(executeAsWorkingDay);
    return attendanceRecordVo;
  }

  /**
   * 构建遵守考勤规则非工作日数据标识
   *
   * @param recordEntity     考勤记录实体
   * @param recordRuleEntity 考勤记录规则信息
   * @return 非工作日数据标识(L.是否存在遵守考勤规则非工作日模块, F.是否存在遵守考勤的非工作日模块数据)
   */
  private Pair<Boolean, Boolean> buildNoWorkAbideFlag(AttendanceRecordRuleEntity recordRuleEntity, AttendanceRecordEntity recordEntity) {
    //是否存在遵守考勤规则非工作日模块(例如加班遵守此考勤规则)
    Boolean haveNoWorkAbide = Boolean.FALSE;
    //是否存在遵守考勤的非工作日模块数据(例如当前日期存在加班)
    Boolean currentHaveAbideData = Boolean.FALSE;
    List<AttendanceRuleNoWorkAbideVo> recordRuleNoWorkAbideList = StringUtils.isNotBlank(recordRuleEntity.getNoWorkAbideInfo())
        ? JSON.parseArray(recordRuleEntity.getNoWorkAbideInfo(), AttendanceRuleNoWorkAbideVo.class) : null;
    //判定记录规则生成时是否存在遵守考勤规则非工作日模块
    Map<String, AttendanceRuleNoWorkAbideVo> recordRuleHaveNoWorkAbideMap = Maps.newHashMap();
    if (!CollectionUtils.isEmpty(recordRuleNoWorkAbideList)) {
      recordRuleHaveNoWorkAbideMap = recordRuleNoWorkAbideList.stream().filter(noWorkAbideVo
          -> BooleanEnum.TRUE.getCapital().equals(noWorkAbideVo.getAbideFlag())).collect(Collectors.toMap(AttendanceRuleNoWorkAbideVo::getKey
          , t -> t, (a, b) -> b));
      haveNoWorkAbide = !recordRuleHaveNoWorkAbideMap.isEmpty();
    }
    //判定记录规则生成时是否存在遵守考勤规则非工作日模块,且该遵守考勤规则非工作日模块点当前是否存在数据
    if (Boolean.TRUE.equals(haveNoWorkAbide) && !CollectionUtils.isEmpty(listeners)) {
      List<AttendanceRuleNoWorkAbideDataVo> noWorkAbideDataList = Lists.newArrayList();
      RuleNoWorkAbideDataDto dataDto = new RuleNoWorkAbideDataDto();
      dataDto.setRuleCodes(Sets.newHashSet(recordEntity.getRuleCode()));
      dataDto.setRuleDate(recordEntity.getRuleDate());
      dataDto.setUserName(recordEntity.getUserName());
      listeners.forEach(listener -> {
        List<AttendanceRuleNoWorkAbideDataVo> list = listener.onRequestByRuleNoWorkAbideDataDto(dataDto);
        if (!CollectionUtils.isEmpty(list)) {
          noWorkAbideDataList.addAll(list);
        }
      });
      for (AttendanceRuleNoWorkAbideDataVo abideDataVo : noWorkAbideDataList) {
        if (BooleanEnum.TRUE.getCapital().equals(abideDataVo.getCurrentHaveAbideData()) && Objects.nonNull(recordRuleHaveNoWorkAbideMap.get(abideDataVo.getKey()))) {
          currentHaveAbideData = Boolean.TRUE;
          break;
        }
      }
    }
    return Pair.of(haveNoWorkAbide, currentHaveAbideData);
  }
}
