package com.biz.crm.sfa.business.attendance.local.service.internal;

import com.biz.crm.sfa.business.attendance.local.entity.AttendanceRulePlaceEntity;
import com.biz.crm.sfa.business.attendance.local.repository.AttendanceRulePlaceRepository;
import com.biz.crm.sfa.business.attendance.local.service.AttendanceRulePlaceService;
import com.biz.crm.sfa.business.attendance.sdk.dto.AttendanceRuleDto;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceClockType;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceElectronFenceType;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 考勤规则地点表服务实现类
 *
 * @author ning.zhang
 * @date 2022-05-05 09:44:10
 */
@Slf4j
@Service("attendanceRulePlaceService")
public class AttendanceRulePlaceServiceImpl implements AttendanceRulePlaceService {

  @Autowired
  private AttendanceRulePlaceRepository attendanceRulePlaceRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Override
  @Transactional
  public void update(AttendanceRuleDto dto) {
    Validate.notBlank(dto.getId(), "考勤规则ID不能为空");
    this.attendanceRulePlaceRepository.deleteByRuleId(dto.getId());
    if (CollectionUtils.isEmpty(dto.getPlaceList())) {
      return;
    }
    this.updateValidation(dto);
    List<AttendanceRulePlaceEntity> entities = dto.getPlaceList().stream().map(placeDto -> {
      AttendanceRulePlaceEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(placeDto, AttendanceRulePlaceEntity.class, HashSet.class, ArrayList.class);
      entity.setRuleId(dto.getId());
      return entity;
    }).collect(Collectors.toList());
    this.attendanceRulePlaceRepository.saveBatch(entities);
  }

  /**
   * 在修改attendanceRuleScope模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(AttendanceRuleDto dto) {
    Set<String> clockTypeSet = Sets.newHashSet();
    Validate.notBlank(dto.getElectronFenceType(), "缺失打卡范围类型");
    dto.getPlaceList().forEach(placeDto -> {
      int placeNo = dto.getPlaceList().indexOf(placeDto) + 1;
      placeDto.setPlaceNo(String.valueOf(placeNo));
      Validate.notBlank(placeDto.getPlaceName(), String.format("序号%s,缺失地点名称", placeNo));
      Validate.isTrue(Objects.nonNull(placeDto.getPlaceRange()) && placeDto.getPlaceRange() > 0
          , String.format("序号%s,地点范围数据错误", placeNo));
      Validate.notBlank(placeDto.getClockType(), String.format("序号%s,缺失打卡类型", placeNo));
      Validate.notNull(placeDto.getPlaceLatitude(), String.format("序号%s,缺失地点纬度", placeNo));
      Validate.notNull(placeDto.getPlaceLongitude(), String.format("序号%s,缺失地点经度", placeNo));
      clockTypeSet.add(placeDto.getClockType());
    });
    if (!AttendanceElectronFenceType.NONE.getDictCode().equals(dto.getElectronFenceType())) {
      Validate.isTrue(clockTypeSet.contains(AttendanceClockType.ON_AND_OFF_WORK.getDictCode())
              || (clockTypeSet.contains(AttendanceClockType.OFF_WORK.getDictCode()) && clockTypeSet.contains(AttendanceClockType.ON_WORK.getDictCode()))
          , "打卡地点必须包含上下班打卡");
    }
  }
}
