package com.biz.crm.sfa.business.attendance.local.service.internal;

import com.biz.crm.sfa.business.attendance.local.entity.AttendanceRuleSpecialDateEntity;
import com.biz.crm.sfa.business.attendance.local.repository.AttendanceRuleSpecialDateRepository;
import com.biz.crm.sfa.business.attendance.local.service.AttendanceRuleSpecialDateService;
import com.biz.crm.sfa.business.attendance.sdk.constant.AttendanceConstant;
import com.biz.crm.sfa.business.attendance.sdk.dto.AttendanceRuleDto;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceSpecialDateType;
import com.biz.crm.business.common.sdk.utils.DateTimeValidateUtil;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 考勤规则特殊日期表服务实现类
 *
 * @author ning.zhang
 * @date 2022-05-05 09:44:48
 */
@Slf4j
@Service("attendanceRuleSpecialDateService")
public class AttendanceRuleSpecialDateServiceImpl implements AttendanceRuleSpecialDateService {

  @Autowired
  private AttendanceRuleSpecialDateRepository attendanceRuleSpecialDateRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Override
  @Transactional
  public void update(AttendanceRuleDto dto) {
    Validate.notBlank(dto.getId(), "考勤规则ID不能为空");
    this.attendanceRuleSpecialDateRepository.deleteByRuleId(dto.getId());
    if (CollectionUtils.isEmpty(dto.getSpecialDateList())) {
      return;
    }
    this.updateValidation(dto);
    List<AttendanceRuleSpecialDateEntity> entities = dto.getSpecialDateList().stream().map(specialDateDto -> {
      AttendanceRuleSpecialDateEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(specialDateDto, AttendanceRuleSpecialDateEntity.class, HashSet.class, ArrayList.class);
      entity.setRuleId(dto.getId());
      return entity;
    }).collect(Collectors.toList());
    this.attendanceRuleSpecialDateRepository.saveBatch(entities);
  }

  /**
   * 在修改attendanceRuleSpecialDate模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(AttendanceRuleDto dto) {
    List<String> mustClockDates = Lists.newArrayList();
    List<String> notClockDates = Lists.newArrayList();
    dto.getSpecialDateList().forEach(specialDateDto -> {
      Validate.notBlank(specialDateDto.getSpecialDate(), "缺失特殊日期");
      Validate.notBlank(specialDateDto.getSpecialDateType(), "缺失特殊日期类型");
      Validate.isTrue(DateTimeValidateUtil.validateDate(specialDateDto.getSpecialDate()),"非法的特殊日期格式，可用的格式：[yyyy-MM-dd]");
      if (AttendanceSpecialDateType.MUST_CLOCK.getDictCode().equals(specialDateDto.getSpecialDateType())) {
        mustClockDates.add(specialDateDto.getSpecialDate());
      }
      if (AttendanceSpecialDateType.NOT_CLOCK.getDictCode().equals(specialDateDto.getSpecialDateType())) {
        notClockDates.add(specialDateDto.getSpecialDate());
      }
    });
    mustClockDates.retainAll(notClockDates);
    Validate.isTrue(CollectionUtils.isEmpty(mustClockDates), "必打卡日期和不必打卡日期重复");
  }
}
