package com.biz.crm.sfa.business.client.local.service.internal;


import cn.hutool.core.bean.BeanUtil;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.sfa.business.client.local.entity.ClientPotential;
import com.biz.crm.sfa.business.client.local.repository.ClientPotentialRepository;
import com.biz.crm.sfa.business.client.local.service.ClientPotentialService;
import com.biz.crm.sfa.business.client.sdk.constant.ClientPotentialConstant;
import com.biz.crm.sfa.business.client.sdk.dto.ClientPotentialPageDto;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * 潜在客户表(ClientPotential)表服务实现类
 *
 * @author songjingen
 * @since 2022-07-29 09:56:25
 */
@Service("clientPotentialService")
public class ClientPotentialServiceImpl implements ClientPotentialService {

  @Autowired
  private ClientPotentialRepository clientPotentialRepository;
  @Autowired
  private GenerateCodeService generateCodeService;

  /**
   * 分页查询数据
   *
   * @param pageable 分页对象
   * @param dto      实体对象
   * @return
   */
  @Override
  public Page<ClientPotential> findByConditions(Pageable pageable, ClientPotentialPageDto dto) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(dto)) {
      dto = new ClientPotentialPageDto();
    }
    return this.clientPotentialRepository.findByConditions(pageable, dto);
  }

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public ClientPotential findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.clientPotentialRepository.getById(id);
  }

  /**
   * 新增数据
   *
   * @param clientPotential 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public ClientPotential create(ClientPotential clientPotential) {
    this.createValidate(clientPotential);
    this.clientPotentialRepository.saveOrUpdate(clientPotential);
    return clientPotential;
  }

  /**
   * 修改新据
   *
   * @param clientPotential 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public ClientPotential update(ClientPotential clientPotential) {
    this.updateValidate(clientPotential);
    ClientPotential current = this.clientPotentialRepository.getById(clientPotential.getId());
    Validate.notNull(current, "未查询到要编辑的数据");
    BeanUtil.copyProperties(clientPotential, current, "id", "delFlag", "enableStatus", "clientCode", "createTime", "createAccount", "createName");
    this.clientPotentialRepository.saveOrUpdate(current);
    return current;
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    this.clientPotentialRepository.updateDelFlagStatusByIds(idList, DelFlagStatusEnum.DELETE);
  }

  /**
   * 启用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 启用结果
   */
  @Transactional
  @Override
  public void enable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "启用数据时，主键集合不能为空！");
    this.clientPotentialRepository.updateEnableStatusByIds(idList, EnableStatusEnum.ENABLE);
  }

  /**
   * 禁用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 禁用结果
   */
  @Transactional
  @Override
  public void disable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "禁用数据时，主键集合不能为空！");
    this.clientPotentialRepository.updateEnableStatusByIds(idList, EnableStatusEnum.DISABLE);
  }

  @Transactional
  @Override
  public void createBatch(List<ClientPotential> clientPotentials) {
    Validate.isTrue(!CollectionUtils.isEmpty(clientPotentials), "批量新增时，入参集合不能为空！");
    clientPotentials.forEach(this::createValidate);
    this.clientPotentialRepository.saveBatch(clientPotentials);
  }

  @Override
  public List<ClientPotential> findByClientTypeAndClientCodes(String clientType, List<String> clientCodes) {
    if (StringUtils.isBlank(clientType) || CollectionUtils.isEmpty(clientCodes)) {
      return new ArrayList<>(0);
    }
    return this.clientPotentialRepository.findByClientTypeAndClientCodes(clientType, clientCodes);
  }

  /**
   * 创建验证
   *
   * @param clientPotential
   */
  private void createValidate(ClientPotential clientPotential) {
    Validate.notNull(clientPotential, "新增时，对象信息不能为空！");
    clientPotential.setId(null);
    clientPotential.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    clientPotential.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    clientPotential.setTenantCode(TenantUtils.getTenantCode());
    clientPotential.setClientCode(this.generateCodeService.generateCode(ClientPotentialConstant.CLIENT_CODE, 1).get(0));
    Validate.notBlank(clientPotential.getTenantCode(), "新增数据时，租户编号不能为空！");
    Validate.notBlank(clientPotential.getClientCode(), "新增数据时，客户编码不能为空！");
    Validate.notBlank(clientPotential.getClientName(), "新增数据时，客户名称不能为空！");
    Validate.notBlank(clientPotential.getClientType(), "新增数据时，客户类型不能为空！");
    Validate.notBlank(clientPotential.getFullName(), "新增数据时，关联人员姓名不能为空！");
    Validate.notBlank(clientPotential.getUserName(), "新增数据时，关联人员账号不能为空！");

  }

  /**
   * 修改验证
   *
   * @param clientPotential
   */
  private void updateValidate(ClientPotential clientPotential) {
    Validate.notNull(clientPotential, "修改时，对象信息不能为空！");
    Validate.notBlank(clientPotential.getId(), "修改数据时，不能为空！");
    Validate.notBlank(clientPotential.getTenantCode(), "修改数据时，租户编号不能为空！");
    Validate.notBlank(clientPotential.getClientCode(), "修改数据时，客户编码不能为空！");
    Validate.notBlank(clientPotential.getClientName(), "修改数据时，客户名称不能为空！");
    Validate.notBlank(clientPotential.getClientType(), "修改数据时，客户类型不能为空！");
    Validate.notBlank(clientPotential.getFullName(), "修改数据时，关联人员姓名不能为空！");
    Validate.notBlank(clientPotential.getUserName(), "修改数据时，关联人员账号不能为空！");

  }
}

