package com.biz.crm.sfa.business.conclusion.local.service.internal;

import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.biz.crm.business.common.sdk.model.LoginUserDetails;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.sfa.business.conclusion.local.entity.ConclusionCallUserEntity;
import com.biz.crm.sfa.business.conclusion.local.entity.ConclusionEntity;
import com.biz.crm.sfa.business.conclusion.local.entity.ConclusionFileEntity;
import com.biz.crm.sfa.business.conclusion.local.entity.ConclusionScopeEntity;
import com.biz.crm.sfa.business.conclusion.local.repository.ConclusionCallUserRepository;
import com.biz.crm.sfa.business.conclusion.local.repository.ConclusionFileRepository;
import com.biz.crm.sfa.business.conclusion.local.repository.ConclusionRepository;
import com.biz.crm.sfa.business.conclusion.local.repository.ConclusionScopeRepository;
import com.biz.crm.sfa.business.conclusion.local.service.ConclusionService;
import com.biz.crm.sfa.business.conclusion.sdk.dto.ConclusionDto;
import com.biz.crm.sfa.business.conclusion.sdk.dto.ConclusionFileDto;
import com.biz.crm.sfa.business.conclusion.sdk.dto.ConclusionListDto;
import com.biz.crm.sfa.business.conclusion.sdk.enums.ConclusionScopeEnum;
import com.biz.crm.sfa.business.conclusion.sdk.enums.ConclusionTypeEnum;
import com.biz.crm.workflow.sdk.dto.PositionInfoDto;
import com.biz.crm.workflow.sdk.listener.PositionInfoListener;
import com.biz.crm.workflow.sdk.vo.PositionVo;
import com.biz.crm.workflow.sdk.vo.response.PositionInfoResponse;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;

/**
 * 工作总结接口实现类
 *
 * @author jerry7
 */
@Slf4j
@Service
public class ConclusionServiceImpl implements ConclusionService {

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Autowired
  private ConclusionRepository conclusionRepository;

  @Autowired
  private ConclusionFileRepository conclusionFileRepository;

  @Autowired
  private ConclusionScopeRepository conclusionScopeRepository;

  @Autowired
  private LoginUserService loginUserService;

  @Autowired
  private NebulaNetEventClient nebulaNetEventClient;

  @Autowired
  private ConclusionCallUserRepository conclusionCallUserRepository;

  /**
   * 创建工作总结/心得
   * 1-校验数据
   * 2-保存工作总结/心得
   * 3-保存附件
   * 4-保存可见范围
   * 5-保存艾特用户列表
   *
   * @param dto 创建DTO
   * @return 创建结果
   */
  @Override
  @Transactional
  public ConclusionEntity create(ConclusionDto dto) {
    //1.
    this.validateConclusion(dto);
    //2.
    ConclusionEntity conclusionEntity = this.nebulaToolkitService.copyObjectByWhiteList(dto, ConclusionEntity.class, HashSet.class, ArrayList.class);
    this.conclusionRepository.save(conclusionEntity);
    //3.
    if (CollectionUtils.isNotEmpty(dto.getConclusionFileList())) {
      List<ConclusionFileEntity> fileEntities = (List<ConclusionFileEntity>) this.nebulaToolkitService.copyCollectionByWhiteList(dto.getConclusionFileList()
              , ConclusionFileDto.class, ConclusionFileEntity.class, HashSet.class, ArrayList.class);
      fileEntities.forEach(file -> {
        file.setConclusionId(conclusionEntity.getId());
      });
      this.conclusionFileRepository.saveOrUpdateBatch(fileEntities);
    }
    //4.
    if (CollectionUtils.isNotEmpty(dto.getVisibleScope())) {
      List<ConclusionScopeEntity> scopeEntityList = Lists.newArrayList();
      dto.getVisibleScope().forEach(userCode -> {
        ConclusionScopeEntity scopeEntity = new ConclusionScopeEntity();
        scopeEntity.setConclusionId(conclusionEntity.getId());
        scopeEntity.setUserCode(userCode);
        scopeEntityList.add(scopeEntity);
      });
      this.conclusionScopeRepository.saveBatch(scopeEntityList);
    }
    //5.
    if (CollectionUtils.isNotEmpty(dto.getCallUserSet())) {
      List<ConclusionCallUserEntity> callUserEntities = Lists.newArrayList();
      dto.getCallUserSet().forEach(userCode -> {
        ConclusionCallUserEntity callUser = new ConclusionCallUserEntity();
        callUser.setConclusionId(conclusionEntity.getId());
        callUser.setUserCode(userCode);
        callUserEntities.add(callUser);
      });
      this.conclusionCallUserRepository.saveBatch(callUserEntities);
    }
    return conclusionEntity;
  }


  /**
   * 工作总结创建数据校验
   * 1- 验证工作总结类别
   * 2- 验证附件
   * 3- 验证可见范围
   * 4- 验证文本
   * 5- 验证当前时间是否已经创建过工作总结
   * 6- 填充创建人信息
   *
   * @param dto 工作总结Dto
   */
  private void validateConclusion(ConclusionDto dto) {
    Validate.notNull(dto, "工作总结实体不能为空");
    // 1.
    Validate.notEmpty(dto.getConclusionType(), "工作总结类型不能为空！");
    ConclusionTypeEnum typeEnum = ConclusionTypeEnum.getByKey(dto.getConclusionType());
    Validate.notNull(typeEnum, "工作总结类别不正确，请检查");
    //2.
    if (CollectionUtils.isNotEmpty(dto.getConclusionFileList())) {
      Validate.isTrue(dto.getConclusionFileList().size() < 6, "附件数量超出限制");
    }
    //3.
    Validate.notNull(dto.getVisibleScopeType(), "工作总结可见范围不能为空");
    ConclusionScopeEnum scopeEnum = ConclusionScopeEnum.getByDictCode(dto.getVisibleScopeType());
    Validate.notNull(scopeEnum, "可见范围类别不正确，请检查");
    if (scopeEnum == ConclusionScopeEnum.CONCLUSION_TYPE_COLLEAGUE) {
      Validate.isTrue(dto.getVisibleScope().size() > 0, "自定义可见范围人数不能为0");
    }
    //4.
    Validate.notNull(dto.getConclusionContent(), "总结/心得内容不能为空！");
    Validate.isTrue(dto.getConclusionContent().length() < 201, "总结/心得内容超过200！");
    if (StringUtils.isNotEmpty(dto.getWorkPlan())) {
      Validate.isTrue(dto.getWorkPlan().length() < 201, "明日计划内容超过200！");
    }
    //5.如果是非心得类工作总结，校验时间段内重复情况
    LoginUserDetails loginUserDetails = this.loginUserService.getLoginDetails(LoginUserDetails.class);
    if (!ConclusionTypeEnum.CONCLUSION_TYPE_EXPERIENCE.getDictCode().equals(dto.getConclusionType())) {
      ConclusionListDto conclusionListDto = new ConclusionListDto();
      conclusionListDto.setCreateName(loginUserDetails.getUsername());
      conclusionListDto.setCreateTimeStart(this.getCreateTimeStart(dto.getConclusionType()));
      conclusionListDto.setCreateTimeEnd(this.getCreateTimeEnd(dto.getConclusionType()));
      conclusionListDto.setConclusionType(dto.getConclusionType());
      List<ConclusionEntity> conclusionEntities = this.conclusionRepository.findByTimesAndType(conclusionListDto);
      Validate.isTrue(CollectionUtils.isEmpty(conclusionEntities), "当前时间段内已提交工作总结！");
    }
    //6.
    dto.setCreateAccount(loginUserDetails.getAccount());
    dto.setCreateTime(DateUtil.date());
    dto.setCreateName(loginUserDetails.getUsername());
    dto.setOrgCode(loginUserDetails.getOrgCode());
    dto.setOrgName(loginUserDetails.getOrgName());
    dto.setPostCode(loginUserDetails.getPostCode());
    dto.setPostName(loginUserDetails.getPostName());
    //获取职位详情，以取得上级职位编码
    SerializableBiConsumer<PositionInfoListener, PositionInfoDto> sf = PositionInfoListener::onFindByIdsOrCodes;
    PositionInfoResponse response = (PositionInfoResponse) nebulaNetEventClient.directPublish(PositionInfoDto.builder().roleCodes(Lists.newArrayList(loginUserDetails.getPostCode())).build(), PositionInfoListener.class, sf);
    if (ObjectUtils.isNotEmpty(response) && CollectionUtils.isNotEmpty(response.getPositionVos())) {
      PositionVo positionVo = response.getPositionVos().get(0);
      dto.setParentPostCode(positionVo.getParentCode());
    }
  }

  /**
   * 获取当前日期所在类型的开始时间
   *
   * @param conclusionTypeEnum 工作总结类型
   * @return 开始时间
   */
  private Date getCreateTimeStart(String conclusionTypeEnum) {
    ConclusionTypeEnum typeEnum = ConclusionTypeEnum.getByKey(conclusionTypeEnum);
    if (ObjectUtils.isEmpty(typeEnum)) {
      return null;
    }
    switch (typeEnum) {
      case CONCLUSION_TYPE_DAILY:
        return DateUtil.beginOfDay(DateUtil.date());
      case CONCLUSION_TYPE_WEEKLY:
        return DateUtil.beginOfWeek(DateUtil.date());
      case CONCLUSION_TYPE_MONTHLY:
        return DateUtil.beginOfMonth(DateUtil.date());
      default:
        return null;
    }
  }

  /**
   * 获取当前日期所在类型的结束时间
   *
   * @param conclusionTypeEnum 工作总结类型
   * @return 结束时间
   */
  private Date getCreateTimeEnd(String conclusionTypeEnum) {
    ConclusionTypeEnum typeEnum = ConclusionTypeEnum.getByKey(conclusionTypeEnum);
    if (ObjectUtils.isEmpty(typeEnum)) {
      return null;
    }
    switch (typeEnum) {
      case CONCLUSION_TYPE_DAILY:
        return DateUtil.endOfDay(DateUtil.date());
      case CONCLUSION_TYPE_WEEKLY:
        return DateUtil.endOfWeek(DateUtil.date());
      case CONCLUSION_TYPE_MONTHLY:
        return DateUtil.endOfMonth(DateUtil.date());
      default:
        return null;
    }
  }
}
