package com.biz.crm.sfa.business.data.sample.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.sfa.business.data.sample.local.entity.DataSampleEntity;
import com.biz.crm.sfa.business.data.sample.local.model.DataSampleConditionModel;
import com.biz.crm.sfa.business.data.sample.local.repository.DataSampleRepository;
import com.biz.crm.sfa.business.data.sample.local.service.DataSamplePictureService;
import com.biz.crm.sfa.business.data.sample.local.service.DataSampleService;
import com.biz.crm.sfa.business.data.sample.sdk.dto.DataSampleDto;
import com.biz.crm.sfa.business.data.sample.sdk.dto.DataSamplePageDto;
import com.biz.crm.sfa.business.data.sample.sdk.event.DataSampleEventListener;
import com.biz.crm.sfa.business.data.sample.sdk.vo.DataSampleVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;

/**
 * 数据示例表服务实现类
 *
 * @author ning.zhang
 * @date 2022-07-27 16:15:09
 */
@Slf4j
@Service("dataSampleService")
public class DataSampleServiceImpl implements DataSampleService {

  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private DataSampleRepository dataSampleRepository;
  @Autowired(required = false)
  private List<DataSampleEventListener> listeners;
  @Autowired
  private DataSamplePictureService dataSamplePictureService;

  @Override
  @Transactional
  public DataSampleEntity create(DataSampleDto dto) {
    this.createValidation(dto);
    DataSampleEntity entity = this.nebulaToolkitService.copyObjectByBlankList(dto, DataSampleEntity.class, HashSet.class, ArrayList.class);
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    this.dataSampleRepository.save(entity);
    //绑定关联数据
    this.dataSamplePictureService.update(entity);
    //新增数据示例事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      DataSampleVo vo = this.nebulaToolkitService.copyObjectByBlankList(entity, DataSampleVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(listener -> {
        listener.onCreate(vo);
      });
    }
    return entity;
  }

  @Override
  @Transactional
  public DataSampleEntity update(DataSampleDto dto) {
    this.updateValidation(dto);
    DataSampleConditionModel model = new DataSampleConditionModel();
    model.setIds(Sets.newHashSet(dto.getId()));
    model.setTenantCode(dto.getTenantCode());
    List<DataSampleEntity> entities = this.dataSampleRepository.findByDataSampleConditionModel(model);
    Validate.isTrue(!CollectionUtils.isEmpty(entities), "数据示例信息不存在");
    DataSampleEntity oldEntity = entities.get(0);
    Validate.isTrue(oldEntity.getSampleCode().equals(dto.getSampleCode()), "示例编码不能编辑");
    DataSampleEntity updateEntity = this.nebulaToolkitService.copyObjectByBlankList(dto, DataSampleEntity.class, HashSet.class, ArrayList.class);
    this.dataSampleRepository.updateById(updateEntity);
    //绑定关联数据
    this.dataSamplePictureService.update(updateEntity);
    //更新数据示例事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      DataSampleVo oldVo = this.nebulaToolkitService.copyObjectByBlankList(oldEntity, DataSampleVo.class, HashSet.class, ArrayList.class);
      DataSampleVo newVo = this.nebulaToolkitService.copyObjectByBlankList(updateEntity, DataSampleVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(listener -> {
        listener.onUpdate(oldVo, newVo);
      });
    }
    return updateEntity;
  }

  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    List<DataSampleEntity> entities = this.dataSampleRepository.listByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据删除个数不匹配");
    this.dataSampleRepository.updateDelFlagByIds(ids);
    //删除数据示例事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      List<DataSampleVo> voList = (List<DataSampleVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, DataSampleEntity.class
          , DataSampleVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(listener -> {
        listener.onDelete(voList);
      });
    }
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    DataSampleConditionModel model = new DataSampleConditionModel();
    model.setIds(Sets.newHashSet(ids));
    model.setTenantCode(TenantUtils.getTenantCode());
    List<DataSampleEntity> entities = this.dataSampleRepository.findByDataSampleConditionModel(model);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据启用个数不匹配");
    this.dataSampleRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
    //启用数据示例事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      List<DataSampleVo> voList = (List<DataSampleVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, DataSampleEntity.class
          , DataSampleVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(listener -> {
        listener.onEnable(voList);
      });
    }
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    DataSampleConditionModel model = new DataSampleConditionModel();
    model.setIds(Sets.newHashSet(ids));
    model.setTenantCode(TenantUtils.getTenantCode());
    List<DataSampleEntity> entities = this.dataSampleRepository.findByDataSampleConditionModel(model);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据禁用个数不匹配");
    this.dataSampleRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
    //启用数据示例事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      List<DataSampleVo> voList = (List<DataSampleVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, DataSampleEntity.class
          , DataSampleVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(listener -> {
        listener.onDisable(voList);
      });
    }
  }

  @Override
  public Page<DataSampleEntity> findByConditions(Pageable pageable, DataSamplePageDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new DataSamplePageDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    return this.dataSampleRepository.findByConditions(pageable, dto);
  }

  @Override
  public DataSampleEntity findById(String id) {
    if (Objects.isNull(id)) {
      return null;
    }
    DataSampleConditionModel model = new DataSampleConditionModel();
    model.setIds(Sets.newHashSet(id));
    model.setTenantCode(TenantUtils.getTenantCode());
    List<DataSampleEntity> entities = this.dataSampleRepository.findByDataSampleConditionModel(model);
    return CollectionUtils.isEmpty(entities) ? null : entities.get(0);
  }


  /**
   * 在创建DataSample模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(DataSampleDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getSampleName(), "缺失示例名称");
    Validate.notBlank(dto.getSampleType(), "缺失示例类型");
    Validate.notBlank(dto.getSampleCode(), "缺失示例编码");
    Validate.isTrue(Objects.nonNull(dto.getSortIndex()) && dto.getSortIndex() >= 0, "错误的排序");
    DataSampleConditionModel model = new DataSampleConditionModel();
    model.setSampleCodes(Sets.newHashSet(dto.getSampleCode()));
    model.setTenantCode(dto.getTenantCode());
    model.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    List<DataSampleEntity> entities = this.dataSampleRepository.findByDataSampleConditionModel(model);
    Validate.isTrue(CollectionUtils.isEmpty(entities), "示例编码已存在");
  }

  /**
   * 在修改DataSample模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(DataSampleDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getId(), "修改信息时，id不能为空！");
    Validate.notBlank(dto.getSampleName(), "缺失示例名称");
    Validate.notBlank(dto.getSampleType(), "缺失示例类型");
    Validate.notBlank(dto.getSampleCode(), "缺失示例编码");
    Validate.isTrue(Objects.nonNull(dto.getSortIndex()) && dto.getSortIndex() >= 0, "错误的排序");
  }
}
