package com.biz.crm.sfa.business.help.defense.local.service.internal;


import com.biz.crm.sfa.business.help.defense.local.entity.HelpDefensePlanDetail;
import com.biz.crm.sfa.business.help.defense.local.entity.HelpDefensePlanDetailException;
import com.biz.crm.sfa.business.help.defense.local.repository.HelpDefensePlanDetailRepository;
import com.biz.crm.sfa.business.help.defense.local.service.HelpDefensePlanDetailExceptionService;
import com.biz.crm.sfa.business.help.defense.local.service.HelpDefensePlanDetailService;
import com.biz.crm.sfa.business.help.defense.sdk.enums.HelpStatusEnum;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 协访计划明细表(HelpDefensePlanDetail)表服务实现类
 *
 * @author songjingen
 * @since 2022-06-07 15:43:21
 */
@Service("helpDefensePlanDetailService")
public class HelpDefensePlanDetailServiceImpl implements HelpDefensePlanDetailService {

  @Autowired
  private HelpDefensePlanDetailRepository helpDefensePlanDetailRepository;
  @Autowired
  private HelpDefensePlanDetailExceptionService helpDefensePlanDetailExceptionService;

  /**
   * 新增数据
   *
   * @param helpDefensePlanDetails 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public void createBatch(List<HelpDefensePlanDetail> helpDefensePlanDetails) {
    Validate.isTrue(!CollectionUtils.isEmpty(helpDefensePlanDetails), "批量新增协访计划明细时，入参集合为空！");
    helpDefensePlanDetails.forEach(this::createValidate);
    this.helpDefensePlanDetailRepository.saveBatch(helpDefensePlanDetails);
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    this.helpDefensePlanDetailRepository.removeByIds(idList);
  }

  @Override
  public Map<String, Integer> findScheduleByUserNameAndVisitDate(String userName, Date visitDate) {
    if (StringUtils.isBlank(userName) || visitDate == null) {
      return new HashMap<>(0);
    }
    List<HelpDefensePlanDetail> totalList = this.helpDefensePlanDetailRepository.findByUserNameAndVisitDate(userName, visitDate);
    if (CollectionUtils.isEmpty(totalList)) {
      return new HashMap<>(0);
    }
    //筛选已完成数据
    List<HelpDefensePlanDetail> completedList = totalList.stream().filter(visitPlanDetail -> HelpStatusEnum.COMPLETE.getDictCode().equals(visitPlanDetail.getHelpStatus())).collect(Collectors.toList());
    HashMap<String, Integer> hashMap = new HashMap<>(2);
    hashMap.put("completed", CollectionUtils.isEmpty(completedList) ? 0 : completedList.size());
    hashMap.put("total", totalList.size());
    return hashMap;
  }

  @Override
  public List<HelpDefensePlanDetail> findByHelpUserNameAndHelpDate(String helpUserName, Date helpDefenseDate) {
    if (StringUtils.isAnyBlank(helpUserName) || helpDefenseDate == null) {
      return null;
    }
    return this.helpDefensePlanDetailRepository.findByUserNameAndVisitDate(helpUserName, helpDefenseDate);
  }

  @Transactional
  @Override
  @DynamicTaskService(cornExpression = "0 5 0 * * ?", taskDesc = "协访计划定时任务")
  public void executeTask() {
    try {
      Date before = DateUtils.addDays(new Date(), -1);
      String beforeStr = DateFormatUtils.format(before, "yyyy-MM-dd");
      Date beforeDate = DateUtils.parseDate(beforeStr, "yyyy-MM-dd");
      List<HelpDefensePlanDetail> list = this.helpDefensePlanDetailRepository.findByHelpStatusAndHelpDate(HelpStatusEnum.IN.getDictCode(), beforeDate);
      if (CollectionUtils.isEmpty(list)) {
        return;
      }
      ArrayList<HelpDefensePlanDetailException> exceptions = new ArrayList<>();
      for (HelpDefensePlanDetail visitPlanDetail : list) {
        visitPlanDetail.setHelpStatus(HelpStatusEnum.EXCEPTION.getDictCode());
        HelpDefensePlanDetailException exception = new HelpDefensePlanDetailException();
        exception.setHelpPlanDetailId(visitPlanDetail.getId());
        exception.setRemark("计划已过期，更新为异常！");
        exceptions.add(exception);
      }
      //更新拜访状态
      this.helpDefensePlanDetailRepository.saveOrUpdateBatch(list);
      //增加异常记录
      this.helpDefensePlanDetailExceptionService.createBatch(exceptions);
    } catch (ParseException e) {
      e.printStackTrace();
    }
  }

  @Override
  public List<HelpDefensePlanDetail> findByPlanCodes(Set<String> planCodes) {
    if (CollectionUtils.isEmpty(planCodes)) {
      return new ArrayList<>(0);
    }
    return this.helpDefensePlanDetailRepository.findByPlanCodes(planCodes);
  }

  /**
   * 创建验证
   *
   * @param helpDefensePlanDetail
   */
  private void createValidate(HelpDefensePlanDetail helpDefensePlanDetail) {
    Validate.notNull(helpDefensePlanDetail, "新增时，对象信息不能为空！");
    helpDefensePlanDetail.setId(null);
    Validate.notBlank(helpDefensePlanDetail.getClientCode(), "新增数据时，客户编码不能为空！");
    Validate.notBlank(helpDefensePlanDetail.getClientType(), "新增数据时，客户类型不能为空！");
    Validate.notNull(helpDefensePlanDetail.getHelpDate(), "新增数据时，协访日期不能为空！");
    Validate.notBlank(helpDefensePlanDetail.getHelpPlanCode(), "新增数据时，协访计划编码不能为空！");
    Validate.notBlank(helpDefensePlanDetail.getHelpStatus(), "新增数据时，协访状态不能为空！");

  }
}

