package com.biz.crm.sfa.business.help.defense.local.service.internal;


import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.user.sdk.service.UserFeignVoService;
import com.biz.crm.mdm.business.user.sdk.vo.UserVo;
import com.biz.crm.sfa.business.help.defense.local.entity.HelpDefense;
import com.biz.crm.sfa.business.help.defense.local.entity.HelpDefenseClientInfo;
import com.biz.crm.sfa.business.help.defense.local.entity.HelpDefensePlanDetail;
import com.biz.crm.sfa.business.help.defense.local.repository.HelpDefenseRepository;
import com.biz.crm.sfa.business.help.defense.local.service.HelpDefenseClientInfoService;
import com.biz.crm.sfa.business.help.defense.local.service.HelpDefensePlanDetailService;
import com.biz.crm.sfa.business.help.defense.local.service.HelpDefenseService;
import com.biz.crm.sfa.business.help.defense.sdk.constant.HelpDefenseConstant;
import com.biz.crm.sfa.business.help.defense.sdk.enums.HelpStatusEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 协访计划制定表(HelpDefense)表服务实现类
 *
 * @author songjingen
 * @since 2022-05-19 11:21:57
 */
@Service("helpDefenseService")
public class HelpDefenseServiceImpl implements HelpDefenseService {

  @Autowired
  private HelpDefenseRepository helpDefenseRepository;

  @Autowired
  private GenerateCodeService generateCodeService;

  @Autowired
  private HelpDefenseClientInfoService helpDefenseClientInfoService;

  @Autowired
  private HelpDefensePlanDetailService helpDefensePlanDetailService;

  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Autowired
  private UserFeignVoService userFeignVoService;

  /**
   * 新增数据
   *
   * @param helpDefenses 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public void createBatch(List<HelpDefense> helpDefenses) {
    Validate.isTrue(!CollectionUtils.isEmpty(helpDefenses), "批量新增时，入参集合不能为空！");
    //验证协访用户和被协防用户的在当前时间是否有协防计划明细，如果有则不进行新增，如果没有则不进行
    this.validateHelpPlanExist(helpDefenses);
    List<HelpDefenseClientInfo> infos = new ArrayList<>();
    List<HelpDefensePlanDetail> planDetails = new ArrayList<>();
    helpDefenses.stream().forEach(helpDefense -> {
      this.createValidate(helpDefense);
      List<HelpDefenseClientInfo> clientInfos = helpDefense.getClientInfos();
      if (!CollectionUtils.isEmpty(clientInfos)) {
        for (HelpDefenseClientInfo clientInfo : clientInfos) {
          clientInfo.setHelpPlanCode(helpDefense.getHelpPlanCode());
          HelpDefensePlanDetail helpDefensePlanDetail = this.nebulaToolkitService.copyObjectByWhiteList(clientInfo, HelpDefensePlanDetail.class, HashSet.class, ArrayList.class);
          helpDefensePlanDetail.setHelpDate(helpDefense.getHelpDefenseDate());
          helpDefensePlanDetail.setHelpStatus(HelpStatusEnum.NO.getDictCode());
          infos.add(clientInfo);
          planDetails.add(helpDefensePlanDetail);
        }
      }
    });
    this.helpDefenseRepository.saveBatch(helpDefenses);
    //保存关联表数据
    this.helpDefenseClientInfoService.createBatch(infos);
    //生成协访计划明细
    this.helpDefensePlanDetailService.createBatch(planDetails);
  }

  /**
   * 修改新据
   *
   * @param object 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public void update(JSONObject object) {
    Validate.notNull(object, "编辑时，入参对象不能为空！");
    String id = object.getString("id");
    Validate.notBlank(id, "编辑时，id不能为空！");
    JSONArray helpDefensesArr = object.getJSONArray("helpDefenses");
    Validate.isTrue(!CollectionUtils.isEmpty(helpDefensesArr), "编辑时，协访计划数据集合不能为空！");
    List<HelpDefense> helpDefenses = JSONArray.parseArray(helpDefensesArr.toJSONString(), HelpDefense.class);
    HelpDefense defense = this.helpDefenseRepository.getById(id);
    Validate.notNull(defense, "编辑时，未查询到数据！");
    //删除明细数据
    List<HelpDefensePlanDetail> defenseList = this.helpDefensePlanDetailService.findByHelpUserNameAndHelpDate(defense.getHelpUserName(), defense.getHelpDefenseDate());
    if (!CollectionUtils.isEmpty(defenseList)) {
      //验证是否存在非未拜访的数据
      this.validateHelpDetailAllStatusNo(helpDefenses, defenseList);
      List<String> ids = defenseList.stream().map(HelpDefensePlanDetail::getId).collect(Collectors.toList());
      this.helpDefensePlanDetailService.delete(ids);
    }
    //先根据当前id的拜访时间和协访人删除数据，在创建新数据
    this.helpDefenseRepository.updateDelFlagByHelpDefenseDateAndHelpUserName(defense.getHelpDefenseDate(), defense.getHelpUserName(), DelFlagStatusEnum.DELETE);
    this.createBatch(helpDefenses);
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    List<HelpDefense> list = this.helpDefenseRepository.findByIds(idList);
    Validate.isTrue(!CollectionUtils.isEmpty(list), "删除数据时，未查询到相关的数据！");
    Set<String> planCodes = list.stream().map(HelpDefense::getHelpPlanCode).collect(Collectors.toSet());
    List<HelpDefensePlanDetail> defensePlanDetails = this.helpDefensePlanDetailService.findByPlanCodes(planCodes);
    if (!CollectionUtils.isEmpty(defensePlanDetails)) {
      this.validateHelpDetailAllStatusNo(list, defensePlanDetails);
    }
    this.helpDefenseRepository.removeByIds(idList);
  }

  /**
   * 启用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 启用结果
   */
  @Transactional
  @Override
  public void enable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "启用数据时，主键集合不能为空！");
    this.helpDefenseRepository.updateEnableStatusByIds(idList, EnableStatusEnum.ENABLE);
  }

  /**
   * 禁用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 禁用结果
   */
  @Transactional
  @Override
  public void disable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "禁用数据时，主键集合不能为空！");
    this.helpDefenseRepository.updateEnableStatusByIds(idList, EnableStatusEnum.DISABLE);
  }

  /**
   * 创建验证
   *
   * @param helpDefense
   */
  private void createValidate(HelpDefense helpDefense) {
    Validate.notNull(helpDefense, "新增时，对象信息不能为空！");
    helpDefense.setId(null);
    helpDefense.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    helpDefense.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    helpDefense.setTenantCode(TenantUtils.getTenantCode());
    helpDefense.setHelpDefenseYearMonth(DateUtil.format(helpDefense.getHelpDefenseDate(), DateUtil.newSimpleFormat("yyyy-MM")));
    helpDefense.setHelpPlanCode(this.generateCodeService.generateCode(HelpDefenseConstant.HELP_PLAN_CODE, 1).get(0));
    //设置协访人信息
    List<UserVo> userVoList = this.userFeignVoService.findByUserNames(Lists.newArrayList(helpDefense.getHelpUserName()));
    if (!CollectionUtils.isEmpty(userVoList)) {
      UserVo userVo = userVoList.stream().findFirst().get();
      helpDefense.setHelpPosCode(userVo.getPositionCode());
      helpDefense.setHelpPosName(userVo.getPositionName());
      helpDefense.setHelpOrgCode(userVo.getOrgCode());
      helpDefense.setHelpOrgName(userVo.getOrgName());
    }
    //设置被协访人信息
    List<UserVo> coverUserVoList = this.userFeignVoService.findByUserNames(Lists.newArrayList(helpDefense.getCoverHelpUserName()));
    if (!CollectionUtils.isEmpty(coverUserVoList)) {
      UserVo userVo = coverUserVoList.stream().findFirst().get();
      helpDefense.setCoverHelpPosCode(userVo.getPositionCode());
      helpDefense.setCoverHelpPosName(userVo.getPositionName());
      helpDefense.setCoverHelpOrgCode(userVo.getOrgCode());
      helpDefense.setCoverHelpOrgName(userVo.getOrgName());
    }
    Validate.notBlank(helpDefense.getTenantCode(), "新增数据时，租户编号不能为空！");
    Validate.notBlank(helpDefense.getCoverHelpUserName(), "新增数据时，被协访人员账号不能为空！");
    Validate.notNull(helpDefense.getHelpDefenseDate(), "新增数据时，协访日期不能为空！");
    Validate.notBlank(helpDefense.getHelpPlanCode(), "新增数据时，协访计划编码不能为空！");
    Validate.notBlank(helpDefense.getHelpUserName(), "新增数据时，拜访计划编码不能为空！");
    Validate.isTrue(!CollectionUtils.isEmpty(helpDefense.getClientInfos()), "新增数据时，客户信息不能为空！");
    //唯一性验证（协访人员账号+被协访人员账号+协访日期）
    HelpDefense helpDefenseData = this.helpDefenseRepository.findByUserNameAndHelpDefenseDate(TenantUtils.getTenantCode(), helpDefense.getHelpUserName(), helpDefense.getCoverHelpUserName(), helpDefense.getHelpDefenseDate());
    Validate.isTrue(helpDefenseData == null, "新增数据时，协访人员账号：%s，被协访人员账号：%s，协访日期：%S，的数据已存在！", helpDefense.getHelpUserName(), helpDefense.getCoverHelpUserName(), DateFormatUtils.format(helpDefense.getHelpDefenseDate(), "yyyy-MM-dd"));
    try {
      String nowDateStr = DateFormatUtils.format(new Date(), "yyyy-MM-dd");
      Date nowDate = DateUtils.parseDate(nowDateStr, "yyyy-MM-dd");
      Validate.isTrue(helpDefense.getHelpDefenseDate().compareTo(nowDate) >= 0 && helpDefense.getHelpDefenseDate().compareTo(DateUtils.addDays(nowDate, 7)) <= 0, "新增数据时, 拜访时间只能设置未来7天内！");
    } catch (ParseException e) {
      e.printStackTrace();
    }

  }


  /**
   * 验证协访用户和被协防用户的在当前时间是否有协防计划明细，如果有则不进行新增，如果没有则不进行
   */
  private void validateHelpPlanExist(List<HelpDefense> helpDefenses) {
    Set<String> userNames = new HashSet<>();
    Set<String> covertUserNames = new HashSet<>();
    Set<Date> helpDates = new HashSet<>();
    for (HelpDefense helpDefense : helpDefenses) {
      userNames.add(helpDefense.getHelpUserName());
      covertUserNames.add(helpDefense.getCoverHelpUserName());
      helpDates.add(helpDefense.getHelpDefenseDate());
    }
    List<HelpDefense> defenses = this.helpDefenseRepository.findByHelpUserNamesAndCovertUserNamesAndHelpDates(userNames, covertUserNames, helpDates);
    if (!CollectionUtils.isEmpty(defenses)) {
      //k-协防人_被协防人_时间，v-数据集合
      Map<String, List<HelpDefense>> listMap = defenses.stream().collect(Collectors.groupingBy(helpDefense -> StringUtils.joinWith("_", helpDefense.getHelpUserName(), helpDefense.getCoverHelpUserName(), helpDefense.getHelpDefenseDate())));
      for (HelpDefense helpDefens : helpDefenses) {
        String key = StringUtils.joinWith("_", helpDefens.getHelpUserName(), helpDefens.getCoverHelpUserName(), helpDefens.getHelpDefenseDate());
        Validate.isTrue(CollectionUtils.isEmpty(listMap.get(key)), "协防人【%s】配置的被协防人【%s】在【%s】存在协访计划！", helpDefens.getHelpRealName(), helpDefens.getCoverHelpRealName(), DateFormatUtils.format(helpDefens.getHelpDefenseDate(), "yyyy-MM-dd"));
      }
    }
  }

  /**
   * 验证协访计划明细是否非未拜访的数据
   *
   * @param helpDefenses
   * @param defenseList
   */
  private void validateHelpDetailAllStatusNo(List<HelpDefense> helpDefenses, List<HelpDefensePlanDetail> defenseList) {
    List<HelpDefensePlanDetail> notNoDetails = defenseList.stream().filter(helpDefensePlanDetail -> !helpDefensePlanDetail.getHelpStatus().equals(HelpStatusEnum.NO.getDictCode())).collect(Collectors.toList());
    if (CollectionUtils.isEmpty(notNoDetails)) {
      return;
    }
    Set<String> stringSet = notNoDetails.stream().map(HelpDefensePlanDetail::getHelpPlanCode).collect(Collectors.toSet());
    List<HelpDefense> list = this.helpDefenseRepository.findByPlanCodes(stringSet);
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    //k-协防人_被协防人_时间，v-数据集合
    Map<String, List<HelpDefense>> listMap = list.stream().collect(Collectors.groupingBy(helpDefense -> StringUtils.joinWith("_", helpDefense.getHelpUserName(), helpDefense.getCoverHelpUserName(), helpDefense.getHelpDefenseDate())));
    for (HelpDefense helpDefens : helpDefenses) {
      String key = StringUtils.joinWith("_", helpDefens.getHelpUserName(), helpDefens.getCoverHelpUserName(), helpDefens.getHelpDefenseDate());
      Validate.isTrue(CollectionUtils.isEmpty(listMap.get(key)), "当前存在拜访任务，不允许编辑或者删除！");
    }
  }

}

