package com.biz.crm.sfa.business.holiday.local.repository;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.sfa.business.holiday.local.entity.HolidayEntity;
import com.biz.crm.sfa.business.holiday.local.mapper.HolidayMapper;
import com.biz.crm.sfa.business.holiday.sdk.dto.HolidayConditionDto;
import com.biz.crm.sfa.business.holiday.sdk.dto.HolidayPageDto;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * 节假日表的数据库访问类 {@link HolidayEntity}
 *
 * @author ning.zhang
 * @date 2022-05-11 14:48:54
 */
@Component
public class HolidayRepository extends ServiceImpl<HolidayMapper, HolidayEntity> {

  /**
   * 通过节假日ID集合获取节假日信息
   *
   * @param ids 节假日ID集合
   * @return 节假日信息
   */
  public List<HolidayEntity> findByIds(List<String> ids) {
    return this.baseMapper.findByIds(ids);
  }

  /**
   * 通过年份查询节假日信息
   *
   * @param year       年份
   * @param tenantCode 租户编码
   * @return 节假日信息
   */
  public HolidayEntity findByYear(String year, String tenantCode) {
    return this.lambdaQuery()
        .eq(HolidayEntity::getYear, year)
        .eq(HolidayEntity::getTenantCode, tenantCode)
        .one();
  }

  /**
   * 通过日期获取节假日信息
   *
   * @param date       日期
   * @param tenantCode 租户编码
   * @return 节假日信息
   */
  public List<HolidayEntity> findByDate(String date, String tenantCode) {
    return this.baseMapper.findByDate(date, tenantCode);
  }

  /**
   * 通过节假日条件获取节假日信息
   *
   * @param dto 参数dto
   * @return 节假日信息
   */
  public List<HolidayEntity> findByHolidayConditionDto(HolidayConditionDto dto) {
    return this.baseMapper.findByHolidayConditionDto(dto);
  }

  /**
   * 批量删除
   *
   * @param ids ID集合
   */
  public void updateDelFlagByIds(List<String> ids) {
    this.lambdaUpdate()
        .in(HolidayEntity::getId, ids)
        .set(HolidayEntity::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
        .update();
  }

  /**
   * 节假日分页列表
   *
   * @param dto      请求参数dto
   * @param pageable 分页信息
   * @return Page<HolidayEntity> 节假日分页信息
   */
  public Page<HolidayEntity> findByConditions(Pageable pageable, HolidayPageDto dto) {
    Page<HolidayEntity> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.baseMapper.findByConditions(page, dto);
  }


  /**
   * 查询所有
   *
   * @param tenantCode 租户编码
   */
  public List<HolidayEntity> findAll(String tenantCode) {
    return this.lambdaQuery()
        .eq(HolidayEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(HolidayEntity::getTenantCode, tenantCode)
        .orderByAsc(HolidayEntity::getYear)
        .list();
  }
}
