package com.biz.crm.sfa.business.holiday.local.service.internal;

import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.sfa.business.holiday.local.entity.HolidayDateEntity;
import com.biz.crm.sfa.business.holiday.local.entity.HolidayEntity;
import com.biz.crm.sfa.business.holiday.local.repository.HolidayDateRepository;
import com.biz.crm.sfa.business.holiday.local.repository.HolidayRepository;
import com.biz.crm.sfa.business.holiday.local.service.HolidayDateService;
import com.biz.crm.sfa.business.holiday.sdk.dto.HolidayConditionDto;
import com.biz.crm.sfa.business.holiday.sdk.dto.HolidayDateDto;
import com.biz.crm.sfa.business.holiday.sdk.dto.HolidayDto;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

/**
 * 节假日日期表服务实现类
 *
 * @author ning.zhang
 * @date 2022-05-11 14:49:53
 */
@Slf4j
@Service("holidayDateService")
public class HolidayDateServiceImpl implements HolidayDateService {

  @Autowired
  private HolidayDateRepository holidayDateRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private HolidayRepository holidayRepository;

  @Override
  @Transactional
  public List<HolidayDateEntity> update(HolidayDto dto) {
    Validate.notBlank(dto.getId(), "节假日ID不能为空");
    this.holidayDateRepository.deleteByHolidayId(dto.getId());
    this.updateValidation(dto);
    List<HolidayDateEntity> entities = Lists.newArrayList();
    dto.getInfoList().forEach(holidayInfoDto -> {
      holidayInfoDto.getDateList().forEach(holidayDateDto -> {
        HolidayDateEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(holidayDateDto, HolidayDateEntity.class, HashSet.class, ArrayList.class);
        entity.setHolidayId(dto.getId());
        entity.setHolidayName(holidayInfoDto.getHolidayName());
        entities.add(entity);
      });
    });
    this.holidayDateRepository.saveBatch(entities);
    return entities;
  }

  /**
   * 在修改holidayDate模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(HolidayDto dto) {
    Set<String> holidayDates = Sets.newHashSet();
    dto.getInfoList().forEach(holidayInfoDto -> {
      Validate.notBlank(holidayInfoDto.getHolidayName(), "缺失节假日名称");
      Validate.notBlank(holidayInfoDto.getEndDate(), String.format("节日[%s],缺失结束日期", holidayInfoDto.getHolidayName()));
      Validate.notBlank(holidayInfoDto.getStartDate(), String.format("节日[%s],缺失开始日期", holidayInfoDto.getHolidayName()));
      LocalDate endDate = parseDateFormat(holidayInfoDto.getEndDate());
      LocalDate startDate = parseDateFormat(holidayInfoDto.getStartDate());
      Validate.notNull(endDate, String.format("节日[%s],存在非法的结束日期格式，可用的格式：[yyyy-MM-dd]", holidayInfoDto.getHolidayName()));
      Validate.notNull(startDate, String.format("节日[%s],存在非法的开始日期格式，可用的格式：[yyyy-MM-dd]", holidayInfoDto.getHolidayName()));
      Validate.isTrue(holidayInfoDto.getStartDate().substring(0, 4).equals(dto.getYear()), String.format("节日[%s],开始日期不在所选年份内", holidayInfoDto.getHolidayName()));
      Validate.isTrue(holidayInfoDto.getEndDate().substring(0, 4).equals(dto.getYear()), String.format("节日[%s],结束日期不在所选年份内", holidayInfoDto.getHolidayName()));
      Validate.isTrue(holidayInfoDto.getEndDate().compareTo(holidayInfoDto.getStartDate()) >= 0, String.format("节日[%s],开始日期不能大于结束日期", holidayInfoDto.getHolidayName()));
      Validate.isTrue(!CollectionUtils.isEmpty(holidayInfoDto.getDateList()), "缺失节假日日期信息");
      Map<String, HolidayDateDto> holidayDateMap = Maps.newHashMap();
      holidayInfoDto.getDateList().forEach(holidayDateDto -> {
        holidayDateDto.setId(null);
        Validate.notBlank(holidayDateDto.getHolidayDate(), String.format("节日[%s],缺失节日日期", holidayInfoDto.getHolidayName()));
        Validate.notBlank(holidayDateDto.getCalculationSalary(), String.format("节日[%s],缺失是否计算薪资", holidayInfoDto.getHolidayName()));
        Validate.notNull(parseDateFormat(holidayDateDto.getHolidayDate()), String.format("节日[%s],存在非法的节日日期格式，可用的格式：[yyyy-MM-dd]", holidayInfoDto.getHolidayName()));
        Validate.isTrue(holidayInfoDto.getStartDate().compareTo(holidayDateDto.getHolidayDate()) <= 0
                && holidayInfoDto.getEndDate().compareTo(holidayDateDto.getHolidayDate()) >= 0
            , String.format("节日[%s],日期[%s]不在所选范围内", holidayInfoDto.getHolidayName(), holidayDateDto.getHolidayDate()));
        Validate.isTrue(Objects.isNull(holidayDateMap.get(holidayDateDto.getHolidayDate()))
            , String.format("节日[%s],存在重复的节日日期[%s]", holidayInfoDto.getHolidayName(), holidayDateDto.getHolidayDate()));
        dto.setSalaryDays(dto.getSalaryDays() + (BooleanEnum.TRUE.getCapital().equals(holidayDateDto.getCalculationSalary()) ? 1 : 0));
        dto.setHolidayDays(dto.getHolidayDays() + 1);
        holidayDateMap.put(holidayDateDto.getHolidayDate(), holidayDateDto);
      });
      holidayDates.addAll(holidayDateMap.keySet());
    });
    HolidayConditionDto conditionDto = new HolidayConditionDto();
    conditionDto.setTenantCode(TenantUtils.getTenantCode());
    conditionDto.setDates(holidayDates);
    List<HolidayEntity> holidayEntities = this.holidayRepository.findByHolidayConditionDto(conditionDto);
    if (!CollectionUtils.isEmpty(holidayEntities)) {
      throw new IllegalArgumentException(String.format("节日日期[%s]已存在"
          ,holidayEntities.get(0).getDateList().get(0).getHolidayDate()));
    }
  }

  /**
   * 转换日期格式
   *
   * @param dateStr 待检查日期字符串
   * @return 日期
   */
  private LocalDate parseDateFormat(String dateStr) {
    LocalDate result;
    try {
      result = LocalDate.parse(dateStr, DateTimeFormatter.ofPattern("yyyy-MM-dd"));
    } catch (Exception e) {
      result = null;
    }
    return result;
  }
}
