package com.biz.crm.sfa.business.holiday.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.sfa.business.holiday.local.entity.HolidayDateEntity;
import com.biz.crm.sfa.business.holiday.local.entity.HolidayEntity;
import com.biz.crm.sfa.business.holiday.local.repository.HolidayRepository;
import com.biz.crm.sfa.business.holiday.local.service.HolidayDateService;
import com.biz.crm.sfa.business.holiday.local.service.HolidayService;
import com.biz.crm.sfa.business.holiday.sdk.dto.HolidayDto;
import com.biz.crm.sfa.business.holiday.sdk.event.HolidayEventListener;
import com.biz.crm.sfa.business.holiday.sdk.vo.HolidayVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;

/**
 * 节假日表服务实现类
 *
 * @author ning.zhang
 * @date 2022-05-11 14:48:54
 */
@Slf4j
@Service("holidayService")
public class HolidayServiceImpl implements HolidayService {

  @Autowired
  private HolidayRepository holidayRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private HolidayDateService holidayDateService;
  @Autowired(required = false)
  private List<HolidayEventListener> listeners;

  @Override
  @Transactional
  public HolidayEntity create(HolidayDto dto) {
    this.createValidation(dto);
    HolidayEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, HolidayEntity.class, HashSet.class, ArrayList.class);
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    this.holidayRepository.save(entity);
    //绑定关联数据
    dto.setId(entity.getId());
    List<HolidayDateEntity> dateEntities = this.holidayDateService.update(dto);
    entity.setDateList(dateEntities);
    entity.setHolidayDays(dto.getHolidayDays());
    entity.setSalaryDays(dto.getSalaryDays());
    this.holidayRepository.updateById(entity);
    //新增节假日事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      HolidayVo vo = this.nebulaToolkitService.copyObjectByWhiteList(entity, HolidayVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(listener -> {
        listener.onCreate(vo);
      });
    }
    return entity;
  }

  @Override
  @Transactional
  public HolidayEntity update(HolidayDto dto) {
    this.updateValidation(dto);
    List<HolidayEntity> entities = this.holidayRepository.findByIds(Lists.newArrayList(dto.getId()));
    Validate.isTrue(!CollectionUtils.isEmpty(entities), "节假日信息不存在");
    HolidayEntity updateEntity = this.nebulaToolkitService.copyObjectByWhiteList(dto, HolidayEntity.class, HashSet.class, ArrayList.class);
    this.holidayRepository.updateById(updateEntity);
    //绑定关联数据
    List<HolidayDateEntity> dateEntities = this.holidayDateService.update(dto);
    updateEntity.setDateList(dateEntities);
    updateEntity.setHolidayDays(dto.getHolidayDays());
    updateEntity.setSalaryDays(dto.getSalaryDays());
    this.holidayRepository.updateById(updateEntity);
    //更新节假日事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      HolidayVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(entities.get(0), HolidayVo.class, HashSet.class, ArrayList.class);
      HolidayVo newVo = this.nebulaToolkitService.copyObjectByWhiteList(updateEntity, HolidayVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(listener -> {
        listener.onUpdate(oldVo, newVo);
      });
    }
    return updateEntity;
  }

  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    List<HolidayEntity> entities = this.holidayRepository.listByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据删除个数不匹配");
    this.holidayRepository.updateDelFlagByIds(ids);
    //删除节假日事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      List<HolidayVo> voList = (List<HolidayVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, HolidayEntity.class
          , HolidayVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(listener -> {
        listener.onDelete(voList);
      });
    }
  }

  @Override
  public Page<HolidayEntity> findByConditions(Pageable pageable, HolidayDto dto) {
    return null;
  }

  /**
   * 在创建attendanceRule模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(HolidayDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setSalaryDays(0);
    dto.setHolidayDays(0);
    Validate.notBlank(dto.getYear(), "缺失年份");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getInfoList()), "缺失节假日明细信息");
    try {
      LocalDate.parse(String.format("%s-01-01",dto.getYear()), DateTimeFormatter.ofPattern("yyyy-MM-dd"));
    } catch (Exception e) {
      throw new IllegalArgumentException("非法的年份格式，可用的格式：[yyyy]", e);
    }
    HolidayEntity holidayEntity = this.holidayRepository.findByYear(dto.getYear(), dto.getYear());
    Validate.isTrue(Objects.isNull(holidayEntity), "已存在相同年份的节假日信息");
  }

  /**
   * 在修改holidayRule模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(HolidayDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setSalaryDays(0);
    dto.setHolidayDays(0);
    Validate.notBlank(dto.getId(), "修改信息时，id不能为空！");
    Validate.notBlank(dto.getYear(), "缺失年份");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getInfoList()), "缺失节假日明细信息");
    try {
      LocalDate.parse(String.format("%s-01-01",dto.getYear()), DateTimeFormatter.ofPattern("yyyy-MM-dd"));
    } catch (Exception e) {
      throw new IllegalArgumentException("非法的年份格式，可用的格式：[yyyy]", e);
    }
    HolidayEntity holidayEntity = this.holidayRepository.findByYear(dto.getYear(), dto.getYear());
    Validate.isTrue(Objects.isNull(holidayEntity) || holidayEntity.getId().equals(dto.getId()), "已存在相同年份的节假日信息");
  }
}
