package com.biz.crm.sfa.business.holiday.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.sfa.business.holiday.local.entity.HolidayDateEntity;
import com.biz.crm.sfa.business.holiday.local.entity.HolidayEntity;
import com.biz.crm.sfa.business.holiday.local.repository.HolidayRepository;
import com.biz.crm.sfa.business.holiday.sdk.dto.HolidayConditionDto;
import com.biz.crm.sfa.business.holiday.sdk.dto.HolidayPageDto;
import com.biz.crm.sfa.business.holiday.sdk.service.HolidayVoService;
import com.biz.crm.sfa.business.holiday.sdk.vo.HolidayDateVo;
import com.biz.crm.sfa.business.holiday.sdk.vo.HolidayInfoVo;
import com.biz.crm.sfa.business.holiday.sdk.vo.HolidayVo;
import com.bizunited.nebula.common.service.CopyObjectCallback;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 节假日Vo服务接口实现类
 *
 * @author ning.zhang
 * @date 2022/5/11
 */
@Slf4j
@Service
public class HolidayVoServiceImpl implements HolidayVoService, CopyObjectCallback<HolidayEntity, HolidayVo> {

  @Autowired
  private HolidayRepository holidayRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public HolidayVo findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    List<HolidayEntity> entities = this.holidayRepository.findByIds(Lists.newArrayList(id));
    if (CollectionUtils.isEmpty(entities)) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByBlankList(entities.get(0), HolidayVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public Page<HolidayVo> findByConditions(Pageable pageable, HolidayPageDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new HolidayPageDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    Page<HolidayEntity> entityPage = this.holidayRepository.findByConditions(pageable, dto);
    List<HolidayEntity> entities = entityPage.getRecords();
    Page<HolidayVo> pageResult = new Page<>(entityPage.getCurrent(), entityPage.getSize(), entityPage.getTotal());
    if (!CollectionUtils.isEmpty(entities)) {
      pageResult.setRecords((List<HolidayVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, HolidayEntity.class
          , HolidayVo.class, HashSet.class, ArrayList.class));
    }
    return pageResult;
  }

  @Override
  public HolidayVo findByDate(String date) {
    if (StringUtils.isBlank(date)) {
      return null;
    }
    List<HolidayEntity> holidayEntities = this.holidayRepository.findByDate(date, TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(holidayEntities)) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByBlankList(holidayEntities.get(0), HolidayVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public List<HolidayVo> findByHolidayConditionDto(HolidayConditionDto dto) {
    if (Objects.isNull(dto)) {
      return Lists.newLinkedList();
    }
    dto.setTenantCode(TenantUtils.getTenantCode());
    List<HolidayEntity> entities = this.holidayRepository.findByHolidayConditionDto(dto);
    if (CollectionUtils.isEmpty(entities)) {
      return Lists.newLinkedList();
    }
    return (List<HolidayVo>) this.nebulaToolkitService.copyCollectionByBlankList(entities, HolidayEntity.class
        , HolidayVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public List<String> findYearList() {
    List<HolidayEntity> entityList = this.holidayRepository.findAll(TenantUtils.getTenantCode());
    List<String> yearList = entityList.stream().map(HolidayEntity::getYear).distinct().collect(Collectors.toList());
    return yearList;
  }

  @Override
  public boolean validate(Class<?> sourceClass, Class<?> targetClass) {
    return HolidayEntity.class.isAssignableFrom(sourceClass) && HolidayVo.class.isAssignableFrom(targetClass);
  }

  @Override
  public void callback(Class<HolidayEntity> sourceClass, List<HolidayEntity> sourceObjects,
                       Class<HolidayVo> targetClass, List<HolidayVo> targetObjects) {
    for (int i = 0; i < sourceObjects.size(); i++) {
      HolidayEntity holidayEntity = sourceObjects.get(i);
      HolidayVo holidayVo = targetObjects.get(i);
      List<HolidayDateEntity> dateList = holidayEntity.getDateList();
      Map<String, HolidayInfoVo> holidayInfoMap = Maps.newHashMap();
      if (CollectionUtils.isEmpty(dateList)) {
        continue;
      }
      dateList.sort(Comparator.comparing(HolidayDateEntity::getHolidayDate));
      dateList.forEach(holidayDateEntity -> {
        HolidayInfoVo holidayInfoVo = holidayInfoMap.getOrDefault(holidayDateEntity.getHolidayId(), new HolidayInfoVo());
        if (CollectionUtils.isEmpty(holidayInfoVo.getDateList())) {
          holidayInfoVo.setDateList(Lists.newArrayList());
          holidayInfoVo.setStartDate(holidayDateEntity.getHolidayDate());
          holidayInfoVo.setEndDate(holidayDateEntity.getHolidayDate());
          holidayInfoVo.setHolidayName(holidayDateEntity.getHolidayName());
        }
        HolidayDateVo holidayDateVo = new HolidayDateVo();
        holidayDateVo.setHolidayDate(holidayDateEntity.getHolidayDate());
        holidayDateVo.setCalculationSalary(holidayDateEntity.getCalculationSalary());
        holidayInfoVo.getDateList().add(holidayDateVo);
        holidayInfoVo.setStartDate(holidayInfoVo.getStartDate().compareTo(holidayDateEntity.getHolidayDate()) > 0
            ? holidayDateEntity.getHolidayDate() : holidayInfoVo.getStartDate());
        holidayInfoVo.setEndDate(holidayInfoVo.getEndDate().compareTo(holidayDateEntity.getHolidayDate()) < 0
            ? holidayDateEntity.getHolidayDate() : holidayInfoVo.getStartDate());
        holidayInfoMap.put(holidayDateEntity.getHolidayId(), holidayInfoVo);
      });
      holidayVo.setInfoList(Lists.newArrayList(holidayInfoMap.values()));
    }
  }
}
