package com.biz.crm.sfa.business.integral.rule.local.service.internal;

import com.biz.crm.sfa.business.integral.rule.local.repository.IntegralRuleScopeRepository;
import com.biz.crm.sfa.business.integral.rule.local.entity.IntegralRuleScope;
import com.biz.crm.sfa.business.integral.rule.local.service.IntegralRuleScopeService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Set;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.beans.factory.annotation.Autowired;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import java.util.List;

/**
 * 积分范围表(IntegralRuleScope)表服务实现类
 *
 * @author rentao
 * @since 2022-06-22 12:31:38
 */
@Service
@Slf4j
public class IntegralRuleScopeServiceImpl implements IntegralRuleScopeService {

  @Autowired private IntegralRuleScopeRepository integralRuleScopeRepository;

  /**
   * 通过integralRuleCode查询集合数据
   * @param integralRuleCode
   * @return 集合数据
   */
  @Override
  public List<IntegralRuleScope> findByIntegralRuleCode(String integralRuleCode) {
    if (StringUtils.isBlank(integralRuleCode)) {
      return null;
    }
    return this.integralRuleScopeRepository.findByIntegralRuleCode(integralRuleCode);
  }

  @Override
  public List<IntegralRuleScope> findByScopeCodes(List<String> scopeCodes) {
    if (CollectionUtils.isEmpty(scopeCodes)) {
      return null;
    }
    return this.integralRuleScopeRepository.findByScopeCodes(scopeCodes);
  }

  /**
   * 批量创建
   * @param integralRuleScopes
   * @return
   */
  @Override
  @Transactional
  public void createBatch(List<IntegralRuleScope> integralRuleScopes) {
    Validate.isTrue(!CollectionUtils.isEmpty(integralRuleScopes), "新增时，数据不存在");
    integralRuleScopes.forEach(
        e -> {
          this.createValidate(e);
        });
    this.integralRuleScopeRepository.saveBatch(integralRuleScopes);
  }

  /**
   * 批量修改
   * @param integralRuleScopes 实体对象
   * @return 修改结果
   */
  @Override
  @Transactional
  public void updateBatch(List<IntegralRuleScope> integralRuleScopes) {
    Validate.isTrue(!CollectionUtils.isEmpty(integralRuleScopes), "编辑时，数据不存在");
    Set<String> integralRuleCodes = new HashSet<>();
    integralRuleScopes.forEach(
        e -> {
          this.updateValidate(e);
          integralRuleCodes.add(e.getIntegralRuleCode());
          e.setId(null);
        });
    Validate.isTrue(!CollectionUtils.isEmpty(integralRuleCodes) && integralRuleCodes.size() == 1 ,"一次编辑只能操作规则编码相同的一批数据");
    this.integralRuleScopeRepository.deleteByIntegralRuleCode(integralRuleCodes);
    this.integralRuleScopeRepository.saveBatch(integralRuleScopes);
  }

  /**
   * 禁用数据
   * @param integralRuleCodes
   * @return 删除结果
   */
  @Override
  @Transactional
  public void disableByIntegralRuleCode(List<String> integralRuleCodes) {
    this.integralRuleScopeRepository.updateStateByIntegralRuleCode(integralRuleCodes, 0);
  }

  /**
   * 删除数据
   * @param integralRuleCodes
   * @return 删除结果
   */
  @Override
  @Transactional
  public void deleteByIntegralRuleCode(Set<String> integralRuleCodes) {
    this.integralRuleScopeRepository.deleteByIntegralRuleCode(integralRuleCodes);
  }

  /**
   * 启用数据
   * @param integralRuleCodes
   * @return 删除结果
   */
  @Override
  @Transactional
  public void enableByIntegralRuleCode(List<String> integralRuleCodes) {
    this.integralRuleScopeRepository.updateStateByIntegralRuleCode(integralRuleCodes, 1);
  }

  /**
   * 创建验证
   *
   * @param integralRuleScope
   */
  private void createValidate(IntegralRuleScope integralRuleScope) {
    Validate.notNull(integralRuleScope, "数据操作时，对象信息不能为空！");
    integralRuleScope.setId(null);
    Validate.notBlank(integralRuleScope.getIntegralRuleCode(), "数据操作时，积分规则编码不能为空！");
    Validate.notBlank(integralRuleScope.getScopeCode(), "数据操作时，被选定的业务编号不能为空！");
    Validate.notBlank(integralRuleScope.getScopeName(), "数据操作时，被选定的业务名称不能为空！");
    Validate.notBlank(integralRuleScope.getScopeType(), "数据操作时，被选定的业务类型不能为空！");
    integralRuleScope.setTenantCode(TenantUtils.getTenantCode());
  }

  /**
   * 修改验证
   *
   * @param integralRuleScope
   */
  private void updateValidate(IntegralRuleScope integralRuleScope) {
    Validate.notNull(integralRuleScope, "修改时，对象信息不能为空！");
    Validate.notBlank(integralRuleScope.getIntegralRuleCode(), "修改数据时，积分规则编码不能为空！");
    Validate.notBlank(integralRuleScope.getScopeCode(), "修改数据时，被选定的业务编号不能为空！");
    Validate.notBlank(integralRuleScope.getScopeName(), "修改数据时，被选定的业务名称不能为空！");
    Validate.notBlank(integralRuleScope.getScopeType(), "修改数据时，被选定的业务类型不能为空！");
    integralRuleScope.setTenantCode(TenantUtils.getTenantCode());

  }

}
