package com.biz.crm.sfa.business.integral.rule.local.service.internal;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.sfa.business.integral.rule.local.entity.IntegralCalculateRule;
import com.biz.crm.sfa.business.integral.rule.local.entity.IntegralRuleScope;
import com.biz.crm.sfa.business.integral.rule.local.repository.IntegralRuleRepository;
import com.biz.crm.sfa.business.integral.rule.local.entity.IntegralRule;
import com.biz.crm.sfa.business.integral.rule.local.service.IntegralCalculateRuleService;
import com.biz.crm.sfa.business.integral.rule.local.service.IntegralRuleScopeService;
import com.biz.crm.sfa.business.integral.rule.local.service.IntegralRuleService;
import com.biz.crm.sfa.business.integral.rule.sdk.service.IntegralRuleVoService;
import com.biz.crm.sfa.business.integral.rule.sdk.constant.IntegralRuleConstant;
import com.biz.crm.sfa.business.integral.rule.sdk.dto.IntegralCalculateRuleDto;
import com.biz.crm.sfa.business.integral.rule.sdk.dto.IntegralRuleDto;
import com.biz.crm.sfa.business.integral.rule.sdk.dto.IntegralRuleScopeDto;
import com.biz.crm.sfa.business.integral.rule.sdk.event.IntegralRuleEventListener;
import com.biz.crm.sfa.business.integral.rule.sdk.vo.IntegralRuleVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Set;
import java.util.stream.Collectors;
import org.springframework.stereotype.Service;
import org.springframework.beans.factory.annotation.Autowired;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.data.domain.Pageable;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import java.util.List;
import java.util.Objects;

/**
 * 积分规则主表(IntegralRule)表服务实现类
 *
 * @author rentao
 * @since 2022-06-22 12:31:35
 */
@Service
public class IntegralRuleServiceImpl implements IntegralRuleService {

  @Autowired private IntegralRuleRepository integralRuleRepository;
  @Autowired private NebulaToolkitService nebulaToolkitService;
  @Autowired private IntegralRuleVoService integralRuleVoService;
  @Autowired private GenerateCodeService generateCodeService;
  @Autowired private IntegralCalculateRuleService integralCalculateRuleService;
  @Autowired private IntegralRuleScopeService integralRuleScopeService;
  @Autowired(required = false)
  private List<IntegralRuleEventListener> integralRuleEventListeners;

  /**
   * 分页查询数据
   *
   * @param pageable 分页对象
   * @param integralRule 实体对象
   * @return
   */
  @Override
  public Page<IntegralRule> findByConditions(Pageable pageable, IntegralRule integralRule) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(integralRule)) {
      integralRule = new IntegralRule();
    }
    integralRule.setTenantCode(TenantUtils.getTenantCode());
    return this.integralRuleRepository.findByConditions(pageable, integralRule);
  }

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public IntegralRule findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.integralRuleRepository.getById(id);
  }


  /**
   * 通过integralCode查询单条数据
   *
   * @param integralCode
   * @return 单条数据
   */
  @Override
  public IntegralRule findByCode(String integralCode) {
    if (StringUtils.isBlank(integralCode)) {
      return null;
    }
    return this.integralRuleRepository.findByCode(integralCode);
  }

  /**
   * 新增数据
   *
   * @param integralRuleDto 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public IntegralRule create(IntegralRuleDto integralRuleDto) {
    // 1 数据验证
    this.createValidate(integralRuleDto);
    // 2 数据转换
    IntegralRule integralRule = this.transformCreate(integralRuleDto);
    // 3保存范围
    List<IntegralRuleScopeDto> integralRuleScopeDtos = integralRuleDto.getIntegralRuleScopeDtos();
    List<IntegralCalculateRuleDto> integralCalculateRuleDtos = integralRuleDto.getIntegralCalculateRuleDtos();
    List<IntegralCalculateRule> calculateRules =  (List<IntegralCalculateRule>)this.nebulaToolkitService.copyCollectionByWhiteList(integralCalculateRuleDtos, IntegralCalculateRuleDto.class, IntegralCalculateRule.class, HashSet.class, ArrayList.class);
    List<IntegralRuleScope> integralRuleScopes =  (List<IntegralRuleScope>)this.nebulaToolkitService.copyCollectionByWhiteList(integralRuleScopeDtos, IntegralRuleScopeDto.class, IntegralRuleScope.class, HashSet.class, ArrayList.class);
    calculateRules.forEach(integralCalculateRule -> integralCalculateRule.setIntegralRuleCode(integralRule.getIntegralRuleCode()));
    integralRuleScopes.forEach(integralRuleScope -> integralRuleScope.setIntegralRuleCode(integralRule.getIntegralRuleCode()));
    this.integralCalculateRuleService.createBatch(calculateRules);
    this.integralRuleScopeService.createBatch(integralRuleScopes);
    this.integralRuleRepository.saveOrUpdate(integralRule);
    // 5 通知
    if (!CollectionUtils.isEmpty(this.integralRuleEventListeners)) {
      IntegralRuleVo integralRuleVo =
          this.nebulaToolkitService.copyObjectByBlankList(
              integralRuleDto, IntegralRuleVo.class, HashSet.class, ArrayList.class);
      this.integralRuleEventListeners.forEach(
          eventListener -> {
            eventListener.onCreate(integralRuleVo);
          });
    }
    return integralRule;
  }

  /**
   * 修改新据
   *
   * @param integralRuleDto 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public IntegralRule update(IntegralRuleDto integralRuleDto) {
    // 1 数据验证
    this.updateValidate(integralRuleDto);
    IntegralRuleVo integralRuleVoOld = this.integralRuleVoService.findDetailById(integralRuleDto.getId());
    Validate.notNull(integralRuleVoOld,"编辑的数据不存在！");
    // 2数据转换
    IntegralRule integralRule = this.transformUpdate(integralRuleDto);
    //3 编辑
    List<IntegralRuleScopeDto> integralRuleScopeDtos = integralRuleDto.getIntegralRuleScopeDtos();
    List<IntegralCalculateRuleDto> integralCalculateRuleDtos = integralRuleDto.getIntegralCalculateRuleDtos();
    List<IntegralCalculateRule> calculateRules =  (List<IntegralCalculateRule>)this.nebulaToolkitService.copyCollectionByWhiteList(integralCalculateRuleDtos, IntegralCalculateRuleDto.class, IntegralCalculateRule.class, HashSet.class, ArrayList.class);
    List<IntegralRuleScope> integralRuleScopes =  (List<IntegralRuleScope>)this.nebulaToolkitService.copyCollectionByWhiteList(integralRuleScopeDtos, IntegralRuleScopeDto.class, IntegralRuleScope.class, HashSet.class, ArrayList.class);
    calculateRules.forEach(integralCalculateRule -> integralCalculateRule.setIntegralRuleCode(integralRuleVoOld.getIntegralRuleCode()));
    integralRuleScopes.forEach(integralRuleScope -> integralRuleScope.setIntegralRuleCode(integralRuleVoOld.getIntegralRuleCode()));
    this.integralCalculateRuleService.updateBatch(calculateRules);
    this.integralRuleScopeService.updateBatch(integralRuleScopes);
    this.integralRuleRepository.updateById(integralRule);
    if (!CollectionUtils.isEmpty(this.integralRuleEventListeners)) {
      IntegralRuleVo integralRuleVo =
          this.nebulaToolkitService.copyObjectByBlankList(
              integralRuleDto, IntegralRuleVo.class, HashSet.class, ArrayList.class);
      this.integralRuleEventListeners.forEach(
          eventListener -> {
            eventListener.onUpdate(integralRuleVoOld,integralRuleVo);
          });
    }
    return integralRule;
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    List<IntegralRule> integralRules = this.integralRuleRepository.findByIds(idList);
    this.integralRuleRepository.removeByIds(idList);
    if(!CollectionUtils.isEmpty(integralRules)){
      Set<String> integralRuleCodes = integralRules.stream().map(IntegralRule::getIntegralRuleCode).distinct().collect(Collectors.toSet());
      this.integralCalculateRuleService.deleteByIntegralRuleCode(integralRuleCodes);
      this.integralRuleScopeService.deleteByIntegralRuleCode(integralRuleCodes);
    }
  }

  /**
   * 启用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 启用结果
   */
  @Transactional
  @Override
  public void enable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "启用数据时，主键集合不能为空！");
    this.integralRuleRepository.updateEnableStatusByIds(idList, EnableStatusEnum.ENABLE);
    List<IntegralRule> integralRules = this.integralRuleRepository.findByIds(idList);
    if(!CollectionUtils.isEmpty(integralRules)){
      List<String> integralRuleCodes = integralRules.stream().map(IntegralRule::getIntegralRuleCode).distinct().collect(Collectors.toList());
      this.integralCalculateRuleService.enableByIntegralRuleCode(integralRuleCodes);
      this.integralRuleScopeService.enableByIntegralRuleCode(integralRuleCodes);
    }
  }

  /**
   * 禁用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 禁用结果
   */
  @Transactional
  @Override
  public void disable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "禁用数据时，主键集合不能为空！");
    this.integralRuleRepository.updateEnableStatusByIds(idList, EnableStatusEnum.DISABLE);
    List<IntegralRule> integralRules = this.integralRuleRepository.findByIds(idList);
    if(!CollectionUtils.isEmpty(integralRules)){
      List<String> integralRuleCodes = integralRules.stream().map(IntegralRule::getIntegralRuleCode).distinct().collect(Collectors.toList());
      this.integralCalculateRuleService.disableByIntegralRuleCode(integralRuleCodes);
      this.integralRuleScopeService.disableByIntegralRuleCode(integralRuleCodes);
    }
  }

  /**
   * 创建验证
   *
   * @param integralRuleDto
   */
  private void createValidate(IntegralRuleDto integralRuleDto) {
    Validate.notNull(integralRuleDto, "新增时，对象信息不能为空！");
    integralRuleDto.setId(null);
    Validate.notBlank(integralRuleDto.getIntegralRuleName(), "新增数据时，积分规则名称不能为空！");
    Validate.notNull(integralRuleDto.getStartTime(), "数据操作时，积分规则开始时间（包括）不能为空！");
    Validate.notNull(integralRuleDto.getEndTime(), "新增数据时，积分规则结束时间（包括）不能为空！");
    Validate.isTrue(integralRuleDto.getStartTime().compareTo(integralRuleDto.getEndTime()) <= 0,
        "积分规则开始时间应早于结束时间");
    Validate.isTrue(!CollectionUtils.isEmpty(integralRuleDto.getIntegralCalculateRuleDtos()),"新增数据时，积分规则计算公式不能为空");
    Validate.isTrue(!CollectionUtils.isEmpty(integralRuleDto.getIntegralRuleScopeDtos()),"新增数据时，积分规则适用范围不能为空");
    this.checkOrgAndRole(integralRuleDto.getIntegralRuleScopeDtos(),null);
  }

  /**
   * 修改验证
   *
   * @param integralRuleDto
   */
  private void updateValidate(IntegralRuleDto integralRuleDto) {
    Validate.notNull(integralRuleDto, "修改时，对象信息不能为空！");
    Validate.notBlank(integralRuleDto.getId(), "修改数据时，不能为空！");
    Validate.notBlank(integralRuleDto.getIntegralRuleCode(), "修改数据时，积分规则编码不能为空！");
    Validate.notBlank(integralRuleDto.getIntegralRuleName(), "修改数据时，积分规则名称不能为空！");
    Validate.notNull(integralRuleDto.getStartTime(), "修改数据时，积分规则开始时间（包括）不能为空！");
    Validate.notNull(integralRuleDto.getEndTime(), "修改数据时，积分规则结束时间（包括）不能为空！");
    Validate.isTrue(integralRuleDto.getStartTime().compareTo(integralRuleDto.getEndTime()) <= 0,
        "积分规则开始时间应早于结束时间");
    Validate.isTrue(!CollectionUtils.isEmpty(integralRuleDto.getIntegralCalculateRuleDtos()),"修改数据时，积分规则计算公式不能为空");
    Validate.isTrue(!CollectionUtils.isEmpty(integralRuleDto.getIntegralRuleScopeDtos()),"修改数据时，积分规则适用范围不能为空");
    this.checkOrgAndRole(integralRuleDto.getIntegralRuleScopeDtos(),integralRuleDto.getIntegralRuleCode());
  }

  /**
   * 新增数据填充
   *
   * @param integralRuleDto
   * @author rentao
   * @date
   */
  private IntegralRule transformCreate(IntegralRuleDto integralRuleDto) {
    IntegralRule entity = this.nebulaToolkitService.copyObjectByWhiteList(
            integralRuleDto, IntegralRule.class, HashSet.class, ArrayList.class);
    entity.setIntegralRuleCode(this.generateCodeService.generateCode(IntegralRuleConstant.INTEGRAL_RULE_CODE, 1).get(0));
    entity.setTenantCode(TenantUtils.getTenantCode());
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    integralRuleDto.setIntegralRuleCode(entity.getIntegralRuleCode());
    return entity;
  }

  /**
   * 修改数据填充
   *
   * @param integralRuleDto
   * @author rentao
   * @date
   */
  private IntegralRule transformUpdate(IntegralRuleDto integralRuleDto) {
    IntegralRule entity = this.nebulaToolkitService.copyObjectByWhiteList(
        integralRuleDto, IntegralRule.class, HashSet.class, ArrayList.class);
    entity.setTenantCode(TenantUtils.getTenantCode());
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    integralRuleDto.setIntegralRuleCode(entity.getIntegralRuleCode());
    return entity;
  }


  /**
  *
  * 校验新增规则的组织和角色是否与已存在规则有交叉部分
  * @param integralRuleScopeDtos
  * @param integralRuleCode
  * @author rentao
  * @date
  */
  private void checkOrgAndRole(List<IntegralRuleScopeDto> integralRuleScopeDtos,String integralRuleCode) {
    List<String> scopeCodes = integralRuleScopeDtos.stream().map(IntegralRuleScopeDto::getScopeCode).collect(Collectors.toList());
    //查询已存在的数据
    List<IntegralRuleScope> integralRuleScopes = this.integralRuleScopeService.findByScopeCodes(scopeCodes);
    //编辑排除之前的数据
    if(StringUtils.isNotBlank(integralRuleCode) && !CollectionUtils.isEmpty(integralRuleScopes)){
      integralRuleScopes.removeIf(integralRuleScope -> integralRuleCode.equals(integralRuleScope.getIntegralRuleCode()));
    }
    //若没有查出，说明组织和规则都没有被使用
    if(CollectionUtils.isEmpty(integralRuleScopes)){
      return;
    }
    integralRuleScopeDtos.forEach(integralRuleScopeDto -> {
      List<IntegralRuleScope> scopeList = integralRuleScopes.stream().filter(integralRuleScope ->
          integralRuleScope.getScopeType().equals(integralRuleScopeDto.getScopeType())
              && integralRuleScope.getScopeCode().equals(integralRuleScopeDto.getScopeCode()))
          .collect(
              Collectors.toList());
      Validate.isTrue(CollectionUtils.isEmpty(scopeList),"本次提存在重复数据！");
    });
  }

}
