package com.biz.crm.sfa.leave.local.service.internal;

import cn.hutool.log.Log;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.LoginUserDetails;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.sfa.business.leave.sdk.enums.LeaveTypeEnum;
import com.biz.crm.sfa.business.leave.sdk.enums.LeaveVisibleEnum;
import com.biz.crm.sfa.business.leave.sdk.vo.SfaLeaveApplySummaryVo;
import com.biz.crm.sfa.leave.local.dto.SfaLeaveApplySummaryDto;
import com.biz.crm.sfa.leave.local.dto.SfaLeaveDto;
import com.biz.crm.sfa.leave.local.dto.SfaLeaveListDto;
import com.biz.crm.sfa.leave.local.entity.SfaLeaveEntity;
import com.biz.crm.sfa.leave.local.repository.SfaLeaveAttachmentRepository;
import com.biz.crm.sfa.leave.local.repository.SfaLeaveRepository;
import com.biz.crm.sfa.leave.local.service.SfaLeaveVoService;
import com.biz.crm.sfa.leave.local.vo.SfaLeaveApplyDetailVo;
import com.biz.crm.sfa.leave.local.vo.SfaLeaveVo;
import com.biz.crm.workflow.sdk.constant.enums.ActApproveStatusEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import java.util.Collection;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 请假vo查询 接口实现
 *
 * @author jerry7
 */
@Slf4j
@Service
public class SfaLeaveVoServiceImpl implements SfaLeaveVoService {

  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Autowired
  private SfaLeaveRepository sfaLeaveRepository;

  @Autowired
  private SfaLeaveAttachmentRepository sfaLeaveAttachmentRepository;

  @Autowired
  private LoginUserService loginUserService;


  /**
   * 分页查询请假列表
   *
   * @param reqVo 查询参数
   * @return 请假列表
   */
  @Override
  public Page<SfaLeaveVo> findByConditions(Pageable pageable, SfaLeaveListDto reqVo) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    //补全时间
    if (StringUtils.isNotEmpty(reqVo.getApplicationDateStart())) {
      reqVo.setApplicationDateStart(reqVo.getApplicationDateStart() + " 00:00:00");
    }
    if (StringUtils.isNotEmpty(reqVo.getApplicationDateEnd())) {
      reqVo.setApplicationDateEnd(reqVo.getApplicationDateEnd() + " 23:59:59");
    }
    //查询数据
    Page<SfaLeaveEntity> list = this.sfaLeaveRepository.findByConditions(pageable, reqVo);
    List<SfaLeaveEntity> entities = list.getRecords();
    Page<SfaLeaveVo> pageResult = new Page<>(list.getCurrent(), list.getSize(), list.getTotal());
    if (!org.springframework.util.CollectionUtils.isEmpty(entities)) {
      pageResult.setRecords((List<SfaLeaveVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, SfaLeaveEntity.class
              , SfaLeaveVo.class, HashSet.class, ArrayList.class));
    }
    if (CollectionUtils.isNotEmpty(list.getRecords())) {
      pageResult.getRecords().forEach(vo -> {
        vo.setLeaveTypeName(LeaveTypeEnum.getByDictCode(vo.getLeaveType()).getDictCode());
      });
    }
    return pageResult;
  }

  /*
  *
  * 条件查询请假数据
  * @param sfaLeaveListDto
  * @author rentao
  * @date
  */
  @Override
  public List<SfaLeaveVo> findByConditions(SfaLeaveListDto sfaLeaveListDto) {
    //补全时间
    if (StringUtils.isNotEmpty(sfaLeaveListDto.getApplicationDateStart())) {
      sfaLeaveListDto.setApplicationDateStart(sfaLeaveListDto.getApplicationDateStart() + " 00:00:00");
    }
    if (StringUtils.isNotEmpty(sfaLeaveListDto.getApplicationDateEnd())) {
      sfaLeaveListDto.setApplicationDateEnd(sfaLeaveListDto.getApplicationDateEnd() + " 23:59:59");
    }
    //查询数据
    List<SfaLeaveEntity> entities = this.sfaLeaveRepository.findByConditions(sfaLeaveListDto);
    if (CollectionUtils.isEmpty(entities)) {
      return null;
    }
    List<SfaLeaveVo> sfaLeaveVos = (List<SfaLeaveVo>)this.nebulaToolkitService.copyCollectionByWhiteList(entities, SfaLeaveEntity.class
        , SfaLeaveVo.class, HashSet.class, ArrayList.class);
    sfaLeaveVos.forEach(vo -> {
        vo.setLeaveTypeName(LeaveTypeEnum.getByDictCode(vo.getLeaveType()).getDictCode());
    });
    return sfaLeaveVos;
  }


  /**
   * 根据Id 查询请假详情
   *
   * @param id id
   * @return 请假详情
   */
  @Override
  public SfaLeaveVo queryById(String id) {
    SfaLeaveEntity sfaLeave = this.sfaLeaveRepository.getById(id);
    if (ObjectUtils.isNotEmpty(sfaLeave)) {
      SfaLeaveVo leaveVo = this.nebulaToolkitService.copyObjectByWhiteList(sfaLeave, SfaLeaveVo.class, HashSet.class, ArrayList.class);
      leaveVo.setAttachmentList(this.sfaLeaveAttachmentRepository.findBySourceId(id));
      return leaveVo;
    }
    return null;
  }

  @Override
  public Page<SfaLeaveApplySummaryVo> findSummaryByConditions(Pageable pageable, SfaLeaveApplySummaryDto dto) {
    //分页获取请假人列表
    dto.setProcessStatus(ActApproveStatusEnum.APPROVED.getCode());
    Page<SfaLeaveEntity> list = this.sfaLeaveRepository.findSummaryByConditions(pageable, dto);
    List<SfaLeaveEntity> entities = list.getRecords();
    Page<SfaLeaveApplySummaryVo> pageResult = new Page<>(list.getCurrent(), list.getSize(), list.getTotal());
    if (CollectionUtils.isNotEmpty(entities)) {
      pageResult.setRecords((List<SfaLeaveApplySummaryVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, SfaLeaveEntity.class
              , SfaLeaveApplySummaryVo.class, HashSet.class, ArrayList.class));
    }
    //根据请假人列表当前时间内的请假数据详情
    Set<String> userNames = pageResult.getRecords().stream().map(SfaLeaveApplySummaryVo::getUserName).collect(Collectors.toSet());
    SfaLeaveDto sfaLeaveDto = new SfaLeaveDto();
    sfaLeaveDto.setBeginTime(dto.getLeaveDateStart());
    sfaLeaveDto.setEndTime(dto.getLeaveDateEnd());
    sfaLeaveDto.setProcessStatus(ActApproveStatusEnum.APPROVED.getCode());
    sfaLeaveDto.setUserNames(userNames);
    List<SfaLeaveEntity> details = this.sfaLeaveRepository.findDetails(sfaLeaveDto);
    if (CollectionUtils.isEmpty(details)) {
      return pageResult;
    }
    Map<String, List<SfaLeaveEntity>> userDetailMap = details.stream().collect(Collectors.groupingBy(SfaLeaveEntity::getUserName));
    pageResult.getRecords().forEach(summary -> {
      List<SfaLeaveEntity> leaveEntities = userDetailMap.get(summary.getUserName());
      if (CollectionUtils.isNotEmpty(leaveEntities)) {
        for (SfaLeaveEntity e : leaveEntities) {
          LeaveTypeEnum leaveTypeEnum = LeaveTypeEnum.getByDictCode(e.getLeaveType());
          switch (leaveTypeEnum) {
            case DAYS_OFF: {
              summary.setDaysOff(summary.getDaysOff().add(new BigDecimal(e.getLeaveDuration())));
              break;
            }
            case ANNUAL_LEAVE:
            {
              summary.setAnnualLeave(summary.getAnnualLeave().add(new BigDecimal(e.getLeaveDuration())));
              break;
            }
            case SICK_LEAVE:
            {
              summary.setSickLeave(summary.getSickLeave().add(new BigDecimal(e.getLeaveDuration())));
              break;
            }
            case FUNERAL_LEAVE:
            {
              summary.setFuneralLeave(summary.getFuneralLeave().add(new BigDecimal(e.getLeaveDuration())));
              break;
            }
            case MARRIAGE_LEAVE:
            {
              summary.setMarriageLeave(summary.getMarriageLeave().add(new BigDecimal(e.getLeaveDuration())));
              break;
            }
            case PERSONAL_LEAVE:
            {
              summary.setPersonalLeave(summary.getPersonalLeave().add(new BigDecimal(e.getLeaveDuration())));
              break;
            }
            case MATERNITY_LEAVE:
            {
              summary.setMaternityLeave(summary.getMaternityLeave().add(new BigDecimal(e.getLeaveDuration())));
              break;
            }
            default:
            {
              summary.setPaternityLeave(summary.getPaternityLeave().add(new BigDecimal(e.getLeaveDuration())));
              break;
            }
          }
        }
      }
    });
    return pageResult;
  }
}
