package com.biz.crm.sfa.leave.local.service.internal;

import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.identity.FacturerUserDetails;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.user.sdk.service.UserFeignVoService;
import com.biz.crm.mdm.business.user.sdk.vo.UserVo;
import com.biz.crm.sfa.business.attendance.sdk.constant.AttendanceConstant;
import com.biz.crm.sfa.business.attendance.sdk.dto.RuleConditionDto;
import com.biz.crm.sfa.business.attendance.sdk.service.AttendanceRuleVoService;
import com.biz.crm.sfa.business.attendance.sdk.vo.AttendanceRuleVo;
import com.biz.crm.sfa.business.leave.sdk.constant.SfaLeaveConstant;
import com.biz.crm.sfa.business.leave.sdk.enums.LeaveTypeEnum;
import com.biz.crm.sfa.leave.local.dto.SfaApplyTimeInfoDto;
import com.biz.crm.sfa.leave.local.dto.SfaLeaveDto;
import com.biz.crm.sfa.leave.local.entity.SfaDaysOffEntity;
import com.biz.crm.sfa.leave.local.entity.SfaLeaveEntity;
import com.biz.crm.sfa.leave.local.repository.SfaDaysOffRepository;
import com.biz.crm.sfa.leave.local.repository.SfaLeaveAttachmentRepository;
import com.biz.crm.sfa.leave.local.repository.SfaLeaveRepository;
import com.biz.crm.sfa.leave.local.service.SfaLeaveService;
import com.biz.crm.sfa.leave.local.utils.SfaSignUtils;
import com.biz.crm.workflow.sdk.dto.OrgInfoDto;
import com.biz.crm.workflow.sdk.dto.PositionInfoDto;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessDto;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessMappingDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.biz.crm.workflow.sdk.listener.OrgInfoListener;
import com.biz.crm.workflow.sdk.listener.PositionInfoListener;
import com.biz.crm.workflow.sdk.service.ProcessBusinessMappingService;
import com.biz.crm.workflow.sdk.service.ProcessBusinessService;
import com.biz.crm.workflow.sdk.vo.OrgVo;
import com.biz.crm.workflow.sdk.vo.PositionVo;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessMappingVo;
import com.biz.crm.workflow.sdk.vo.response.OrgInfoResponse;
import com.biz.crm.workflow.sdk.vo.response.PositionInfoResponse;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 请假申请 接口实现
 *
 * @author jerry7
 */
@Slf4j
@Service
public class SfaLeaveServiceImpl implements SfaLeaveService {

  @Autowired
  private LoginUserService loginUserService;
  @Autowired
  private NebulaNetEventClient nebulaNetEventClient;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private AttendanceRuleVoService attendanceRuleVoService;
  @Autowired
  private SfaLeaveRepository sfaLeaveRepository;
  @Autowired
  private SfaLeaveAttachmentRepository sfaLeaveAttachmentRepository;
  @Autowired
  private UserFeignVoService userFeignVoService;
  @Autowired
  private SfaDaysOffRepository sfaDaysOffRepository;
  @Autowired
  private ProcessBusinessService processBusinessService;
  @Autowired
  private ProcessBusinessMappingService processBusinessMappingService;
  @Value("${workflow.process.key.leave}")
  private String processKey;
  /**
   * 请假申请接口
   *
   * @param sfaLeaveDto 请假申请DTO
   * @return 请假申请结果
   */
  @Override
  @Transactional(rollbackFor = Exception.class)
  public SfaLeaveEntity create(SfaLeaveDto sfaLeaveDto) {
    // 获取申请时长+验证信息
    SfaLeaveEntity entity = validateTime(sfaLeaveDto);
    // 判断编辑
    if (StringUtils.isNotEmpty(sfaLeaveDto.getId())) {
      SfaLeaveEntity old = this.sfaLeaveRepository.getById(sfaLeaveDto.getId());
      Validate.notNull(old, "编辑实体已不存在！");
      ProcessBusinessMappingDto businessMappingDto = new ProcessBusinessMappingDto();
      businessMappingDto.setBusinessCode(SfaLeaveConstant.SFA_LEAVE_PROCESS_FROM_TYPE);
      businessMappingDto.setBusinessNo(old.getId());
      ProcessBusinessMappingVo businessMappingVo = this.processBusinessMappingService.findSignalByConditions(businessMappingDto);
      //如果流程信息不存在,则无需校验和处理
      if (Objects.nonNull(businessMappingVo)) {
        // 驳回状态直接创建新的申请
        if (ProcessStatusEnum.REJECT.getDictCode().equals(businessMappingVo.getProcessStatus())) {
          entity.setId(null);
          // 追回的使用原来的申请继续提交
        } else if (ProcessStatusEnum.RECOVER.getDictCode().equals(businessMappingVo.getProcessStatus())) {
          entity.setId(old.getId());
          //编辑时删除旧有附件数据
          this.sfaLeaveAttachmentRepository.deleteBySourceId(old.getId());
          // 非编辑状态判断
        } else {
          Validate.isTrue(!ProcessStatusEnum.COMMIT.getDictCode().equals(businessMappingVo.getProcessStatus()), "审批中请假不允许编辑");
        }
      }
    }
    // 校验请假时间冲突
    List<SfaLeaveEntity> repeatList =
        this.sfaLeaveRepository.findRepeatByDates(entity.getUserName(), entity.getBeginTime(), entity.getEndTime(), entity.getId());
    //剩下审批中或者审批通过的数据(没有找到流程记录说明没有审批流程,认为是有效数据)
    if (!CollectionUtils.isEmpty(repeatList)) {
      List<String> ids = repeatList.stream().map(SfaLeaveEntity::getId).collect(Collectors.toList());
      ProcessBusinessMappingDto businessMappingDto = new ProcessBusinessMappingDto();
      businessMappingDto.setBusinessCode(AttendanceConstant.ATTENDANCE_FILL_CLOCK_APPLY_PROCESS_FORM_TYPE);
      businessMappingDto.setBusinessNos(ids);
      List<ProcessBusinessMappingVo> businessMappingVos = this.processBusinessMappingService.findMultiByByConditions(businessMappingDto);
      Map<String, String> processStatusMap = CollectionUtils.isEmpty(businessMappingVos) ? Maps.newHashMap() : businessMappingVos.stream()
          .collect(Collectors.toMap(ProcessBusinessMappingVo::getBusinessNo, ProcessBusinessMappingVo::getProcessStatus, (a, b) -> b));
      repeatList = repeatList.stream().filter(sfaLeaveEntity -> {
        String processStatus = processStatusMap.getOrDefault(sfaLeaveEntity.getId(), ProcessStatusEnum.PASS.getDictCode());
        return ProcessStatusEnum.PASS.getDictCode().equals(processStatus) || ProcessStatusEnum.COMMIT.getDictCode().equals(processStatus);
      }).collect(Collectors.toList());
    }

    if (!CollectionUtils.isEmpty(repeatList)) {
      for (SfaLeaveEntity vo : repeatList) {
        //只比较有请假时长的
        if (!"0".equals(vo.getLeaveDuration())) {
          SfaSignUtils.verifyDateRepeat(entity.getBeginTime(), entity.getEndTime(), vo.getBeginTime(), vo.getEndTime()
              , sfaLeaveDto.getTimeInfoList(), JSON.parseArray(vo.getTimeInfoListJson(), SfaApplyTimeInfoDto.class));
        }
      }
    }
    entity.setTimeInfoListJson(JSON.toJSONString(sfaLeaveDto.getTimeInfoList()));
    // 申请时间
    entity.setApplicationDate(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss").format(LocalDateTime.now()));
    this.sfaLeaveRepository.saveOrUpdate(entity);
    // 请假附件保存
    sfaLeaveDto.getAttachmentList().forEach(attachment -> {
      attachment.setSourceId(entity.getId());
      attachment.setApplyType(SfaLeaveConstant.SFA_LEAVE_PROCESS_FROM_TYPE);
      attachment.setTenantCode(TenantUtils.getTenantCode());
    });
    this.sfaLeaveAttachmentRepository.saveBatch(sfaLeaveDto.getAttachmentList());
    if (LeaveTypeEnum.DAYS_OFF.getDictCode().equals(sfaLeaveDto.getLeaveType())) {
      this.updateDaysOff(entity, true);
    }
    //发起审批流程
    sfaLeaveDto.setId(entity.getId());
    this.commitProcess(sfaLeaveDto);
    return entity;
  }

  /**
   * 计算请假时长
   *
   * @param reqVo 获取请假时长实体
   * @return 请假时长
   */
  @Override
  public BigDecimal countApplyDays(SfaLeaveDto reqVo) {
    // 校验并获取全部时间的申请时长
    BigDecimal countApplyDays = SfaSignUtils.countApplyDays(reqVo.getBeginTime(), reqVo.getEndTime()
        , reqVo.getTimeInfoList());
    reqVo.setLeaveDuration(countApplyDays.toString());
    RuleConditionDto conditionDto = new RuleConditionDto();
    conditionDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    conditionDto.setOrgCodes(Lists.newArrayList(reqVo.getOrgCode()));
    List<AttendanceRuleVo> attendanceRuleList = this.attendanceRuleVoService.findByRuleConditionDto(conditionDto);
    Validate.isTrue(!CollectionUtils.isEmpty(attendanceRuleList), "当前组织未设置考勤规则，无法请假");
    AttendanceRuleVo attendanceRuleVo = attendanceRuleList.get(0);
    LocalDate date = LocalDate.parse(reqVo.getBeginTime());
    Validate.isTrue(attendanceRuleVo.getWorkingDay().contains(String.valueOf(date.getDayOfWeek().getValue()))
        , "开始时间请选择工作日");
    LocalDate endDate = LocalDate.parse(reqVo.getEndTime());
    Validate.isTrue(attendanceRuleVo.getWorkingDay().contains(String.valueOf(endDate.getDayOfWeek().getValue()))
        , "结束时间请选择工作日");
    // 扣除扣除休息日
    List<SfaApplyTimeInfoDto> timeInfoList = SfaSignUtils.fillTimeInfoAndCheck(reqVo.getTimeInfoList(), reqVo.getBeginTime(), reqVo.getEndTime());
    for (int r = 0; r < timeInfoList.size(); r++) {
      if (r == 0 || r == timeInfoList.size() - 1) { // 开始和结束已经验证过了
        continue;
      }
      // 当天为非工作日
      LocalDate nowDate = LocalDate.parse(timeInfoList.get(r).getTimeStr());
      if (!attendanceRuleVo.getWorkingDay().contains(String.valueOf(nowDate.getDayOfWeek().getValue()))) {
        countApplyDays = countApplyDays.subtract(new BigDecimal(1));
      }
    }
    if (LeaveTypeEnum.DAYS_OFF.getDictCode().equals(reqVo.getLeaveType())) {
      this.checkDaysOff(reqVo);
    }
    return countApplyDays;
  }

  /**
   * 校验并填充初始数据
   */
  private SfaLeaveEntity validateTime(SfaLeaveDto reqVo) {
    List<UserVo> userVoList = this.userFeignVoService.findByUserNames(Lists.newArrayList(reqVo.getUserName()));
    Validate.notEmpty(userVoList, "请假人不存在");
    UserVo userVo = userVoList.stream().findFirst().get();
    reqVo.setRealName(userVo.getFullName());
    reqVo.setPosCode(userVo.getPositionCode());
    reqVo.setPosName(userVo.getPositionName());
    reqVo.setOrgCode(userVo.getOrgCode());
    reqVo.setOrgName(userVo.getOrgName());
    //获取职位详情，以取得上级职位编码
    SerializableBiConsumer<PositionInfoListener, PositionInfoDto> sf = PositionInfoListener::onFindByIdsOrCodes;
    PositionInfoResponse response = (PositionInfoResponse) nebulaNetEventClient.directPublish(PositionInfoDto.builder().roleCodes(Lists.newArrayList(reqVo.getPosCode())).build(), PositionInfoListener.class, sf);
    if (ObjectUtils.isNotEmpty(response) && !CollectionUtils.isEmpty(response.getPositionVos())) {
      PositionVo positionVo = response.getPositionVos().get(0);
      reqVo.setParentPosCode(positionVo.getParentCode());
    }
    //获取组织详情，已取得上级组织编码和名称
    SerializableBiConsumer<OrgInfoListener, OrgInfoDto> orgConsumer = OrgInfoListener::findByOrgCodes;
    OrgInfoDto orgInfoDto = new OrgInfoDto();
    orgInfoDto.setOrgCode(userVo.getOrgCode());
    OrgInfoResponse orgResponse = (OrgInfoResponse) nebulaNetEventClient.directPublish(orgInfoDto, OrgInfoListener.class, orgConsumer);
    if (ObjectUtils.isNotEmpty(orgResponse) && !CollectionUtils.isEmpty(orgResponse.getOrgVos())) {
      OrgVo orgVo = orgResponse.getOrgVos().get(0);
      reqVo.setParentOrgCode(orgVo.getParentCode());
      reqVo.setParentOrgName(orgVo.getParentName());
    }
    SfaLeaveEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(reqVo, SfaLeaveEntity.class, HashSet.class, ArrayList.class);
    // 请假类型校验
    if (StringUtils.isEmpty(entity.getLeaveType())) {
      throw new RuntimeException("请假类型必传");
    }
    //设置创建人信息，当前登录人信息
    FacturerUserDetails loginDetails = this.loginUserService.getLoginDetails(FacturerUserDetails.class);
    // 默认为当前人员申请
    entity.setCreateName(loginDetails.getUsername());
    entity.setCreateAccount(loginDetails.getAccount());
    entity.setCreateTime(DateUtil.date());
    entity.setTenantCode(loginDetails.getTenantCode());
    // 校验申请人信息
    if (StringUtils.isEmpty(entity.getUserName()) || StringUtils.isEmpty(entity.getRealName())
        || StringUtils.isEmpty(entity.getPosCode()) || StringUtils.isEmpty(entity.getOrgCode())) {
      throw new RuntimeException("申请人员信息必须包含以下信息：人员账号、人员姓名、岗位编码、组织编码，请核对");
    }
    BigDecimal countApplyDays = this.countApplyDays(reqVo);
    // 请假-调休或年假，触发校验规则
//    if (LeaveTypeEnum.ANNUAL_LEAVE.getDictCode().equals(entity.getLeaveType())
//            || LeaveTypeEnum.DAYS_OFF.getDictCode().equals(entity.getLeaveType())) {
    // 获取考勤规则
    RuleConditionDto conditionDto = new RuleConditionDto();
    conditionDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    conditionDto.setOrgCodes(Lists.newArrayList(entity.getOrgCode()));
    List<AttendanceRuleVo> attendanceRuleList = this.attendanceRuleVoService.findByRuleConditionDto(conditionDto);
    Validate.isTrue(!CollectionUtils.isEmpty(attendanceRuleList), "当前组织未设置考勤规则，无法请假");
    AttendanceRuleVo attendanceRuleVo = attendanceRuleList.get(0);
    LocalDate date = LocalDate.parse(entity.getBeginTime());
    Validate.isTrue(attendanceRuleVo.getWorkingDay().contains(String.valueOf(date.getDayOfWeek().getValue()))
        , "开始时间请选择工作日");
    LocalDate endDate = LocalDate.parse(entity.getEndTime());
    Validate.isTrue(attendanceRuleVo.getWorkingDay().contains(String.valueOf(endDate.getDayOfWeek().getValue()))
        , "结束时间请选择工作日");
    // 扣除扣除休息日
    List<SfaApplyTimeInfoDto> nonWorkDateList = new ArrayList<>();
    List<SfaApplyTimeInfoDto> timeInfoList = SfaSignUtils.fillTimeInfoAndCheck(reqVo.getTimeInfoList(), entity.getBeginTime(), entity.getEndTime());
    for (int r = 0; r < timeInfoList.size(); r++) {
      if (r == 0 || r == timeInfoList.size() - 1) { // 开始和结束已经验证过了
        continue;
      }
      // 当天为非工作日
      LocalDate nowDate = LocalDate.parse(timeInfoList.get(r).getTimeStr());
      if (!attendanceRuleVo.getWorkingDay().contains(String.valueOf(nowDate.getDayOfWeek().getValue()))) {
        nonWorkDateList.add(timeInfoList.get(r));
      }
    }
    // 保存非工作日信息
    if (nonWorkDateList.size() > 0) {
      entity.setNonWorkDateListJson(JSON.toJSONString(nonWorkDateList));
    }
//    }
    // 设置返回信息
    entity.setLeaveDuration(countApplyDays.toString());
    return entity;
  }

  /**
   * 追加附件信息
   *
   * @param dto 请假请求
   */
  @Override
  @Transactional(rollbackFor = Exception.class)
  public void addAttachment(SfaLeaveDto dto) {
    SfaLeaveEntity entity = this.sfaLeaveRepository.getById(dto.getId());
    Validate.notNull(entity, "请假信息不存在");
    ProcessBusinessMappingDto businessMappingDto = new ProcessBusinessMappingDto();
    businessMappingDto.setBusinessCode(SfaLeaveConstant.SFA_LEAVE_PROCESS_FROM_TYPE);
    businessMappingDto.setBusinessNo(entity.getId());
    ProcessBusinessMappingVo businessMappingVo = this.processBusinessMappingService.findSignalByConditions(businessMappingDto);
    //如果流程记录不存在,无需校验
    if (Objects.nonNull(businessMappingVo)) {
      // 只有审批中的才能追加附件
      Validate.isTrue(ProcessStatusEnum.COMMIT.getDictCode().equals(businessMappingVo.getProcessStatus()), "只有审批中的请假可以追加附件");
    }
    this.sfaLeaveAttachmentRepository.saveOrUpdateBatch(dto.getAttachmentList());
  }

  /**
   * 提交流程
   *
   * @param dto 请假DTO
   */
  private void commitProcess(SfaLeaveDto dto) {
    //发起流程DTO
    ProcessBusinessDto processBusiness = new ProcessBusinessDto();
    processBusiness.setBusinessNo(dto.getId());
    processBusiness.setBusinessCode(SfaLeaveConstant.SFA_LEAVE_PROCESS_FROM_TYPE);
    processBusiness.setProcessKey(processKey);
    processBusiness.setProcessTitle(SfaLeaveConstant.SFA_LEAVE_PROCESS_NAME);
    JSONObject extData = new JSONObject();
    extData.put("duration", dto.getLeaveDuration());
    processBusiness.setExtData(extData.toJSONString());
    this.processBusinessService.processStart(processBusiness);
  }

  /**
   * 校验调休天数
   *
   * @param dto 请假实体
   */
  private void checkDaysOff(SfaLeaveDto dto) {
    SfaDaysOffEntity sfaDaysOffEntity = this.sfaDaysOffRepository.findByUserName(dto.getUserName());
    Validate.notNull(sfaDaysOffEntity, "用户不存在调休天数，无法调休");
    Validate.isTrue(sfaDaysOffEntity.getAvailableDays().compareTo(new BigDecimal(dto.getLeaveDuration())) > -1, "此用户调休仅" + sfaDaysOffEntity.getAvailableDays().doubleValue() + "天，无法进行调休申请");
  }

  /**
   * 更新调休天数
   *
   * @param deduction 是否抵扣/回退
   * @param entity    请假实体
   */
  @Override
  public void updateDaysOff(SfaLeaveEntity entity, boolean deduction) {
    SfaDaysOffEntity sfaDaysOffEntity = this.sfaDaysOffRepository.findByUserName(entity.getUserName());
    if (deduction) {
      sfaDaysOffEntity.setAvailableDays(sfaDaysOffEntity.getAvailableDays().subtract(new BigDecimal(entity.getLeaveDuration())));
      sfaDaysOffEntity.setUsedDays(sfaDaysOffEntity.getUsedDays().add(new BigDecimal(entity.getLeaveDuration())));
    } else {
      sfaDaysOffEntity.setAvailableDays(sfaDaysOffEntity.getAvailableDays().add(new BigDecimal(entity.getLeaveDuration())));
      sfaDaysOffEntity.setUsedDays(sfaDaysOffEntity.getUsedDays().subtract(new BigDecimal(entity.getLeaveDuration())));
    }
    this.sfaDaysOffRepository.updateById(sfaDaysOffEntity);
  }
}

