package com.biz.crm.sfa.leave.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.sfa.business.leave.sdk.constant.SfaLeaveConstant;
import com.biz.crm.sfa.business.leave.sdk.dto.SfaLeaveApplySummaryDto;
import com.biz.crm.sfa.business.leave.sdk.dto.SfaLeaveListDto;
import com.biz.crm.sfa.business.leave.sdk.enums.LeaveTypeEnum;
import com.biz.crm.sfa.business.leave.sdk.event.LeaveUpperDataListener;
import com.biz.crm.sfa.business.leave.sdk.service.SfaLeaveVoService;
import com.biz.crm.sfa.business.leave.sdk.vo.SfaLeaveApplySummaryVo;
import com.biz.crm.sfa.business.leave.sdk.vo.SfaLeaveAttachmentVo;
import com.biz.crm.sfa.business.leave.sdk.vo.SfaLeaveVo;
import com.biz.crm.sfa.leave.local.dto.SfaLeaveDto;
import com.biz.crm.sfa.leave.local.entity.SfaLeaveAttachmentEntity;
import com.biz.crm.sfa.leave.local.entity.SfaLeaveEntity;
import com.biz.crm.sfa.leave.local.repository.SfaLeaveAttachmentRepository;
import com.biz.crm.sfa.leave.local.repository.SfaLeaveRepository;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessMappingDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.biz.crm.workflow.sdk.service.ProcessBusinessMappingService;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessMappingVo;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 请假vo查询 接口实现
 *
 * @author jerry7
 */
@Slf4j
@Service
public class SfaLeaveVoServiceImpl implements SfaLeaveVoService {

  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private SfaLeaveRepository sfaLeaveRepository;
  @Autowired
  private SfaLeaveAttachmentRepository sfaLeaveAttachmentRepository;
  @Autowired
  private ProcessBusinessMappingService processBusinessMappingService;
  @Autowired(required = false)
  private LeaveUpperDataListener leaveUpperDataListener;

  /**
   * 分页查询请假列表
   *
   * @param dto 查询参数
   * @return 请假列表
   */
  @Override
  public Page<SfaLeaveVo> findByConditions(Pageable pageable, SfaLeaveListDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    //补全时间
    if (StringUtils.isNotEmpty(dto.getApplicationDateStart())) {
      dto.setApplicationDateStart(dto.getApplicationDateStart() + " 00:00:00");
    }
    if (StringUtils.isNotEmpty(dto.getApplicationDateEnd())) {
      dto.setApplicationDateEnd(dto.getApplicationDateEnd() + " 23:59:59");
    }
    //查询数据
    if (Objects.isNull(this.leaveUpperDataListener)) {
      return new Page<>(pageable.getPageNumber(), pageable.getPageSize(), 0);
    }
    dto.setBusinessCode(SfaLeaveConstant.SFA_LEAVE_PROCESS_FROM_TYPE);
    dto.setTenantCode(TenantUtils.getTenantCode());
    Page<SfaLeaveVo> entityPage = this.leaveUpperDataListener.findByConditions(pageable, dto);
    if (CollectionUtils.isEmpty(entityPage.getRecords())) {
      return entityPage;
    }
    this.buildProcessInfo(entityPage.getRecords());
    if (!CollectionUtils.isEmpty(entityPage.getRecords())) {
      entityPage.getRecords().forEach(vo -> {
        vo.setLeaveTypeName(LeaveTypeEnum.getByDictCode(vo.getLeaveType()).getDictCode());
      });
    }
    return entityPage;
  }

  /*
   *
   * 条件查询请假数据
   * @param sfaLeaveListDto
   * @author rentao
   * @date
   */
  @Override
  public List<SfaLeaveVo> findBySfaLeaveListDto(SfaLeaveListDto sfaLeaveListDto) {
    //补全时间
    if (StringUtils.isNotEmpty(sfaLeaveListDto.getApplicationDateStart())) {
      sfaLeaveListDto.setApplicationDateStart(sfaLeaveListDto.getApplicationDateStart() + " 00:00:00");
    }
    if (StringUtils.isNotEmpty(sfaLeaveListDto.getApplicationDateEnd())) {
      sfaLeaveListDto.setApplicationDateEnd(sfaLeaveListDto.getApplicationDateEnd() + " 23:59:59");
    }
    //查询数据
    List<SfaLeaveEntity> entities = this.sfaLeaveRepository.findByConditions(sfaLeaveListDto);
    if (CollectionUtils.isEmpty(entities)) {
      return Lists.newLinkedList();
    }
    //解析审批记录
    List<String> ids = entities.stream().map(SfaLeaveEntity::getId).collect(Collectors.toList());
    ProcessBusinessMappingDto businessMappingDto = new ProcessBusinessMappingDto();
    businessMappingDto.setBusinessCode(SfaLeaveConstant.SFA_LEAVE_PROCESS_FROM_TYPE);
    businessMappingDto.setBusinessNos(ids);
    List<ProcessBusinessMappingVo> businessMappingVos = this.processBusinessMappingService.findMultiByByConditions(businessMappingDto);
    Map<String, String> processStatusMap = CollectionUtils.isEmpty(businessMappingVos) ? Maps.newHashMap() : businessMappingVos.stream()
        .collect(Collectors.toMap(ProcessBusinessMappingVo::getBusinessNo, ProcessBusinessMappingVo::getProcessStatus, (a, b) -> b));
    //剩下审批通过的数据(没有找到流程记录说明没有审批流程,认为是有效数据)
    entities = entities.stream().filter(entity -> {
      String processStatus = processStatusMap.getOrDefault(entity.getId(), ProcessStatusEnum.PASS.getDictCode());
      return ProcessStatusEnum.PASS.getDictCode().equals(processStatus);
    }).collect(Collectors.toList());
    if (CollectionUtils.isEmpty(entities)) {
      return Lists.newLinkedList();
    }
    List<SfaLeaveVo> sfaLeaveVos = (List<SfaLeaveVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, SfaLeaveEntity.class
        , SfaLeaveVo.class, HashSet.class, ArrayList.class);
    sfaLeaveVos.forEach(vo -> {
      vo.setLeaveTypeName(LeaveTypeEnum.getByDictCode(vo.getLeaveType()).getDictCode());
    });
    return sfaLeaveVos;
  }


  /**
   * 根据Id 查询请假详情
   *
   * @param id id
   * @return 请假详情
   */
  @Override
  public SfaLeaveVo queryById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    SfaLeaveEntity sfaLeave = this.sfaLeaveRepository.getById(id);
    if (Objects.isNull(sfaLeave)) {
      return null;
    }
    SfaLeaveVo leaveVo = this.nebulaToolkitService.copyObjectByBlankList(sfaLeave, SfaLeaveVo.class, HashSet.class, ArrayList.class);
    this.buildProcessInfo(Lists.newArrayList(leaveVo));
    List<SfaLeaveAttachmentEntity> attachmentEntities = this.sfaLeaveAttachmentRepository.findBySourceId(id);
    if (!CollectionUtils.isEmpty(attachmentEntities)) {
      leaveVo.setAttachmentList((List<SfaLeaveAttachmentVo>) this.nebulaToolkitService.copyCollectionByWhiteList(attachmentEntities, SfaLeaveAttachmentEntity.class
          , SfaLeaveAttachmentVo.class, HashSet.class, ArrayList.class));
    }
    return leaveVo;
  }

  @Override
  public Page<SfaLeaveApplySummaryVo> findSummaryByConditions(Pageable pageable, SfaLeaveApplySummaryDto dto) {
    //分页获取请假人列表
    if (Objects.isNull(this.leaveUpperDataListener)) {
      return new Page<>(pageable.getPageNumber(), pageable.getPageSize(), 0);
    }
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setBusinessCode(SfaLeaveConstant.SFA_LEAVE_PROCESS_FROM_TYPE);
    dto.setProcessStatus(ProcessStatusEnum.PASS.getDictCode());
    Page<SfaLeaveVo> voPage = this.leaveUpperDataListener.findSummaryByConditions(pageable, dto);
    List<SfaLeaveVo> voList = voPage.getRecords();
    Page<SfaLeaveApplySummaryVo> pageResult = new Page<>(voPage.getCurrent(), voPage.getSize(), voPage.getTotal());
    if (!CollectionUtils.isEmpty(voList)) {
      pageResult.setRecords((List<SfaLeaveApplySummaryVo>) this.nebulaToolkitService.copyCollectionByWhiteList(voList, SfaLeaveVo.class
          , SfaLeaveApplySummaryVo.class, HashSet.class, ArrayList.class));
    }
    //根据请假人列表当前时间内的请假数据详情
    Set<String> userNames = pageResult.getRecords().stream().map(SfaLeaveApplySummaryVo::getUserName).collect(Collectors.toSet());
    SfaLeaveDto sfaLeaveDto = new SfaLeaveDto();
    sfaLeaveDto.setBeginTime(dto.getLeaveDateStart());
    sfaLeaveDto.setEndTime(dto.getLeaveDateEnd());
    sfaLeaveDto.setUserNames(userNames);
    List<SfaLeaveEntity> details = this.sfaLeaveRepository.findDetails(sfaLeaveDto);
    if (CollectionUtils.isEmpty(details)) {
      return pageResult;
    }
    List<String> ids = details.stream().map(SfaLeaveEntity::getId).collect(Collectors.toList());
    ProcessBusinessMappingDto businessMappingDto = new ProcessBusinessMappingDto();
    businessMappingDto.setBusinessCode(SfaLeaveConstant.SFA_LEAVE_PROCESS_FROM_TYPE);
    businessMappingDto.setBusinessNos(ids);
    List<ProcessBusinessMappingVo> businessMappingVos = this.processBusinessMappingService.findMultiByByConditions(businessMappingDto);
    Map<String, String> processStatusMap = CollectionUtils.isEmpty(businessMappingVos) ? Maps.newHashMap() : businessMappingVos.stream()
        .collect(Collectors.toMap(ProcessBusinessMappingVo::getBusinessNo, ProcessBusinessMappingVo::getProcessStatus, (a, b) -> b));
    //剩下审批通过的数据(没有找到流程记录说明没有审批流程,认为是有效数据)
    details = details.stream().filter(entity -> {
      String processStatus = processStatusMap.getOrDefault(entity.getId(), ProcessStatusEnum.PASS.getDictCode());
      return ProcessStatusEnum.PASS.getDictCode().equals(processStatus);
    }).collect(Collectors.toList());
    if (CollectionUtils.isEmpty(details)) {
      return pageResult;
    }
    Map<String, List<SfaLeaveEntity>> userDetailMap = details.stream().collect(Collectors.groupingBy(SfaLeaveEntity::getUserName));
    pageResult.getRecords().forEach(summary -> {
      List<SfaLeaveEntity> leaveEntities = userDetailMap.get(summary.getUserName());
      if (!CollectionUtils.isEmpty(leaveEntities)) {
        for (SfaLeaveEntity e : leaveEntities) {
          LeaveTypeEnum leaveTypeEnum = LeaveTypeEnum.getByDictCode(e.getLeaveType());
          switch (leaveTypeEnum) {
            case DAYS_OFF:
              summary.setDaysOff(summary.getDaysOff().add(new BigDecimal(e.getLeaveDuration())));
              break;
            case ANNUAL_LEAVE:
              summary.setAnnualLeave(summary.getAnnualLeave().add(new BigDecimal(e.getLeaveDuration())));
              break;
            case SICK_LEAVE:
              summary.setSickLeave(summary.getSickLeave().add(new BigDecimal(e.getLeaveDuration())));
              break;
            case FUNERAL_LEAVE:
              summary.setFuneralLeave(summary.getFuneralLeave().add(new BigDecimal(e.getLeaveDuration())));
              break;
            case MARRIAGE_LEAVE:
              summary.setMarriageLeave(summary.getMarriageLeave().add(new BigDecimal(e.getLeaveDuration())));
              break;
            case PERSONAL_LEAVE:
              summary.setPersonalLeave(summary.getPersonalLeave().add(new BigDecimal(e.getLeaveDuration())));
              break;
            case MATERNITY_LEAVE:
              summary.setMaternityLeave(summary.getMaternityLeave().add(new BigDecimal(e.getLeaveDuration())));
              break;
            default:
              summary.setPaternityLeave(summary.getPaternityLeave().add(new BigDecimal(e.getLeaveDuration())));
              break;
          }
        }
      }
    });
    return pageResult;
  }

  /**
   * 封装流程信息
   *
   * @param voList 待封装数据
   */
  private void buildProcessInfo(List<SfaLeaveVo> voList) {
    if (CollectionUtils.isEmpty(voList)) {
      return;
    }
    //解析审批记录
    List<String> ids = voList.stream().map(SfaLeaveVo::getId).collect(Collectors.toList());
    ProcessBusinessMappingDto businessMappingDto = new ProcessBusinessMappingDto();
    businessMappingDto.setBusinessCode(SfaLeaveConstant.SFA_LEAVE_PROCESS_FROM_TYPE);
    businessMappingDto.setBusinessNos(ids);
    List<ProcessBusinessMappingVo> businessMappingVos = this.processBusinessMappingService.findMultiByByConditions(businessMappingDto);
    Map<String, ProcessBusinessVo> processMap = CollectionUtils.isEmpty(businessMappingVos) ? Maps.newHashMap() : businessMappingVos.stream()
        .collect(Collectors.toMap(ProcessBusinessMappingVo::getBusinessNo
            , t -> this.nebulaToolkitService.copyObjectByBlankList(t, ProcessBusinessVo.class, HashSet.class, ArrayList.class)
            , (a, b) -> b));
    //转换数据
    voList.forEach(vo -> vo.setProcessBusiness(processMap.get(vo.getId())));
  }
}
