package com.biz.crm.sfa.business.notice.local.controller;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.constant.CommonConstant;
import com.biz.crm.business.common.sdk.model.LoginUserDetails;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.sfa.business.notice.sdk.dto.NoticePageDto;
import com.biz.crm.sfa.business.notice.sdk.service.NoticeVoService;
import com.biz.crm.sfa.business.notice.sdk.vo.NoticeVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Date;
import java.util.List;

/**
 * 公告表(NoticeVo)表相关的http接口
 *
 * @author songjingen
 * @since 2022-05-27 14:17:38
 */
@RestController
@RequestMapping("/v1/notice/noticeVo")
@Slf4j
@Api(tags = "公告管理：NoticeVo：公告vo")
public class NoticeVoController {
  /**
   * 服务对象
   */
  @Autowired
  private NoticeVoService noticeVoService;
  @Autowired
  private LoginUserService loginUserService;

  /**
   * 分页查询所有数据
   *
   * @param pageable 分页对象
   * @param dto      公告条件dto
   * @return 所有数据
   */
  @ApiOperation(value = "分页查询所有数据")
  @GetMapping("findByConditions")
  public Result<Page<NoticeVo>> findByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                 @ApiParam(name = "dto", value = "公告条件dto") NoticePageDto dto) {
    try {
      Page<NoticeVo> page = this.noticeVoService.findByConditions(pageable, dto);
      return Result.ok(page);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @ApiOperation(value = "通过主键查询单条数据")
  @GetMapping("{id}")
  public Result<NoticeVo> findById(@PathVariable @ApiParam(name = "id", value = "主键id") String id) {
    try {
      NoticeVo notice = this.noticeVoService.findById(id);
      return Result.ok(notice);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 移动端-分页查询（在当前用户范围且已发布）所有数据
   *
   * @param pageable 分页对象
   * @param dto      公告条件dto
   * @return 所有数据
   */
  @ApiOperation(value = "移动端-分页查询（在当前用户范围且已发布）所有数据")
  @GetMapping("findByCurrentUserAndNoticePageDto")
  public Result<Page<NoticeVo>> findByCurrentUserAndNoticePageDto(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                  @ApiParam(name = "dto", value = "公告条件dto") NoticePageDto dto) {
    try {
      LoginUserDetails loginDetails = this.loginUserService.getLoginDetails(LoginUserDetails.class);
      List<String> roles = loginDetails.getRoles();
      if (!CollectionUtils.isEmpty(roles) && !roles.contains(CommonConstant.DEFAULT_ROLE_ADMIN)) {
        dto.setQueryFlag(CommonConstant.QUERY_FLAG_H5);
        dto.setPublish(true);
        dto.setRoles(roles);
        dto.setOrgCode(loginDetails.getOrgCode());
        dto.setNowDate(new Date());
      }
      Page<NoticeVo> page = this.noticeVoService.findByConditions(pageable, dto);
      return Result.ok(page);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 移动端-查询最新的公告（在当前用户范围且已发布）数据
   *
   * @return 所有数据
   */
  @ApiOperation(value = "移动端-查询最新的公告（在当前用户范围且已发布）数据")
  @GetMapping("findLatestByCurrentUser")
  public Result<NoticeVo> findLatestByCurrentUser() {
    try {
      return Result.ok(this.noticeVoService.findLatestByCurrentUser());
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 移动端-根据id和翻页标记查询（在当前用户范围且已发布）数据
   *
   * @return 所有数据
   */
  @ApiOperation(value = "移动端-查询公告详情翻页（在当前用户范围且已发布）数据")
  @GetMapping("findByIdAndTurnFlag")
  public Result<NoticeVo> findByIdAndTurnFlag(@ApiParam(name = "turnFlag", value = "翻页标记：up-上一页，down-下一页") @RequestParam(value = "turnFlag") String turnFlag,
                                              @ApiParam(name = "id", value = "主键id") @RequestParam(value = "id") String id) {
    try {
      return Result.ok(this.noticeVoService.findByIdAndTurnFlag(turnFlag, id));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }
}
