package com.biz.crm.sfa.business.notice.local.service.internal;


import com.biz.crm.business.common.identity.FacturerUserDetails;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.sfa.business.notice.local.entity.Notice;
import com.biz.crm.sfa.business.notice.local.entity.NoticeFile;
import com.biz.crm.sfa.business.notice.local.entity.NoticeScope;
import com.biz.crm.sfa.business.notice.local.repository.NoticeRepository;
import com.biz.crm.sfa.business.notice.local.service.NoticeFileService;
import com.biz.crm.sfa.business.notice.local.service.NoticeScopeService;
import com.biz.crm.sfa.business.notice.local.service.NoticeService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 公告表(Notice)表服务实现类
 *
 * @author songjingen
 * @since 2022-05-27 14:17:38
 */
@Service("noticeService")
public class NoticeServiceImpl implements NoticeService {

  @Autowired
  private NoticeRepository noticeRepository;

  @Autowired
  private NoticeFileService noticeFileService;

  @Autowired
  private NoticeScopeService noticeScopeService;

  @Autowired(required = false)
  private LoginUserService loginUserService;

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public Notice findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.noticeRepository.getById(id);
  }

  /**
   * 新增数据
   *
   * @param notice 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public Notice create(Notice notice) {
    this.createValidate(notice);
    this.noticeRepository.saveOrUpdate(notice);
    //保存附件信息
    List<NoticeFile> fileList = notice.getFileList();
    if (!CollectionUtils.isEmpty(fileList)) {
      fileList.stream().forEach(noticeFile -> noticeFile.setNoticeId(notice.getId()));
      this.noticeFileService.createBatch(fileList);
    }
    //保存范围信息
    List<NoticeScope> scopeList = notice.getScopeList();
    scopeList.stream().forEach(noticeScope -> noticeScope.setNoticeId(notice.getId()));
    this.noticeScopeService.createBatch(scopeList);
    return notice;
  }

  /**
   * 修改新据
   *
   * @param notice 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public Notice update(Notice notice) {
    Validate.notNull(notice, "编辑时，入参对象不能为空！");
    String id = notice.getId();
    Validate.notBlank(id, "编辑时，id不能为空！");
    Notice oldNotice = this.noticeRepository.getById(notice);
    Validate.notNull(oldNotice, "编辑时，未查询到数据！");
    BeanUtils.copyProperties(notice, oldNotice, "id", "tenantCode", "delFlag", "enableStatus", "createAccount", "createTime", "createName");
    this.updateValidate(oldNotice);
    this.noticeRepository.saveOrUpdate(notice);
    ArrayList<String> noticeIds = Lists.newArrayList(id);
    //保存附件信息
    this.noticeFileService.deleteByNoticeIds(noticeIds);
    List<NoticeFile> fileList = notice.getFileList();
    if (!CollectionUtils.isEmpty(fileList)) {
      fileList.stream().forEach(noticeFile -> noticeFile.setNoticeId(id));
      this.noticeFileService.createBatch(fileList);
    }
    //保存范围信息
    this.noticeScopeService.deleteByNoticeIds(noticeIds);
    List<NoticeScope> scopeList = notice.getScopeList();
    scopeList.stream().forEach(noticeScope -> noticeScope.setNoticeId(id));
    this.noticeScopeService.createBatch(scopeList);
    return notice;
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    this.noticeRepository.updateDelFlagByIds(idList, DelFlagStatusEnum.DELETE);
    //删除关联表数据
    this.noticeFileService.deleteByNoticeIds(idList);
    this.noticeScopeService.deleteByNoticeIds(idList);
  }

  /**
   * 启用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 启用结果
   */
  @Transactional
  @Override
  public void enable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "启用数据时，主键集合不能为空！");
    this.noticeRepository.updateEnableStatusByIds(idList, EnableStatusEnum.ENABLE);
  }

  /**
   * 禁用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 禁用结果
   */
  @Transactional
  @Override
  public void disable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "禁用数据时，主键集合不能为空！");
    this.noticeRepository.updateEnableStatusByIds(idList, EnableStatusEnum.DISABLE);
  }

  @Override
  public void updateNoticeTopByIds(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "批量更改置顶状态时，主键集合不能为空！");
    List<Notice> notices = this.noticeRepository.findByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(notices), "批量更改置顶状态时，为查询到数据！");
    notices.stream().forEach(notice -> {
      Boolean noticeTop = notice.getNoticeTop();
      if (noticeTop) {
        notice.setNoticeTop(false);
      } else {
        notice.setNoticeTop(true);
      }
    });
    this.noticeRepository.updateBatchById(notices);
  }

  @Override
  public void updatePublishByIds(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "批量更改发布状态时，主键集合不能为空！");
    List<Notice> notices = this.noticeRepository.findByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(notices), "批量更改发布状态时，为查询到数据！");
    notices.stream().forEach(notice -> {
      Boolean publish = notice.getPublish();
      if (publish) {
        notice.setPublish(false);
      } else {
        notice.setPublish(true);
      }
    });
    this.noticeRepository.updateBatchById(notices);
  }

  /**
   * 创建验证
   *
   * @param notice
   */
  private void createValidate(Notice notice) {
    Validate.notNull(notice, "新增时，对象信息不能为空！");
    notice.setId(null);
    notice.setTenantCode(TenantUtils.getTenantCode());
    notice.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    notice.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    notice.setAllowComment(false);
    Validate.notBlank(notice.getTenantCode(), "新增数据时，租户编号不能为空！");
    Validate.notNull(notice.getAllowComment(), "新增数据时，是否允许评论，true-是，false-否不能为空！");
    Validate.notNull(notice.getPublish(), "新增数据时，是否发布，true-是，false-否不能为空！");
    Validate.notBlank(notice.getTitle(), "新增数据时，标题不能为空！");
    Validate.notNull(notice.getNoticeTop(), "新增数据时，是否置顶，true-是，false-否不能为空！");
    Validate.notBlank(notice.getNoticeType(), "新增数据时，类型不能为空！");
    FacturerUserDetails loginDetails = this.loginUserService.getLoginDetails(FacturerUserDetails.class);
    notice.setPublishOrgCode(loginDetails.getOrgCode());
    notice.setPublishOrgName(loginDetails.getOrgName());
  }

  /**
   * 修改验证
   *
   * @param notice
   */
  private void updateValidate(Notice notice) {
    notice.setAllowComment(false);
    Validate.notNull(notice, "修改时，对象信息不能为空！");
    Validate.notBlank(notice.getId(), "修改数据时，不能为空！");
    Validate.notBlank(notice.getTenantCode(), "修改数据时，租户编号不能为空！");
    Validate.notNull(notice.getAllowComment(), "修改数据时，是否允许评论，true-是，false-否不能为空！");
    Validate.notNull(notice.getPublish(), "修改数据时，是否发布，true-是，false-否不能为空！");
    Validate.notBlank(notice.getTitle(), "修改数据时，标题不能为空！");
    Validate.notNull(notice.getNoticeTop(), "修改数据时，是否置顶，true-是，false-否不能为空！");
    Validate.notBlank(notice.getNoticeType(), "修改数据时，类型不能为空！");
    FacturerUserDetails loginDetails = this.loginUserService.getLoginDetails(FacturerUserDetails.class);
    notice.setPublishOrgCode(loginDetails.getOrgCode());
    notice.setPublishOrgName(loginDetails.getOrgName());
  }
}

