package com.biz.crm.sfa.business.order.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.identity.FacturerUserDetails;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.business.common.sdk.utils.DateTimeValidateUtil;
import com.biz.crm.sfa.business.order.local.entity.OrderEntity;
import com.biz.crm.sfa.business.order.local.entity.OrderProductEntity;
import com.biz.crm.sfa.business.order.local.model.OrderConditionModel;
import com.biz.crm.sfa.business.order.local.repository.OrderRepository;
import com.biz.crm.sfa.business.order.local.service.OrderProductService;
import com.biz.crm.sfa.business.order.local.service.OrderService;
import com.biz.crm.sfa.business.order.sdk.dto.OrderPageDto;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.stream.Collectors;

/**
 * 订单表服务实现类
 *
 * @author ning.zhang
 * @date 2022-07-04 18:49:07
 */
@Slf4j
@Service("orderService")
public class OrderServiceImpl implements OrderService {

  @Autowired
  private OrderRepository orderRepository;
  @Autowired(required = false)
  private GenerateCodeService generateCodeService;
  @Autowired
  private OrderProductService orderProductService;
  @Autowired
  private LoginUserService loginUserService;

  @Override
  @Transactional
  public List<OrderEntity> createBatch(List<OrderEntity> entities) {
    this.createValidate(entities);
    this.orderRepository.saveBatch(entities);
    List<OrderProductEntity> productEntities = entities.stream().flatMap(orderEntity
        -> orderEntity.getProducts().stream().map(productEntity -> {
      productEntity.setOrderId(orderEntity.getId());
      return productEntity;
    })).collect(Collectors.toList());
    this.orderProductService.createBatch(productEntities);
    return entities;
  }

  @Override
  public Page<OrderEntity> findByConditions(Pageable pageable, OrderPageDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new OrderPageDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    return this.orderRepository.findByConditions(pageable, dto);
  }

  @Override
  public OrderEntity findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    OrderConditionModel model = new OrderConditionModel();
    model.setOrderIds(Sets.newHashSet(id));
    model.setTenantCode(TenantUtils.getTenantCode());
    List<OrderEntity> orderEntities = this.orderRepository.findByOrderConditionModel(model);
    if (CollectionUtils.isEmpty(orderEntities)) {
      return null;
    }
    return orderEntities.get(0);
  }

  /**
   * 创建验证
   *
   * @param entities 验证对象
   */
  private void createValidate(List<OrderEntity> entities) {
    Validate.isTrue(!CollectionUtils.isEmpty(entities), "新增数据时，订单对象信息不能为空！");
    FacturerUserDetails loginDetails = this.loginUserService.getLoginDetails(FacturerUserDetails.class);
    entities.forEach(orderEntity -> {
      orderEntity.setId(null);
      orderEntity.setPostCode(loginDetails.getPostCode());
      orderEntity.setPostName(loginDetails.getPostName());
      orderEntity.setOrgCode(loginDetails.getOrgCode());
      orderEntity.setOrgName(loginDetails.getOrgName());
      orderEntity.setOrderCode(this.generateCodeService.generateCode("SOR", 1).get(0));
      Validate.notBlank(orderEntity.getSuperiorCustomerCode(), "新增数据时，订单上级客户编码不能为空！");
      Validate.notBlank(orderEntity.getSuperiorCustomerName(), "新增数据时，订单上级客户名称不能为空！");
      Validate.notBlank(orderEntity.getContactName(), "新增数据时，订单联系人不能为空！");
      Validate.notBlank(orderEntity.getContactPhone(), "新增数据时，订单联系人电话不能为空！");
      Validate.notBlank(orderEntity.getDeliveryAddress(), "新增数据时，订单收货地址不能为空！");
      Validate.notBlank(orderEntity.getExpectArrivalDate(), "新增数据时，订单期望到货日期不能为空！");
      Validate.notBlank(orderEntity.getSourceType(), "新增数据时，订单来源类型不能为空！");
      Validate.notBlank(orderEntity.getSourceCode(), "新增数据时，订单来源业务编码不能为空！");
      Validate.notBlank(orderEntity.getTerminalCode(), "新增数据时，终端编码不能为空！");
      Validate.notBlank(orderEntity.getTerminalName(), "新增数据时，终端名称不能为空！");
      Validate.isTrue(DateTimeValidateUtil.validateDate(orderEntity.getExpectArrivalDate())
          , "非法的期望到货日期格式，可用的格式：[yyyy-MM-dd]");
      Validate.isTrue(!CollectionUtils.isEmpty(orderEntity.getProducts()), "新增数据时，订单商品信息数据不能为空！");
    });
  }
}
