package com.biz.crm.sfa.business.order.local.service.register;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.biz.crm.sfa.business.order.local.entity.OrderEntity;
import com.biz.crm.sfa.business.order.local.entity.OrderProductEntity;
import com.biz.crm.sfa.business.order.local.model.OrderConditionModel;
import com.biz.crm.sfa.business.order.local.repository.OrderRepository;
import com.biz.crm.sfa.business.order.local.service.OrderService;
import com.biz.crm.sfa.business.order.sdk.constant.OrderConstant;
import com.biz.crm.sfa.business.order.sdk.enums.OrderSourceType;
import com.biz.crm.sfa.business.template.action.tpm.sdk.constant.TemplateActionTpmConstant;
import com.biz.crm.sfa.business.template.action.tpm.sdk.register.ActionTpmExecuteRegister;
import com.biz.crm.sfa.business.template.action.tpm.sdk.vo.ActionTpmExecuteVo;
import com.biz.crm.tpm.business.activities.sdk.vo.ActivitiesDetailSerialVo;
import com.biz.crm.tpm.business.activities.sdk.vo.ActivitiesVo;
import com.bizunited.nebula.common.util.JsonUtils;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * TPM活动执行事件监听,订单模块实现
 *
 * @author ning.zhang
 * @date 2022/7/4
 */
@Component
public class OrderActionTpmExecuteRegisterImpl implements ActionTpmExecuteRegister {

  @Autowired
  private OrderService orderService;
  @Autowired
  private OrderRepository orderRepository;

  @Override
  public String relationKey() {
    return OrderConstant.ORDER_MODULE_KEY;
  }

  @Override
  public String relationName() {
    return OrderConstant.ORDER_MODULE_NAME;
  }

  @Override
  @Transactional
  public void onCreate(ActionTpmExecuteVo vo) {
    Validate.notNull(vo, "新增数据时，对象信息不能为空！");
    List<JSONObject> list = Objects.isNull(vo.getRelationMap()) ? null : vo.getRelationMap().get(OrderConstant.ORDER_MODULE_KEY);
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    List<OrderEntity> entities = list.stream().map(jsonObject -> {
      OrderEntity entity = JsonUtils.json2Obj(jsonObject.toJSONString(), OrderEntity.class);
      entity.setSourceType(OrderSourceType.ACTION_ORDER.getDictCode());
      entity.setSourceCode(vo.getExecuteCode());
      entity.setTenantCode(TenantUtils.getTenantCode());
      entity.setTerminalCode(vo.getTerminalCode());
      entity.setTerminalName(vo.getTerminalName());
      return entity;
    }).collect(Collectors.toList());
    this.orderService.createBatch(entities);
    //封装活动明细流水信息以供TPM处理资金使用信息
    vo.setActivitiesDetailSerials(ObjectUtils.defaultIfNull(vo.getActivitiesDetailSerials(), Lists.newArrayList()));
    vo.getActivitiesDetailSerials().addAll(entities.stream().map(entity -> {
      ActivitiesDetailSerialVo serialVo = new ActivitiesDetailSerialVo();
      serialVo.setActivitiesCode(vo.getActionCode());
      serialVo.setActivitiesDetailCode(vo.getActionDetailCode());
      serialVo.setActivitiesName(vo.getActionName());
      serialVo.setBtNo(vo.getExecuteCode());
      serialVo.setCustomerCode(entity.getSuperiorCustomerCode());
      serialVo.setCustomerName(entity.getSuperiorCustomerName());
      serialVo.setSerialNo(entity.getOrderCode());
      serialVo.setSerialPrice(BigDecimal.valueOf(entity.getProducts().stream()
          .mapToDouble(value -> value.getQuantity().multiply(value.getPrice()).doubleValue()).sum()));
      serialVo.setSerialTime(entity.getCreateTime());
      serialVo.setCreateTime(entity.getCreateTime());
      serialVo.setTerminalCode(entity.getTerminalCode());
      serialVo.setTerminalName(entity.getTerminalName());
      serialVo.setType(1);
      return serialVo;
    }).collect(Collectors.toList()));
  }

  @Override
  public Map<String, List<JSONObject>> onRequestByExecuteCodes(Set<String> executeCodes) {
    if (CollectionUtils.isEmpty(executeCodes)) {
      return null;
    }
    OrderConditionModel model = new OrderConditionModel();
    model.setTenantCode(TenantUtils.getTenantCode());
    model.setSourceCodes(executeCodes);
    model.setSourceType(OrderSourceType.ACTION_ORDER.getDictCode());
    List<OrderEntity> entities = this.orderRepository.findByOrderConditionModel(model);
    if (CollectionUtils.isEmpty(entities)) {
      return null;
    }
    return entities.stream().collect(
        Collectors.groupingBy(OrderEntity::getSourceCode,
            Collectors.mapping(o -> {
              JSONObject jsonObject = JsonUtils.toJSONObject(o);
              jsonObject.put("createTime", DateFormatUtils.format(o.getCreateTime(), TemplateActionTpmConstant.YYYY_MM_DD_HH_MM_SS_STR));
              jsonObject.put("modifyTime", DateFormatUtils.format(o.getModifyTime(), TemplateActionTpmConstant.YYYY_MM_DD_HH_MM_SS_STR));
              return jsonObject;
            }, Collectors.toList())));
  }
}
