package com.biz.crm.sfa.business.overtime.local.service.internal;

import cn.hutool.core.date.DateTime;
import com.alibaba.fastjson.JSON;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.LoginUserDetails;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.business.common.sdk.utils.DateTimeValidateUtil;
import com.biz.crm.mdm.business.position.sdk.service.PositionVoService;
import com.biz.crm.mdm.business.position.sdk.vo.PositionVo;
import com.biz.crm.mdm.business.user.sdk.vo.UserVo;
import com.biz.crm.sfa.business.attendance.sdk.dto.RuleConditionDto;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceRuleScopeType;
import com.biz.crm.sfa.business.attendance.sdk.enums.AttendanceSpecialDateType;
import com.biz.crm.sfa.business.attendance.sdk.service.AttendanceRuleVoService;
import com.biz.crm.sfa.business.attendance.sdk.vo.AttendanceRuleScopeVo;
import com.biz.crm.sfa.business.attendance.sdk.vo.AttendanceRuleSpecialDateVo;
import com.biz.crm.sfa.business.attendance.sdk.vo.AttendanceRuleVo;
import com.biz.crm.sfa.business.holiday.sdk.dto.HolidayConditionDto;
import com.biz.crm.sfa.business.holiday.sdk.service.HolidayVoService;
import com.biz.crm.sfa.business.holiday.sdk.vo.HolidayDateVo;
import com.biz.crm.sfa.business.holiday.sdk.vo.HolidayVo;
import com.biz.crm.sfa.business.overtime.local.entity.OvertimeApplyEntity;
import com.biz.crm.sfa.business.overtime.local.entity.OvertimeApplyTimeEntity;
import com.biz.crm.sfa.business.overtime.local.entity.OvertimeRuleEntity;
import com.biz.crm.sfa.business.overtime.local.repository.OvertimeApplyRepository;
import com.biz.crm.sfa.business.overtime.local.repository.OvertimeRuleRepository;
import com.biz.crm.sfa.business.overtime.local.service.OvertimeApplyService;
import com.biz.crm.sfa.business.overtime.local.service.OvertimeApplyTimeService;
import com.biz.crm.sfa.business.overtime.sdk.constant.OvertimeConstant;
import com.biz.crm.sfa.business.overtime.sdk.dto.OvertimeApplyConditionDto;
import com.biz.crm.sfa.business.overtime.sdk.dto.OvertimeApplyDto;
import com.biz.crm.sfa.business.overtime.sdk.dto.OvertimeApplyTimeDto;
import com.biz.crm.sfa.business.overtime.sdk.enums.OvertimeTimeType;
import com.biz.crm.sfa.business.overtime.sdk.enums.OvertimeType;
import com.biz.crm.workflow.sdk.constant.enums.ActApproveStatusEnum;
import com.biz.crm.workflow.sdk.dto.OrgInfoDto;
import com.biz.crm.workflow.sdk.dto.PositionInfoDto;
import com.biz.crm.workflow.sdk.dto.StartProcessDto;
import com.biz.crm.workflow.sdk.listener.OrgInfoListener;
import com.biz.crm.workflow.sdk.listener.PositionInfoListener;
import com.biz.crm.workflow.sdk.listener.ProcessListener;
import com.biz.crm.workflow.sdk.vo.OrgVo;
import com.biz.crm.workflow.sdk.vo.response.CommitWorkflowResponse;
import com.biz.crm.workflow.sdk.vo.response.OrgInfoResponse;
import com.biz.crm.workflow.sdk.vo.response.PositionInfoResponse;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 加班申请表服务实现类
 *
 * @author ning.zhang
 * @date 2022-05-30 11:05:26
 */
@Slf4j
@Service("overtimeApplyService")
public class OvertimeApplyServiceImpl implements OvertimeApplyService {

  @Autowired
  private OvertimeApplyRepository overtimeApplyRepository;
  @Autowired
  private GenerateCodeService generateCodeService;
  @Autowired
  private LoginUserService loginUserService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private HolidayVoService holidayVoService;
  @Autowired
  private AttendanceRuleVoService attendanceRuleVoService;
  @Autowired
  private OvertimeApplyTimeService overtimeApplyTimeService;
  @Autowired
  private NebulaNetEventClient nebulaNetEventClient;
  @Autowired
  private OvertimeRuleRepository overtimeRuleRepository;
  @Autowired
  private PositionVoService positionVoService;

  @Override
  @Transactional
  public OvertimeApplyEntity create(OvertimeApplyDto dto) {
    this.createValidation(dto);
    OvertimeApplyEntity entity = this.nebulaToolkitService
        .copyObjectByWhiteList(dto, OvertimeApplyEntity.class, HashSet.class, ArrayList.class, "recordIds");
    entity.setTimeInfoListJson(JSON.toJSONString(dto.getTimeList()));
    entity.setApplyCode(this.generateCodeService.generateCode("OA", 1).get(0));
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    entity.setProcessStatus(ActApproveStatusEnum.APPROVING.getCode());
    this.overtimeApplyRepository.save(entity);
    dto.setId(entity.getId());
    dto.setApplyCode(entity.getApplyCode());
    entity.setProcessNumber(this.commitProcess(dto));
    this.overtimeApplyRepository.updateById(entity);
    return entity;
  }

  @Override
  @Transactional
  public OvertimeApplyEntity update(OvertimeApplyDto dto) {
    this.updateValidation(dto);
    OvertimeApplyEntity applyEntity = this.overtimeApplyRepository.getById(dto.getId());
    Validate.notNull(applyEntity, "加班申请记录不存在!");
    Validate.isTrue(ActApproveStatusEnum.REJECTED.getCode().equals(applyEntity.getProcessStatus())
        || ActApproveStatusEnum.INTERRUPT.getCode().equals(applyEntity.getProcessStatus()), "该申请不支持编辑");
    //驳回状态直接创建新的申请,追回的使用原来的申请继续提交
    if (ActApproveStatusEnum.REJECTED.getCode().equals(applyEntity.getProcessStatus())) {
      dto.setId(null);
    }
    OvertimeApplyEntity entity = this.nebulaToolkitService
        .copyObjectByWhiteList(dto, OvertimeApplyEntity.class, HashSet.class, ArrayList.class, "recordIds");
    entity.setTimeInfoListJson(JSON.toJSONString(dto.getTimeList()));
    entity.setApplyCode(StringUtils.isNotBlank(entity.getId()) ? applyEntity.getApplyCode() : this.generateCodeService.generateCode("OA", 1).get(0));
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    entity.setProcessStatus(ActApproveStatusEnum.APPROVING.getCode());
    this.overtimeApplyRepository.saveOrUpdate(entity);
    dto.setId(entity.getId());
    dto.setApplyCode(entity.getApplyCode());
    entity.setProcessNumber(this.commitProcess(dto));
    this.overtimeApplyRepository.updateById(entity);
    return entity;
  }

  /**
   * 加班申请工作流进行审批，提交成功返回流程实例ID，提交失败则抛出异常
   *
   * @param dto 加班申请信息
   */
  private String commitProcess(OvertimeApplyDto dto) {
    StartProcessDto startProcessDto = new StartProcessDto();
    startProcessDto.setProcessKey(OvertimeConstant.OVERTIME_APPLY_PROCESS_KEY);
    startProcessDto.setBusinessId(dto.getId());
    startProcessDto.setBusinessNo(dto.getApplyCode());
    startProcessDto.setRemark(dto.getProcessRemark());
    startProcessDto.setFormType(OvertimeConstant.OVERTIME_APPLY_PROCESS_FORM_TYPE);
    startProcessDto.setProcessTitle(OvertimeConstant.OVERTIME_APPLY_PROCESS_NAME);
    startProcessDto.setMenuCode(dto.getCompetenceCode());
    SerializableBiConsumer<ProcessListener, StartProcessDto> sf = ProcessListener::onStartProcess;
    CommitWorkflowResponse response = (CommitWorkflowResponse) this.nebulaNetEventClient.directPublish(startProcessDto, ProcessListener.class, sf);
    Validate.isTrue(StringUtils.isNotEmpty(response.getProcessInstanceId()), "发起流程失败！");
    return response.getProcessInstanceId();
  }

  /**
   * 在创建overtimeApply模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(OvertimeApplyDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getBeginTime(), "缺失开始时间");
    Validate.notBlank(dto.getEndTime(), "缺失结束时间");
    Validate.isTrue(Objects.nonNull(dto.getApplyDuration()) && dto.getApplyDuration().compareTo(BigDecimal.ZERO) > 0, "申请时长错误");
    Validate.isTrue(DateTimeValidateUtil.validateDate(dto.getBeginTime()), "非法的开始时间格式，可用的格式：[yyyy-MM-dd]");
    Validate.isTrue(DateTimeValidateUtil.validateDate(dto.getEndTime()), "非法的结束时间格式，可用的格式：[yyyy-MM-dd]");
    Validate.isTrue(dto.getBeginTime().compareTo(dto.getEndTime()) <= 0, "开始时间不能大于结束时间");
    LoginUserDetails loginDetails = this.loginUserService.getLoginDetails(LoginUserDetails.class);
    dto.setUserName(loginDetails.getAccount());
    dto.setRealName(loginDetails.getUsername());
    dto.setOrgCode(loginDetails.getOrgCode());
    dto.setOrgName(loginDetails.getOrgName());
    dto.setPosCode(loginDetails.getPostCode());
    dto.setPosName(loginDetails.getPostName());
    dto.setApplyDate(LocalDateTime.now().format(OvertimeConstant.YYYY_MM_DD_HH_MM_SS));
    //获取组织详情，已取得上级组织编码和名称
    SerializableBiConsumer<OrgInfoListener, OrgInfoDto> orgConsumer = OrgInfoListener::findByOrgCodes;
    OrgInfoDto orgInfoDto = new OrgInfoDto();
    orgInfoDto.setOrgCode(dto.getOrgCode());
    OrgInfoResponse orgResponse = (OrgInfoResponse) nebulaNetEventClient.directPublish(orgInfoDto, OrgInfoListener.class, orgConsumer);
    if (ObjectUtils.isNotEmpty(orgResponse) && !CollectionUtils.isEmpty(orgResponse.getRoleInfoVos())) {
      OrgVo orgVo = orgResponse.getRoleInfoVos().get(0);
      dto.setParentOrgCode(orgVo.getParentCode());
      dto.setParentOrgName(orgVo.getParentName());
    }
    List<OvertimeApplyTimeEntity> timeEntities = this.validateApplyTime(dto);
    this.validateAttendanceRule(dto, timeEntities);
  }

  /**
   * 在修改overtimeApply模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(OvertimeApplyDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getId(), "修改信息时，id不能为空！");
    Validate.notBlank(dto.getBeginTime(), "缺失开始时间");
    Validate.notBlank(dto.getEndTime(), "缺失结束时间");
    Validate.isTrue(Objects.nonNull(dto.getApplyDuration()) && dto.getApplyDuration().compareTo(BigDecimal.ZERO) > 0, "申请时长错误");
    Validate.isTrue(DateTimeValidateUtil.validateDate(dto.getBeginTime()), "非法的开始时间格式，可用的格式：[yyyy-MM-dd]");
    Validate.isTrue(DateTimeValidateUtil.validateDate(dto.getEndTime()), "非法的结束时间格式，可用的格式：[yyyy-MM-dd]");
    Validate.isTrue(dto.getBeginTime().compareTo(dto.getEndTime()) <= 0, "开始时间不能大于结束时间");
    LoginUserDetails loginDetails = this.loginUserService.getLoginDetails(LoginUserDetails.class);
    dto.setUserName(loginDetails.getAccount());
    dto.setOrgCode(loginDetails.getOrgCode());
    dto.setPosCode(loginDetails.getPostCode());
    dto.setApplyDate(LocalDateTime.now().format(OvertimeConstant.YYYY_MM_DD_HH_MM_SS));
    List<OvertimeApplyTimeEntity> timeEntities = this.validateApplyTime(dto);
    this.validateAttendanceRule(dto, timeEntities);
  }

  /**
   * 校验考勤规则信息
   *
   * @param dto          参数dto
   * @param timeEntities 申请时间明细信息
   */
  private void validateAttendanceRule(OvertimeApplyDto dto, List<OvertimeApplyTimeEntity> timeEntities) {
    dto.setOvertimeType(OvertimeType.NORMAL.getDictCode());
    List<OvertimeRuleEntity> overtimeRuleEntities = this.overtimeRuleRepository.findByAbideFlag(BooleanEnum.TRUE.getCapital(), TenantUtils.getTenantCode());
    Validate.isTrue(!CollectionUtils.isEmpty(overtimeRuleEntities), "当前用户未配置考勤规则");
    RuleConditionDto conditionDto = new RuleConditionDto();
    conditionDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    conditionDto.setOrgCodes(Lists.newArrayList(dto.getOrgCode()));
    conditionDto.setRuleCodes(overtimeRuleEntities.stream().map(OvertimeRuleEntity::getRuleCode).collect(Collectors.toList()));
    List<AttendanceRuleVo> attendanceRuleList = this.attendanceRuleVoService.findByRuleConditionDto(conditionDto);
    Validate.isTrue(!CollectionUtils.isEmpty(attendanceRuleList), "当前用户未配置考勤规则");
    attendanceRuleList = attendanceRuleList.stream().filter(ruleVo -> {
      List<AttendanceRuleScopeVo> scopeList = ruleVo.getScopeList();
      if (CollectionUtils.isEmpty(scopeList)) {
        return false;
      }
      Map<String, List<AttendanceRuleScopeVo>> scopeMap = scopeList.stream().collect(Collectors.groupingBy(AttendanceRuleScopeVo::getScopeType));
      //如果规则只有组织范围,那么匹配成功
      if (scopeMap.keySet().size() == 1 && !CollectionUtils.isEmpty(scopeMap.get(AttendanceRuleScopeType.ORG.getDictCode()))) {
        return true;
      }
      //如果规则既有组织范围又有职级范围,那么需要匹配到用户的职级编码才算匹配成功
      if (scopeMap.keySet().size() > 1) {
        List<PositionVo> positionList = this.positionVoService.findByIdsOrCodes(Lists.newLinkedList(), Lists.newArrayList(dto.getPosCode()));
        List<AttendanceRuleScopeVo> scopeVoList = scopeMap.getOrDefault(AttendanceRuleScopeType.POSITION_LEVEL.getDictCode(), Lists.newArrayList());
        if (CollectionUtils.isEmpty(positionList)) {
          return false;
        }
        scopeVoList = scopeVoList.stream().filter(scopeVo -> scopeVo.getScopeCode().equals(positionList.get(0).getPositionLevelCode())).collect(Collectors.toList());
        return !CollectionUtils.isEmpty(scopeVoList);
      }
      return false;
    }).collect(Collectors.toList());
    Validate.isTrue(!CollectionUtils.isEmpty(attendanceRuleList), "当前用户未配置考勤规则");
    AttendanceRuleVo attendanceRuleVo = attendanceRuleList.get(0);
    dto.setRuleCode(attendanceRuleVo.getRuleCode());
    List<HolidayDateVo> holidayDateList = this.validateHoliday(dto, timeEntities, attendanceRuleVo);
    //节假日允许加班
    if (!CollectionUtils.isEmpty(holidayDateList)) {
      return;
    }
    //特殊日期必须打卡,不能申请加班
    if (!CollectionUtils.isEmpty(attendanceRuleVo.getSpecialDateList())) {
      List<AttendanceRuleSpecialDateVo> specialDateList = attendanceRuleVo.getSpecialDateList().stream().filter(specialDateVo -> Objects.nonNull(specialDateVo)
          && dto.getBeginTime().compareTo(specialDateVo.getSpecialDate()) <= 0
          && dto.getEndTime().compareTo(specialDateVo.getSpecialDate()) >= 0
          && AttendanceSpecialDateType.MUST_CLOCK.getDictCode().equals(specialDateVo.getSpecialDateType())).collect(Collectors.toList());
      if (!CollectionUtils.isEmpty(specialDateList)) {
        throw new IllegalArgumentException(String.format("日期不能申请加班-工作日：%s", specialDateList.get(0).getSpecialDate()));
      }
    }
    //工作日不能申请加班
    timeEntities.forEach(entity -> {
      LocalDate date = LocalDate.parse(entity.getItemTime());
      Validate.isTrue(!attendanceRuleVo.getWorkingDay().contains(String.valueOf(date.getDayOfWeek().getValue()))
          , String.format("日期不能申请加班-工作日：%s", date.format(OvertimeConstant.YYYY_MM_DD)));
    });
  }

  /**
   * 校验节假日
   *
   * @param dto              参数dto
   * @param timeEntities     申请时间明细信息
   * @param attendanceRuleVo 考勤规则
   * @return 匹配的节假日
   */
  private List<HolidayDateVo> validateHoliday(OvertimeApplyDto dto, List<OvertimeApplyTimeEntity> timeEntities, AttendanceRuleVo attendanceRuleVo) {
    List<HolidayDateVo> holidayDateList = Lists.newArrayList();
    HolidayConditionDto holidayConditionDto = new HolidayConditionDto();
    holidayConditionDto.setYears(Sets.newHashSet(String.valueOf(LocalDate.parse(dto.getEndTime()).getYear())
        , String.valueOf(LocalDate.parse(dto.getBeginTime()).getYear())));
    List<HolidayVo> holidayList = this.holidayVoService.findByHolidayConditionDto(holidayConditionDto);
    Map<String, HolidayVo> holidayMap = CollectionUtils.isEmpty(holidayList) ? Maps.newHashMap()
        : holidayList.stream().collect(Collectors.toMap(HolidayVo::getYear, t -> t, (a, b) -> b));
    dto.setHolidayDays(BigDecimal.ZERO);
    if (BooleanEnum.TRUE.getCapital().equals(attendanceRuleVo.getSyncHoliday())) {
      holidayConditionDto.getYears().forEach(year -> {
        Validate.notNull(holidayMap.get(year), String.format("请联系管理员配置%s年度节假日信息", year));
      });
      Map<String, OvertimeApplyTimeEntity> timeEntityMap = timeEntities.stream().collect(Collectors.toMap(OvertimeApplyTimeEntity::getItemTime, t -> t));
      //合并所有节假日日期,过滤不在时间范围内的节假日,转换为是否计算薪资列表
      holidayDateList = holidayList.stream()
          .flatMap(holidayVo -> holidayVo.getInfoList().stream().flatMap(holidayInfoVo -> holidayInfoVo.getDateList().stream()))
          .collect(Collectors.toList()).stream()
          .filter(holidayDateVo -> Objects.nonNull(holidayDateVo)
              && dto.getBeginTime().compareTo(holidayDateVo.getHolidayDate()) <= 0 && dto.getEndTime().compareTo(holidayDateVo.getHolidayDate()) >= 0)
          .collect(Collectors.toList());
      if (CollectionUtils.isEmpty(holidayDateList)) {
        return holidayDateList;
      }
      //如果是节假日直接结束校验
      holidayDateList.forEach(holidayDateVo -> {
        //是否包含计算薪资节假日,并计算薪资节假日天数
        if (holidayDateVo.getCalculationSalary().equals(BooleanEnum.TRUE.getCapital())) {
          OvertimeApplyTimeEntity timeEntity = timeEntityMap.get(holidayDateVo.getHolidayDate());
          dto.setOvertimeType(OvertimeType.HOLIDAY.getDictCode());
          dto.setHolidayDays(dto.getHolidayDays().add(OvertimeTimeType.ALL_DAY.getDictCode().equals(timeEntity.getTimeType()) ? BigDecimal.ONE : BigDecimal.valueOf(0.5)));
        }
      });
    }
    return holidayDateList;
  }

  /**
   * 校验申请时间明细信息
   *
   * @param dto 参数dto
   * @return 申请时间明细信息
   */
  private List<OvertimeApplyTimeEntity> validateApplyTime(OvertimeApplyDto dto) {
    List<OvertimeApplyTimeDto> timeList = dto.getTimeList();
    Validate.isTrue(!CollectionUtils.isEmpty(timeList) && timeList.size() <= 2, "日期时间明细错误");
    timeList.forEach(timeDto -> {
      Validate.notBlank(timeDto.getItemTime(), "缺失明细时间");
      Validate.isTrue(DateTimeValidateUtil.validateDate(timeDto.getItemTime()), "非法的明细时间格式，可用的格式：[yyyy-MM-dd]");
      Validate.notNull(OvertimeTimeType.getByDictCode(timeDto.getTimeType()), "时间类型错误");
    });
    List<OvertimeApplyTimeEntity> timeEntities = this.overtimeApplyTimeService.buildTimeEntities(timeList);
    Validate.isTrue(dto.getBeginTime().compareTo(timeEntities.get(0).getItemTime()) == 0, "申请开始时间与明细开始时间不一致");
    Validate.isTrue(dto.getEndTime().compareTo(timeEntities.get(timeEntities.size() - 1).getItemTime()) == 0, "申请结束时间与明细结束时间不一致");
    //申请时长计算
    BigDecimal applyDuration = timeEntities.stream().map(entity
        -> OvertimeTimeType.getByDictCode(entity.getTimeType()).getDays()).reduce(BigDecimal.ZERO, BigDecimal::add);
    Validate.isTrue(applyDuration.compareTo(dto.getApplyDuration()) == 0, "申请时长和明细时间不一致");
    //校验时间冲突
    this.validateConflictTime(dto, timeEntities);
    return timeEntities;
  }

  /**
   * 校验时间冲突
   *
   * @param dto          参数dto
   * @param timeEntities 申请时间明细信息
   */
  private void validateConflictTime(OvertimeApplyDto dto, List<OvertimeApplyTimeEntity> timeEntities) {
    OvertimeApplyConditionDto applyConditionDto = new OvertimeApplyConditionDto();
    //编辑操作排除自身
    if (StringUtils.isNotBlank(dto.getId())) {
      applyConditionDto.setExcludeIds(Lists.newArrayList(dto.getId()));
    }
    applyConditionDto.setTenantCode(dto.getTenantCode());
    applyConditionDto.setEndTime(dto.getEndTime());
    applyConditionDto.setBeginTime(dto.getBeginTime());
    applyConditionDto.setUserName(dto.getUserName());
    applyConditionDto.setExcludeProcessStatusList(Lists.newArrayList(ActApproveStatusEnum.REJECTED.getCode(), ActApproveStatusEnum.INTERRUPT.getCode()));
    List<OvertimeApplyEntity> applyEntities = this.overtimeApplyRepository.findByOvertimeApplyConditionDto(applyConditionDto);
    if (CollectionUtils.isEmpty(applyEntities)) {
      return;
    }
    //构建时间-类型列表
    List<String> repeatTimeList = applyEntities.stream()
        //转换填充合并所有申请的时间明细信息
        .flatMap(entity -> this.overtimeApplyTimeService.buildTimeEntities(JSON.parseArray(entity.getTimeInfoListJson(), OvertimeApplyTimeDto.class)).stream())
        //拆分全天类型为上午和下午
        .flatMap(entity -> {
          Set<String> typeList = Sets.newHashSet(entity.getTimeType());
          if (OvertimeTimeType.ALL_DAY.getDictCode().equals(entity.getTimeType())) {
            typeList.add(OvertimeTimeType.FORENOON.getDictCode());
            typeList.add(OvertimeTimeType.AFTERNOON.getDictCode());
          }
          return typeList.stream().map(type -> {
            OvertimeApplyTimeEntity timeEntity = new OvertimeApplyTimeEntity();
            timeEntity.setTimeType(type);
            timeEntity.setItemTime(entity.getItemTime());
            return timeEntity;
          });
        })
        .map(entity -> String.format("%s%s", entity.getItemTime(), entity.getTimeType()))
        .collect(Collectors.toList());
    timeEntities.forEach(entity -> {
      Set<String> typeList = Sets.newHashSet(entity.getTimeType());
      if (OvertimeTimeType.ALL_DAY.getDictCode().equals(entity.getTimeType())) {
        typeList.add(OvertimeTimeType.FORENOON.getDictCode());
        typeList.add(OvertimeTimeType.AFTERNOON.getDictCode());
      }
      typeList.forEach(type -> {
        Validate.isTrue(!repeatTimeList.contains(String.format("%s%s", entity.getItemTime(), type))
            , String.format("已存在%s的申请，请重新选择日期！", entity.getItemTime()));
      });
    });
  }
}
