package com.biz.crm.sfa.business.overtime.local.service.notifier;

import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.sfa.business.attendance.sdk.dto.RuleNoWorkAbideDataDto;
import com.biz.crm.sfa.business.attendance.sdk.event.AttendanceRuleEventListener;
import com.biz.crm.sfa.business.attendance.sdk.vo.AttendanceRuleNoWorkAbideDataVo;
import com.biz.crm.sfa.business.attendance.sdk.vo.AttendanceRuleNoWorkAbideVo;
import com.biz.crm.sfa.business.attendance.sdk.vo.AttendanceRuleVo;
import com.biz.crm.sfa.business.overtime.local.entity.OvertimeApplyEntity;
import com.biz.crm.sfa.business.overtime.local.entity.OvertimeRuleEntity;
import com.biz.crm.sfa.business.overtime.local.repository.OvertimeApplyRepository;
import com.biz.crm.sfa.business.overtime.local.repository.OvertimeRuleRepository;
import com.biz.crm.sfa.business.overtime.sdk.constant.OvertimeConstant;
import com.biz.crm.sfa.business.overtime.sdk.dto.OvertimeApplyConditionDto;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessMappingDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.biz.crm.workflow.sdk.service.ProcessBusinessMappingService;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessMappingVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 加班考勤规则事件通知实现类
 *
 * @author ning.zhang
 * @date 2022/6/1
 */
@Component
public class OvertimeAttendanceRuleEventListenerImpl implements AttendanceRuleEventListener {

  @Autowired
  private OvertimeRuleRepository overtimeRuleRepository;
  @Autowired
  private OvertimeApplyRepository overtimeApplyRepository;
  @Autowired
  private ProcessBusinessMappingService processBusinessMappingService;

  @Override
  @Transactional
  public void onCreate(AttendanceRuleVo vo) {
    OvertimeRuleEntity ruleEntity = this.convertVoToEntity(vo);
    if (Objects.isNull(ruleEntity)) {
      return;
    }
    this.overtimeRuleRepository.save(ruleEntity);
  }

  @Override
  @Transactional
  public void onUpdate(AttendanceRuleVo oldVo, AttendanceRuleVo newVo) {
    this.overtimeRuleRepository.deleteByRuleCodes(Sets.newHashSet(oldVo.getRuleCode()), TenantUtils.getTenantCode());
    OvertimeRuleEntity ruleEntity = this.convertVoToEntity(newVo);
    if (Objects.isNull(ruleEntity)) {
      return;
    }
    this.overtimeRuleRepository.save(ruleEntity);
  }

  @Override
  @Transactional
  public void onDisable(List<AttendanceRuleVo> voList) {
    // do nothing
  }

  @Override
  @Transactional
  public void onEnable(List<AttendanceRuleVo> voList) {
    // do nothing
  }

  @Override
  @Transactional
  public void onDelete(List<AttendanceRuleVo> voList) {
    Set<String> ruleCodes = voList.stream().map(AttendanceRuleVo::getRuleCode).collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(ruleCodes)) {
      return;
    }
    this.overtimeRuleRepository.deleteByRuleCodes(ruleCodes, TenantUtils.getTenantCode());
  }

  @Override
  public List<AttendanceRuleNoWorkAbideVo> onRequestByRuleCodes(Set<String> ruleCodes) {
    if (CollectionUtils.isEmpty(ruleCodes)) {
      return Lists.newLinkedList();
    }

    List<OvertimeRuleEntity> ruleEntities = this.overtimeRuleRepository
        .findByRuleCodes(Sets.newHashSet(ruleCodes), TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(ruleEntities)) {
      return Lists.newLinkedList();
    }
    return ruleEntities.stream().map(overtimeRuleEntity -> {
      AttendanceRuleNoWorkAbideVo noWorkAbideVo = new AttendanceRuleNoWorkAbideVo();
      noWorkAbideVo.setKey(OvertimeConstant.OVERTIME_KEY);
      noWorkAbideVo.setRuleCode(overtimeRuleEntity.getRuleCode());
      noWorkAbideVo.setAbideFlag(overtimeRuleEntity.getAbideFlag());
      return noWorkAbideVo;
    }).collect(Collectors.toList());
  }

  @Override
  public List<AttendanceRuleNoWorkAbideDataVo> onRequestByRuleNoWorkAbideDataDto(RuleNoWorkAbideDataDto dto) {
    if (Objects.isNull(dto) || CollectionUtils.isEmpty(dto.getRuleCodes()) || StringUtils.isAnyBlank(dto.getRuleDate(), dto.getUserName())) {
      return Lists.newLinkedList();
    }
    OvertimeApplyConditionDto conditionDto = new OvertimeApplyConditionDto();
    conditionDto.setUserName(dto.getUserName());
    conditionDto.setQueryDate(dto.getRuleDate());
    conditionDto.setTenantCode(TenantUtils.getTenantCode());
    conditionDto.setRuleCodes(dto.getRuleCodes());
    List<OvertimeApplyEntity> applyEntities = this.overtimeApplyRepository.findByOvertimeApplyConditionDto(conditionDto);
    List<String> applyCodes = applyEntities.stream().map(OvertimeApplyEntity::getApplyCode).collect(Collectors.toList());
    ProcessBusinessMappingDto businessMappingDto = new ProcessBusinessMappingDto();
    businessMappingDto.setBusinessCode(OvertimeConstant.OVERTIME_APPLY_PROCESS_FORM_TYPE);
    businessMappingDto.setBusinessNos(applyCodes);
    List<ProcessBusinessMappingVo> businessMappingVos = this.processBusinessMappingService.findMultiByByConditions(businessMappingDto);
    Map<String, String> processStatusMap = CollectionUtils.isEmpty(businessMappingVos) ? Maps.newHashMap() : businessMappingVos.stream()
        .collect(Collectors.toMap(ProcessBusinessMappingVo::getBusinessNo, ProcessBusinessMappingVo::getProcessStatus, (a, b) -> b));
    //剩下审批中或者审批通过的数据(没有找到流程记录说明没有审批流程,认为是有效数据)
    applyEntities = applyEntities.stream().filter(entity -> {
      String processStatus = processStatusMap.getOrDefault(entity.getApplyCode(), ProcessStatusEnum.PASS.getDictCode());
      return ProcessStatusEnum.COMMIT.getDictCode().equals(processStatus) || ProcessStatusEnum.PASS.getDictCode().equals(processStatus);
    }).collect(Collectors.toList());
    Map<String, List<OvertimeApplyEntity>> applyEntityMap = CollectionUtils.isEmpty(applyEntities) ? Maps.newHashMap()
        : applyEntities.stream().collect(Collectors.groupingBy(OvertimeApplyEntity::getRuleCode));
    return dto.getRuleCodes().stream().map(ruleCode -> {
      String currentHaveAbideData = CollectionUtils.isEmpty(applyEntityMap.get(ruleCode))
          ? BooleanEnum.FALSE.getCapital() : BooleanEnum.TRUE.getCapital();
      AttendanceRuleNoWorkAbideDataVo abideDataVo = new AttendanceRuleNoWorkAbideDataVo();
      abideDataVo.setKey(OvertimeConstant.OVERTIME_KEY);
      abideDataVo.setRuleCode(ruleCode);
      abideDataVo.setCurrentHaveAbideData(currentHaveAbideData);
      return abideDataVo;
    }).collect(Collectors.toList());
  }

  private OvertimeRuleEntity convertVoToEntity(AttendanceRuleVo vo) {
    Validate.notNull(vo, "缺失参数对象");
    Validate.notBlank(vo.getRuleCode(), "缺失规则编码");
    List<AttendanceRuleNoWorkAbideVo> noWorkAbideInfoList = vo.getNoWorkAbideInfoList();
    if (CollectionUtils.isEmpty(noWorkAbideInfoList)) {
      return null;
    }
    List<AttendanceRuleNoWorkAbideVo> ruleOvertimeList = noWorkAbideInfoList.stream().filter(noWorkAbideVo -> OvertimeConstant.OVERTIME_KEY.equals(noWorkAbideVo.getKey())).collect(Collectors.toList());
    if (CollectionUtils.isEmpty(ruleOvertimeList)) {
      return null;
    }
    AttendanceRuleNoWorkAbideVo workAbideVo = ruleOvertimeList.get(0);
    Validate.notBlank(workAbideVo.getAbideFlag(), "缺失是否遵守此规则");
    OvertimeRuleEntity overtimeRuleEntity = new OvertimeRuleEntity();
    overtimeRuleEntity.setAbideFlag(workAbideVo.getAbideFlag());
    overtimeRuleEntity.setRuleCode(vo.getRuleCode());
    overtimeRuleEntity.setTenantCode(TenantUtils.getTenantCode());
    return overtimeRuleEntity;
  }
}
