package com.biz.crm.sfa.business.step.local.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mdm.business.position.sdk.service.PositionVoService;
import com.biz.crm.mdm.business.user.sdk.service.UserVoService;
import com.biz.crm.sfa.business.client.sdk.constant.ClientDictTypeConstant;
import com.biz.crm.sfa.business.client.sdk.enums.ClientTypeEnum;
import com.biz.crm.sfa.business.step.local.entity.StepForm;
import com.biz.crm.sfa.business.step.local.entity.StepRole;
import com.biz.crm.sfa.business.step.local.entity.StepRoleFormMapping;
import com.biz.crm.sfa.business.step.local.repository.StepFormRepository;
import com.biz.crm.sfa.business.step.local.repository.StepRoleRepository;
import com.biz.crm.sfa.business.step.local.service.StepRoleFormMappingService;
import com.biz.crm.sfa.business.step.local.service.StepRoleService;
import com.biz.crm.sfa.business.step.sdk.dto.StepRoleDataDto;
import com.biz.crm.sfa.business.step.sdk.dto.StepRolePageDto;
import com.biz.crm.sfa.business.step.sdk.enums.VisitTypeEnum;
import com.biz.crm.sfa.business.step.sdk.register.NecessaryFormRegister;
import com.biz.crm.sfa.business.step.sdk.strategy.ExecutePlanCallBackStrategy;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 步骤角色配置表(StepRole)表服务实现类
 *
 * @author songjingen
 * @since 2022-06-13 13:59:33
 */
@Service("stepRoleService")
public class StepRoleServiceImpl implements StepRoleService {

  @Autowired
  private StepRoleRepository stepRoleRepository;

  @Autowired
  private StepRoleFormMappingService stepRoleFormMappingService;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private DictDataVoService dictDataVoService;

  @Autowired(required = false)
  private List<NecessaryFormRegister> necessaryFormRegisters;

  @Autowired
  private StepFormRepository stepFormRepository;

  @Autowired(required = false)
  private List<ExecutePlanCallBackStrategy> executePlanCallBackStrategies;

  @Autowired(required = false)
  private UserVoService userVoService;

  @Autowired(required = false)
  private PositionVoService positionVoService;


  /**
   * 分页查询数据
   *
   * @param pageable 分页对象
   * @param dto      实体对象
   * @return
   */
  @Override
  public Page<StepRole> findByConditions(Pageable pageable, StepRolePageDto dto) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(dto)) {
      dto = new StepRolePageDto();
    }
    dto.setTenantCode(TenantUtils.getTenantCode());
    Page<StepRole> byConditions = this.stepRoleRepository.findByConditions(pageable, dto);
    List<StepRole> records = byConditions.getRecords();
    if (CollectionUtils.isEmpty(records)) {
      return new Page<>();
    }
    this.buildStepRole(records);
    return byConditions;
  }

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public StepRole findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    StepRole stepRole = this.stepRoleRepository.getById(id);
    if (stepRole == null) {
      return null;
    }
    List<StepRoleFormMapping> stepRoleFormMappings = this.stepRoleFormMappingService.findByStepRoleId(id);
    stepRole.setRoleFormMappings(stepRoleFormMappings);
    this.buildStepRole(Lists.newArrayList(stepRole));
    return stepRole;
  }

  /**
   * 新增数据
   * 1、创建验证数据重复性
   * 2、创建表单步骤关联数据验证
   * 3、组装步骤角色配置表数据，并保存
   * 4、保存关联步骤信息
   *
   * @param stepRoleDataDto 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public void create(StepRoleDataDto stepRoleDataDto) {
    //1、======
    this.createRoleDataValidate(stepRoleDataDto);
    //转换成entity集合
    List<StepRoleDataDto.StepFormData> stepFormData = stepRoleDataDto.getStepFormData();
    List<StepRoleFormMapping> stepRoleFormMappings = (List<StepRoleFormMapping>) this.nebulaToolkitService.copyCollectionByWhiteList(stepFormData, StepRoleDataDto.StepFormData.class, StepRoleFormMapping.class, HashSet.class, ArrayList.class);
    //2、======
    this.stepRoleFormMappingsValidate(stepRoleFormMappings);
    //3、======
    ArrayList<StepRole> stepRoles = new ArrayList<>();
    for (StepRoleDataDto.RoleData roleDatum : stepRoleDataDto.getRoleData()) {
      StepRole stepRole = this.nebulaToolkitService.copyObjectByWhiteList(stepRoleDataDto, StepRole.class, HashSet.class, ArrayList.class);
      stepRole.setRoleCode(roleDatum.getRoleCode());
      stepRole.setRoleName(roleDatum.getRoleName());
      this.createValidate(stepRole);
      stepRoles.add(stepRole);
    }
    this.stepRoleRepository.saveBatch(stepRoles);
    //4、======
    ArrayList<StepRoleFormMapping> formMappings = new ArrayList<>();
    stepRoles.stream().forEach(stepRole -> {
      String stepRoleId = stepRole.getId();
      for (StepRoleFormMapping stepRoleFormMapping : stepRoleFormMappings) {
        StepRoleFormMapping mapping = new StepRoleFormMapping();
        BeanUtils.copyProperties(stepRoleFormMapping, mapping);
        mapping.setStepRoleId(stepRoleId);
        formMappings.add(mapping);
      }
    });
    this.stepRoleFormMappingService.createBatch(formMappings);
  }

  /**
   * 修改新据
   * 1、创建表单步骤关联数据验证
   * 2、查询要修改的数据、并保存
   * 3、保存关联步骤信息
   *
   * @param stepRoleDataDto 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public StepRole update(StepRoleDataDto stepRoleDataDto) {
    Validate.notNull(stepRoleDataDto, "修改时，对象信息不能为空！");
    Validate.notBlank(stepRoleDataDto.getId(), "修改数据时，id不能为空！");
    List<StepRoleDataDto.StepFormData> stepFormData = stepRoleDataDto.getStepFormData();
    Validate.isTrue(!CollectionUtils.isEmpty(stepFormData), "修改数据时，关联步骤集合不能为空！");
    List<StepRoleFormMapping> roleFormMappings = (List<StepRoleFormMapping>) this.nebulaToolkitService.copyCollectionByWhiteList(stepFormData, StepRoleDataDto.StepFormData.class, StepRoleFormMapping.class, HashSet.class, ArrayList.class);
    //1、======
    this.stepRoleFormMappingsValidate(roleFormMappings);
    //2、======
    StepRole role = this.stepRoleRepository.getById(stepRoleDataDto.getId());
    Validate.notNull(role, "修改数据时，未查询到要修改的数据！");
    //步骤角色配置表只允许修改是否离线属性
    role.setOffLine(stepRoleDataDto.getOffLine());
    this.updateValidate(role);
    this.stepRoleRepository.saveOrUpdate(role);
    //3、======
    this.stepRoleFormMappingService.deleteByStepRoleIds(Lists.newArrayList(role.getId()));
    roleFormMappings.stream().forEach(stepRoleFormMapping -> stepRoleFormMapping.setStepRoleId(role.getId()));
    this.stepRoleFormMappingService.createBatch(roleFormMappings);
    return role;
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    List<StepRole> stepRoles = this.stepRoleRepository.findByIds(new HashSet<>(idList));
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，未查询到数据！");
    List<StepRole> enableStepRoles = stepRoles.stream().filter(stepRole -> EnableStatusEnum.ENABLE.getCode().equals(stepRole.getEnableStatus())).collect(Collectors.toList());
    Validate.isTrue(CollectionUtils.isEmpty(enableStepRoles), "存在未禁用的数据，请禁用后再次删除！！");
    this.stepRoleRepository.updateDelFlagByIds(idList, DelFlagStatusEnum.DELETE);
  }

  /**
   * 启用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 启用结果
   */
  @Transactional
  @Override
  public void enable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "启用数据时，主键集合不能为空！");
    this.stepRoleRepository.updateEnableStatusByIds(idList, EnableStatusEnum.ENABLE);
  }

  /**
   * 禁用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 禁用结果
   */
  @Transactional
  @Override
  public void disable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "禁用数据时，主键集合不能为空！");
    //验证当前角色有没有进行中的计划
    List<StepRole> stepRoles = this.stepRoleRepository.findByIds(new HashSet<>(idList));
    Validate.isTrue(!CollectionUtils.isEmpty(stepRoles), "禁用数据时，未查询到数据！");
    Set<String> visitTypes = stepRoles.stream().map(StepRole::getVisitType).collect(Collectors.toSet());
    Set<String> roles = this.findRoleCodesByVisitTypes(visitTypes);
    if (!CollectionUtils.isEmpty(roles)) {
      Set<String> disableRoles = stepRoles.stream().map(StepRole::getRoleCode).collect(Collectors.toSet());
      disableRoles.retainAll(roles);
      if (!CollectionUtils.isEmpty(disableRoles)) {
        throw new RuntimeException("角色编码" + disableRoles.toString() + "当前存在执行中的计划，不能禁用！");
      }
    }
    this.stepRoleRepository.updateEnableStatusByIds(idList, EnableStatusEnum.DISABLE);
  }

  /**
   * 创建验证
   *
   * @param stepRole
   */
  private void createValidate(StepRole stepRole) {
    Validate.notNull(stepRole, "新增时，对象信息不能为空！");
    stepRole.setId(null);
    stepRole.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    stepRole.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    stepRole.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(stepRole.getTenantCode(), "新增数据时，租户编号不能为空！");
    Validate.notBlank(stepRole.getClientType(), "新增数据时，客户类型不能为空！");
    Validate.notBlank(stepRole.getRoleCode(), "新增数据时，角色编码不能为空！");
    Validate.notBlank(stepRole.getVisitType(), "新增数据时，拜访类型不能为空！");

  }

  /**
   * 修改验证
   *
   * @param stepRole
   */
  private void updateValidate(StepRole stepRole) {
    Validate.notNull(stepRole, "修改时，对象信息不能为空！");
    Validate.notBlank(stepRole.getId(), "修改数据时，不能为空！");
    Validate.notBlank(stepRole.getTenantCode(), "修改数据时，租户编号不能为空！");
    Validate.notBlank(stepRole.getClientType(), "修改数据时，客户类型不能为空！");
    Validate.notBlank(stepRole.getRoleCode(), "修改数据时，角色编码不能为空！");
    Validate.notBlank(stepRole.getVisitType(), "修改数据时，拜访类型不能为空！");

  }

  /**
   * 创建验证角色数据重复性
   *
   * @param stepRoleDataDto
   */
  private void createRoleDataValidate(StepRoleDataDto stepRoleDataDto) {
    Validate.notNull(stepRoleDataDto, "新增数据时，入参数据不能为空！");
    List<StepRoleDataDto.RoleData> roleData = stepRoleDataDto.getRoleData();
    Validate.isTrue(!CollectionUtils.isEmpty(roleData), "新增数据时，角色集合不能为空!");
    Validate.isTrue(!CollectionUtils.isEmpty(stepRoleDataDto.getStepFormData()), "新增数据时，步骤表单集合不能为空!");
    List<String> roleCodes = roleData.stream()
        .map(v -> {
          if (org.apache.commons.lang3.StringUtils.isBlank(v.getRoleCode())) {
            throw new RuntimeException("提交了空值角色编码");
          }
          return v.getRoleCode();
        }).collect(Collectors.toList());
    String visitType = stepRoleDataDto.getVisitType();
    String clientType = stepRoleDataDto.getClientType();
    //拜访类型为陌拜类型时不需要验证客户类型，反之都需要验证
    List<StepRole> list = new ArrayList<>();
    if (VisitTypeEnum.STRANGE.getDictCode().equals(visitType)) {
      list = this.stepRoleRepository.findByVisitTypeAndRoleCodes(visitType, roleCodes);
    } else {
      list = this.stepRoleRepository.findByVisitTypeAndClientTypeAndRoleCodes(visitType, clientType, roleCodes);
    }
    Map<String, StepRole> roleMapConfig = list.stream()
        .collect(Collectors.toMap(StepRole::getRoleCode, v -> v, (t, t2) -> t2));
    for (StepRoleDataDto.RoleData role : roleData) {
      if (roleMapConfig.containsKey(role.getRoleCode())) {
        throw new RuntimeException("角色[" + role.getRoleName() + "]已存在[" + VisitTypeEnum.getByDictCode(visitType).getValue() + "]步骤配置,不能重复配置");
      }
    }
  }

  /**
   * 创建表单步骤关联数据验证
   * 1、验证是否有重复性步骤
   * 2、验证表单存在性并且验证首位顺序
   *
   * @param stepRoleFormMappings
   */
  private void stepRoleFormMappingsValidate(List<StepRoleFormMapping> stepRoleFormMappings) {
    //1、======
    Set<String> stepCodes = stepRoleFormMappings.stream().map(StepRoleFormMapping::getStepCode).collect(Collectors.toSet());
    Validate.isTrue(stepCodes.size() == stepRoleFormMappings.size(), "新增数据时,选择的拜访步骤中存在重复步骤!");
    //2、======
    //查询根据步骤编码查询表单步骤信息
    List<StepForm> stepForms = this.stepFormRepository.findByStepCodes(stepCodes);
    Validate.isTrue(!CollectionUtils.isEmpty(stepForms), "未查询到当前步骤关联的步骤表单信息！");
    Map<String, String> stepFormMap = stepForms.stream().collect(Collectors.toMap(StepForm::getStepCode, o -> o.getFormCode()));
    Map<String, StepRoleFormMapping> stepRoleFormMappingMap = new HashMap<>();
    for (StepRoleFormMapping stepRoleFormMapping : stepRoleFormMappings) {
      String formCode = stepFormMap.get(stepRoleFormMapping.getStepCode());
      if (StringUtils.isNotBlank(formCode)) {
        stepRoleFormMapping.setFormCode(formCode);
        stepRoleFormMappingMap.put(formCode, stepRoleFormMapping);
      }
    }
    if (!CollectionUtils.isEmpty(necessaryFormRegisters)) {
      //循环校验，必须存在的表单
      for (NecessaryFormRegister necessaryFormRegister : necessaryFormRegisters) {
        StepRoleFormMapping stepRoleFormMapping = stepRoleFormMappingMap.get(necessaryFormRegister.dynamicFormCode());
        Validate.notNull(stepRoleFormMapping, "配置步骤时，系统预设的表单【%s】必须存在！", necessaryFormRegister.dynamicFormName());
      }
      //验证首位和末位信息
      NecessaryFormRegister firstNecessaryFormRegister = necessaryFormRegisters.stream().filter(NecessaryFormRegister::first).sorted().collect(Collectors.toList()).get(0);
      NecessaryFormRegister lastNecessaryFormRegister = necessaryFormRegisters.stream().filter(NecessaryFormRegister::last).sorted().collect(Collectors.toList()).get(0);
      int firstCount = 0;
      int lastCount = 0;
      //最大的拜访顺序
      Integer lastSortNum = stepRoleFormMappings.stream().sorted(Comparator.comparing(StepRoleFormMapping::getSortNum).reversed()).collect(Collectors.toList()).get(0).getSortNum();
      for (StepRoleFormMapping stepRoleFormMapping : stepRoleFormMappings) {
        Integer sortNum = stepRoleFormMapping.getSortNum();
        Boolean doNot = stepRoleFormMapping.getDoNot();
        if (firstNecessaryFormRegister != null && StringUtils.equals(stepRoleFormMapping.getFormCode(), firstNecessaryFormRegister.dynamicFormCode())) {
          Validate.isTrue(sortNum == 1, "新增数据时，当前表单【%s】的步骤【%s】的拜访顺序必须是在首位！", firstNecessaryFormRegister.dynamicFormName(), stepRoleFormMapping.getStepName());
          Validate.isTrue(doNot, "新增数据时，当前表单【%s】的步骤【%s】的必做！", firstNecessaryFormRegister.dynamicFormName(), stepRoleFormMapping.getStepName());
          firstCount++;
        }
        if (lastNecessaryFormRegister != null && StringUtils.equals(stepRoleFormMapping.getFormCode(), lastNecessaryFormRegister.dynamicFormCode())) {
          Validate.isTrue(sortNum >= lastSortNum, "新增数据时，当前表单【%s】的步骤【%s】的拜访顺序必须是在末位！", lastNecessaryFormRegister.dynamicFormName(), stepRoleFormMapping.getStepName());
          Validate.isTrue(doNot, "新增数据时，当前表单【%s】的步骤【%s】的必做！", lastNecessaryFormRegister.dynamicFormName(), stepRoleFormMapping.getStepName());
          lastCount++;
        } else {
          Validate.isTrue(sortNum < lastSortNum, "新增数据时，系统配置的非末位表单【%s】步骤的拜访顺序必须小于提交的步骤表单总数！", lastNecessaryFormRegister.dynamicFormName());
        }

      }
      Validate.isTrue(firstCount == 1 && lastCount == 1, "新增数据时，配置的表单必须有且只能有一个首位表单【%s】步骤和一个末位表单【%s】步骤！", firstNecessaryFormRegister.dynamicFormName(), lastNecessaryFormRegister.dynamicFormName());
    }
  }

  /**
   * 构建数据
   *
   * @param stepRoles
   */
  private void buildStepRole(List<StepRole> stepRoles) {
    //查询数据字典
    Map<String, String> terminalTypeMap = new HashMap<>();
    Map<String, String> dealerTypeMap = new HashMap<>();
    ArrayList<String> list = Lists.newArrayList();
    list.add(ClientDictTypeConstant.TERMINAL_TYPE);
    list.add(ClientDictTypeConstant.CUSTOMER_TYPE);
    Map<String, List<DictDataVo>> dictTypeCodeList = this.dictDataVoService.findByDictTypeCodeList(list);
    if (dictTypeCodeList != null) {
      List<DictDataVo> terminalDicts = dictTypeCodeList.get(ClientDictTypeConstant.TERMINAL_TYPE);
      if (!CollectionUtils.isEmpty(terminalDicts)) {
        terminalTypeMap = terminalDicts.stream().collect(Collectors.toMap(DictDataVo::getDictCode, DictDataVo::getDictValue));
      }
      List<DictDataVo> dealerDicts = dictTypeCodeList.get(ClientDictTypeConstant.CUSTOMER_TYPE);
      if (!CollectionUtils.isEmpty(dealerDicts)) {
        dealerTypeMap = dealerDicts.stream().collect(Collectors.toMap(DictDataVo::getDictCode, DictDataVo::getDictValue));
      }
    }
    for (StepRole stepRole : stepRoles) {
      String clientType = stepRole.getClientType();
      String clientSubType = stepRole.getClientSubType();
      if (ClientTypeEnum.TERMINAL.getDictCode().equals(clientType)) {
        stepRole.setClientSubType(terminalTypeMap.get(clientSubType));
      }
      if (ClientTypeEnum.DEALER.getDictCode().equals(clientType)) {
        stepRole.setClientSubType(dealerTypeMap.get(clientSubType));
      }
    }
  }

  /**
   * 根据拜访类型获取当前日期拜访中的角色
   *
   * @return
   */
  private Set<String> findRoleCodesByVisitTypes(Set<String> visitTypes) {
    Set<String> set = new HashSet<>();
    if (!CollectionUtils.isEmpty(executePlanCallBackStrategies)) {
      for (ExecutePlanCallBackStrategy executePlanCallBackStrategy : executePlanCallBackStrategies) {
        if (visitTypes.contains(executePlanCallBackStrategy.visitType())) {
          Set<String> strings = executePlanCallBackStrategy.onFindUserNamesByNowAndExecuteStatusIn();
          set.addAll(strings);
        }
      }
    }
    //通过用户账号查询角色信息
    if (CollectionUtils.isEmpty(set)) {
      return null;
    }
    Set<String> positionCodes = this.userVoService.findPositionCodesByUserNames(Lists.newArrayList(set));
    if (CollectionUtils.isEmpty(positionCodes)) {
      return null;
    }
    return this.positionVoService.findRolesByPositionCodes(Lists.newArrayList(positionCodes));
  }
}

