package com.biz.crm.sfa.business.step.local.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.sfa.business.client.sdk.constant.ClientDictTypeConstant;
import com.biz.crm.sfa.business.client.sdk.enums.ClientTypeEnum;
import com.biz.crm.sfa.business.step.local.entity.StepForm;
import com.biz.crm.sfa.business.step.local.entity.StepFormLogo;
import com.biz.crm.sfa.business.step.local.repository.StepFormRepository;
import com.biz.crm.sfa.business.step.local.service.StepFormLogoService;
import com.biz.crm.sfa.business.step.local.service.StepFormService;
import com.biz.crm.sfa.business.step.sdk.constant.StepConstant;
import com.biz.crm.sfa.business.step.sdk.dto.StepFormPageDto;
import com.biz.crm.sfa.business.step.sdk.enums.DateTypeEnum;
import com.biz.crm.sfa.business.step.sdk.enums.VisitTypeEnum;
import com.biz.crm.sfa.business.step.sdk.strategy.StepFormValidationStrategy;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 步骤表单配置表(StepForm)表服务实现类
 *
 * @author songjingen
 * @since 2022-06-09 14:02:38
 */
@Service("stepFormService")
public class StepFormServiceImpl implements StepFormService {

  @Autowired
  private StepFormRepository stepFormRepository;

  @Autowired
  private StepFormLogoService stepFormLogoService;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired(required = false)
  private DictDataVoService dictDataVoService;

  @Autowired(required = false)
  private List<StepFormValidationStrategy> stepFormValidationStrategies;

  /**
   * 分页查询数据
   *
   * @param pageable 分页对象
   * @param dto      实体对象
   * @return
   */
  @Override
  public Page<StepForm> findByConditions(Pageable pageable, StepFormPageDto dto) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(dto)) {
      dto = new StepFormPageDto();
    }
    dto.setTenantCode(TenantUtils.getTenantCode());
    Page<StepForm> byConditions = this.stepFormRepository.findByConditions(pageable, dto);
    List<StepForm> records = byConditions.getRecords();
    if (CollectionUtils.isEmpty(records)) {
      return new Page<>();
    }
    this.buildStepForm(records);
    return byConditions;
  }

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public StepForm findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    StepForm stepForm = this.stepFormRepository.getById(id);
    if (stepForm == null) {
      return null;
    }
    List<StepFormLogo> stepFormLogos = this.stepFormLogoService.findByStepCode(stepForm.getStepCode());
    stepForm.setLogos(stepFormLogos);
    return stepForm;
  }

  /**
   * 新增数据
   *
   * @param stepForm 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public StepForm create(StepForm stepForm) {
    this.createValidate(stepForm);
    this.stepFormRepository.saveOrUpdate(stepForm);
    //保存logo信息
    List<StepFormLogo> logos = stepForm.getLogos();
    Validate.isTrue(!CollectionUtils.isEmpty(logos), "新增logo时，入参不能为空！");
    logos.stream().forEach(stepFormLogo -> stepFormLogo.setStepCode(stepForm.getStepCode()));
    this.stepFormLogoService.createBatch(logos);
    return stepForm;
  }

  /**
   * 修改新据
   *
   * @param stepForm 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public StepForm update(StepForm stepForm) {
    StepForm form = this.stepFormRepository.getById(stepForm.getId());
    Validate.notNull(form, "编辑时，未查询到要编辑的信息！");
    if (!form.getStepName().equals(stepForm.getStepName())) {
      //验证步骤名称重复性
      List<StepForm> stepForms = this.stepFormRepository.findByStepName(stepForm.getStepName());
      Validate.isTrue(CollectionUtils.isEmpty(stepForms), "编辑时，步骤名称重复！");
    }
    BeanUtils.copyProperties(stepForm, form, "id", "stepCode", "tenantCode", "delFlag", "enableStatus", "createAccount", "createTime", "createName");
    this.updateValidate(form);
    this.stepFormRepository.saveOrUpdate(form);
    //保存logo信息
    List<StepFormLogo> logos = stepForm.getLogos();
    Validate.isTrue(!CollectionUtils.isEmpty(logos), "新增logo时，入参不能为空！");
    logos.stream().forEach(stepFormLogo -> stepFormLogo.setStepCode(form.getStepCode()));
    this.stepFormLogoService.deleteByStepCode(form.getStepCode());
    this.stepFormLogoService.createBatch(logos);
    return form;
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    this.stepFormValidate(idList);
    this.stepFormRepository.updateDelFlagByIds(idList);
  }

  /**
   * 启用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 启用结果
   */
  @Transactional
  @Override
  public void enable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "启用数据时，主键集合不能为空！");
    this.stepFormRepository.updateEnableStatusByIds(idList, EnableStatusEnum.ENABLE);
  }

  /**
   * 禁用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 禁用结果
   */
  @Transactional
  @Override
  public void disable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "禁用数据时，主键集合不能为空！");
    this.stepFormValidate(idList);
    this.stepFormRepository.updateEnableStatusByIds(idList, EnableStatusEnum.DISABLE);
  }

  /**
   * 构建数据
   *
   * @param stepForms
   */
  private void buildStepForm(List<StepForm> stepForms) {
    //查询数据字典
    Map<String, String> terminalTypeMap = new HashMap<>();
    Map<String, String> dealerTypeMap = new HashMap<>();
    ArrayList<String> list = Lists.newArrayList();
    list.add(ClientDictTypeConstant.TERMINAL_TYPE);
    list.add(ClientDictTypeConstant.CUSTOMER_TYPE);
    Map<String, List<DictDataVo>> dictTypeCodeList = this.dictDataVoService.findByDictTypeCodeList(list);
    if (dictTypeCodeList != null) {
      List<DictDataVo> terminalDicts = dictTypeCodeList.get(ClientDictTypeConstant.TERMINAL_TYPE);
      if (!CollectionUtils.isEmpty(terminalDicts)) {
        terminalTypeMap = terminalDicts.stream().collect(Collectors.toMap(DictDataVo::getDictCode, DictDataVo::getDictValue));
      }
      List<DictDataVo> dealerDicts = dictTypeCodeList.get(ClientDictTypeConstant.CUSTOMER_TYPE);
      if (!CollectionUtils.isEmpty(dealerDicts)) {
        dealerTypeMap = dealerDicts.stream().collect(Collectors.toMap(DictDataVo::getDictCode, DictDataVo::getDictValue));
      }
    }
    for (StepForm stepForm : stepForms) {
      String clientType = stepForm.getClientType();
      String clientSubType = stepForm.getClientSubType();
      if (ClientTypeEnum.TERMINAL.getDictCode().equals(clientType)) {
        stepForm.setClientSubType(terminalTypeMap.get(clientSubType));
      }
      if (ClientTypeEnum.DEALER.getDictCode().equals(clientType)) {
        stepForm.setClientSubType(dealerTypeMap.get(clientSubType));
      }
    }
  }

  /**
   * 创建验证
   *
   * @param stepForm
   */
  private void createValidate(StepForm stepForm) {
    Validate.notNull(stepForm, "新增时，对象信息不能为空！");
    stepForm.setId(null);
    stepForm.setTenantCode(TenantUtils.getTenantCode());
    stepForm.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    stepForm.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    stepForm.setStepCode(this.generateCodeService.generateCode(StepConstant.step_code, 1).get(0));
    Validate.notBlank(stepForm.getTenantCode(), "新增数据时，租户编号不能为空！");
    Validate.notBlank(stepForm.getFormCode(), "新增数据时，表单编码不能为空！");
    Validate.notBlank(stepForm.getFormName(), "新增数据时，表单名称不能为空！");
    Validate.notBlank(stepForm.getStepCode(), "新增数据时，步骤编码不能为空！");
    Validate.notBlank(stepForm.getStepName(), "新增数据时，步骤名称不能为空！");
    Validate.notBlank(stepForm.getVisitType(), "新增数据时，拜访类型不能为空！");
    if (!VisitTypeEnum.STRANGE.getDictCode().equals(stepForm.getVisitType())) {
      Validate.notBlank(stepForm.getClientType(), "新增数据时，客户类型不能为空！");
    }
    Validate.notBlank(stepForm.getDateType(), "新增数据时，有效期类型不能为空！");
    if (DateTypeEnum.LONG.getDictCode().equals(stepForm.getDateType())) {
      stepForm.setStartTime(null);
      stepForm.setEndTime(null);
    } else {
      Validate.notNull(stepForm.getStartTime(), "新增数据时，有效期开始日期不能为空！");
      Validate.notNull(stepForm.getEndTime(), "新增数据时，有效期结束日期不能为空！");
      Validate.isTrue(stepForm.getStartTime().compareTo(stepForm.getEndTime()) <= 0, "新增数据时，有效期开始日期不能在结束日期之后！");

    }
    Validate.notNull(stepForm.getLocateType(), "新增数据时，是否获取定位信息: 1-是，0-否不能为空！");
    if (stepForm.getLocateType()) {
      Validate.notNull(stepForm.getDistance(), "新增数据时，距离限制不能为空！");
    }
    //验证步骤名称重复性
    List<StepForm> stepForms = this.stepFormRepository.findByStepName(stepForm.getStepName());
    Validate.isTrue(CollectionUtils.isEmpty(stepForms), "新增数据时，步骤名称重复！");

  }

  /**
   * 修改验证
   *
   * @param stepForm
   */
  private void updateValidate(StepForm stepForm) {
    Validate.notNull(stepForm, "修改时，对象信息不能为空！");
    Validate.notBlank(stepForm.getId(), "修改数据时，不能为空！");
    Validate.notBlank(stepForm.getTenantCode(), "修改数据时，租户编号不能为空！");
    Validate.notBlank(stepForm.getFormCode(), "修改数据时，表单编码不能为空！");
    Validate.notBlank(stepForm.getFormName(), "修改数据时，表单名称不能为空！");
    Validate.notBlank(stepForm.getStepCode(), "修改数据时，步骤编码不能为空！");
    Validate.notBlank(stepForm.getStepName(), "修改数据时，步骤名称不能为空！");
    Validate.notBlank(stepForm.getVisitType(), "新增数据时，拜访类型不能为空！");
    if (!VisitTypeEnum.STRANGE.getDictCode().equals(stepForm.getVisitType())) {
      Validate.notBlank(stepForm.getClientType(), "新增数据时，客户类型不能为空！");
    }
    Validate.notBlank(stepForm.getDateType(), "修改数据时，有效期类型不能为空！");
    if (DateTypeEnum.LONG.getDictCode().equals(stepForm.getDateType())) {
      stepForm.setStartTime(null);
      stepForm.setEndTime(null);
    } else {
      Validate.notNull(stepForm.getStartTime(), "新增数据时，有效期开始日期不能为空！");
      Validate.notNull(stepForm.getEndTime(), "新增数据时，有效期结束日期不能为空！");
      Validate.isTrue(stepForm.getStartTime().compareTo(stepForm.getEndTime()) <= 0, "新增数据时，有效期开始日期不能在结束日期之后！");

    }
    Validate.notNull(stepForm.getLocateType(), "修改数据时，是否获取定位信息: 1-是，0-否不能为空！");
    if (stepForm.getLocateType()) {
      Validate.notNull(stepForm.getDistance(), "修改数据时，距离限制不能为空！");
    }
  }

  /**
   * 根据id验证表单步骤信息
   *
   * @param idList
   */
  private void stepFormValidate(List<String> idList) {
    List<StepForm> stepForms = this.stepFormRepository.findByIds(idList);
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "禁用数据时，未查询到要操作的信息！");
    //验证表单
    Set<String> stepCodes = stepForms.stream().map(StepForm::getStepCode).collect(Collectors.toSet());
    if (!CollectionUtils.isEmpty(stepFormValidationStrategies)) {
      for (StepFormValidationStrategy stepFormValidationStrategy : stepFormValidationStrategies) {
        stepFormValidationStrategy.stepCodeUsedValidate(stepCodes);
      }
    }
  }
}

